/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.core.resource;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.net.URLConnection;

import shohaku.core.helpers.HEval;
import shohaku.core.lang.NoSuchResourceException;
import shohaku.core.lang.feature.FeatureFactory;

/**
 * リソースのクラスパスとクラスローダから入力ストリームを返す入力リソースを提供します。
 */
public class ClassPathInputResource implements InputResource {

    /** クラスパス。 */
    private final String path;

    /** リソースを読み取る為のクラスローダ。 */
    private final ClassLoader classLoader;

    /** リソースを読み取る為のクラス。 */
    private final Class clazz;

    /**
     * クラスパスを格納して初期化します。
     * 
     * @param path
     *            クラスパス
     * @throws IllegalArgumentException
     *             クラスパスが空の場合
     */
    public ClassPathInputResource(String path) {
        this(path, null, null);
    }

    /**
     * クラスパスとクラスローダを格納して初期化します。
     * 
     * @param path
     *            クラスパス
     * @param classLoader
     *            リソースを読み取る為のクラスローダ、null の場合は無視されます。
     * @throws IllegalArgumentException
     *             クラスパスが空の場合
     */
    public ClassPathInputResource(String path, ClassLoader classLoader) {
        this(path, classLoader, null);
    }

    /**
     * クラスパスとリードオブジェクトを格納して初期化します。
     * 
     * @param path
     *            クラスパス
     * @param clazz
     *            リソースを読み取る為のクラス、null の場合は無視されます
     * @throws IllegalArgumentException
     *             クラスパスが空の場合
     */
    public ClassPathInputResource(String path, Class clazz) {
        this(path, null, clazz);
    }

    /* クラスパスとリードオブジェクトを格納して初期化します。 */
    private ClassPathInputResource(String path, ClassLoader classLoader, Class clazz) {
        if (HEval.isBlank(path)) {
            throw new IllegalArgumentException("path is blank");
        }
        this.path = ((path.startsWith("/")) ? path.substring(1) : path);
        this.classLoader = classLoader;
        this.clazz = clazz;
    }

    /**
     * クラスパスが示すリソースの入力ストリームを返却します。
     * 
     * @return 入力ストリーム
     * @throws IOException
     *             リソースが発見出来無い場合
     * @see shohaku.core.resource.InputResource#getInputStream()
     */
    public InputStream getInputStream() throws IOException {
        try {
            return getResourceAsStream();
        } catch (NoSuchResourceException e) {
            IOException fe = new FileNotFoundException("resource not exist");
            fe.initCause(e);
            throw fe;
        }
    }

    public boolean exists() {
        try {
            InputStream is = getResourceAsStream();
            try {
                is.close();
            } catch (IOException e) {
                // no op
            }
            return true;
        } catch (NoSuchResourceException e) {
            return false;
        }
    }

    public long getLastModified() throws IOException {
        try {
            URLConnection con = getURL().openConnection();
            con.setUseCaches(false);// キャッシュ不要
            return con.getLastModified();
        } catch (NoSuchResourceException e) {
            IOException fe = new FileNotFoundException("resource not exist.");
            fe.initCause(e);
            throw fe;
        }
    }

    private InputStream getResourceAsStream() throws NoSuchResourceException {
        if (this.classLoader != null) {
            return FeatureFactory.getLoader().getResourceAsStream(this.path, this.classLoader);
        }
        if (this.clazz != null) {
            return FeatureFactory.getLoader().getResourceAsStream(this.path, this.clazz);
        }
        return FeatureFactory.getLoader().getResourceAsStream(this.path);
    }

    private URL getURL() throws NoSuchResourceException {
        if (this.classLoader != null) {
            return FeatureFactory.getLoader().getResource(this.path, this.classLoader);
        }
        if (this.clazz != null) {
            return FeatureFactory.getLoader().getResource(this.path, this.clazz);
        }
        return FeatureFactory.getLoader().getResource(this.path);
    }
}