/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.core.lang;

import java.util.Collections;
import java.util.Map;

/**
 * 松柏コアライブラリ固有のシステム情報を提供します。
 */
public class ShohakuCoreSystem {

    /**
     * 松柏コアAPIのバージョンを返却します。
     * 
     * @return 松柏コアAPIのバージョン
     */
    public static String getLibraryVersion() {
        return SystemUtils.getLibraryVersion(ShohakuCoreSystem.class);
    }

    /**
     * 松柏コアライブラリ固有のプロパティセットを返却します。<br>
     * 返されるマップは普遍マップです。
     * 
     * @return 松柏コアシステムのプロパティセット
     */
    public static Map getLibraryProperties() {
        return PropertiesCache.libraryProperties;
    }

    /**
     * キーの示す松柏コアライブラリ固有のプロパティを返却します。<br>
     * キーに対応するプロパティが無い場合は null が返されます。
     * 
     * @param key
     *            キー
     * @return 松柏コアライブラリ固有のプロパティ
     */
    public static Object getLibraryProperty(String key) {
        return getLibraryProperties().get(key);
    }

    /**
     * '@' で始まるクラスと関連づけられたキーの示す松柏コアライブラリ固有のプロパティを返却します。<br>
     * キーに対応するプロパティが無い場合は null が返されます。
     * 
     * @param clazz
     *            クラス
     * @param key
     *            キー
     * @return 松柏コアライブラリ固有のプロパティ
     */
    public static Object getLibraryClassProperty(Class clazz, String key) {
        return getLibraryProperties().get("@" + clazz.getName() + "." + key);
    }

    /**
     * '@' で始まるパッケージと関連づけられたキーの示す松柏コアライブラリ固有のプロパティを返却します。<br>
     * キーに対応するプロパティが無い場合は null が返されます。
     * 
     * @param pkg
     *            パッケージ
     * @param key
     *            キー
     * @return 松柏コアライブラリ固有のプロパティ
     */
    public static Object getLibraryPackageProperty(Package pkg, String key) {
        return getLibraryProperties().get("@" + pkg.getName() + "." + key);
    }

    /* 松柏コアシステムのプロパティを保持するクラス */
    private static final class PropertiesCache {
        static final Map libraryProperties;
        static {
            final Map properties = SystemUtils.getLibraryProperties(ShohakuCoreSystem.class);
            libraryProperties = Collections.unmodifiableMap(properties);
        }
    }

}
