/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.core.lang;

import shohaku.core.helpers.HHash;

/**
 * 数値（double）の範囲を表す機能を提供します、不変オブジェクトとして定義されます。
 */
public final class RangeDouble implements RangeNumber {

    /* serialVersionUID */
    private static final long serialVersionUID = -1742053906414808338L;

    /* 最大値。 */
    private final double min;

    /* 最小値。 */
    private final double max;

    /**
     * 最小値と最大値を 0.0 で初期化します。
     */
    public RangeDouble() {
        this.min = 0.0;
        this.max = 0.0;
    }

    /**
     * 最小値と最大値を指定して初期化します。
     * 
     * @param min
     *            最小値
     * @param max
     *            最大値
     */
    public RangeDouble(double min, double max) {
        this.min = min;
        this.max = max;
    }

    /**
     * 最大値を Double で返却します。
     * 
     * @return 最大値
     */
    public Number getMaxValue() {
        return Boxing.box(max);
    }

    /**
     * 最小値を Double で返却します。
     * 
     * @return 最小値
     */
    public Number getMinValue() {
        return Boxing.box(min);
    }

    /**
     * 最大値を返却します。
     * 
     * @return 最大値
     */
    public double getMax() {
        return max;
    }

    /**
     * 最小値を返却します。
     * 
     * @return 最小値
     */
    public double getMin() {
        return min;
    }

    /**
     * 範囲に含まれるか検証します。
     * 
     * @param n
     *            検証する値
     * @return 含まれる場合は true
     */
    public boolean contain(double n) {
        return (min <= n && n <= max);
    }

    /**
     * Double.hashCode() と同一の算出方法で (min + max) を返却します。
     * 
     * @return ハッシュコード
     * @see java.lang.Object#hashCode()
     */
    public int hashCode() {
        return (HHash.hash(min) + HHash.hash(max));
    }

    /**
     * 引数が同値であるか検証します。 <br>
     * RangeDouble 型で最大値と最小値が同一の場合のみ true を返却します。
     * 
     * @param o
     *            比較先のオブジェクト
     * @return RangeDouble 型で最大値と最小値が同一の場合のみ true
     * @see java.lang.Object#equals(java.lang.Object)
     */
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof RangeDouble)) {
            return false;
        }
        RangeDouble r = (RangeDouble) o;
        return (r.min == min && r.max == max);
    }

    /**
     * このオブジェクトの文字列表現を返却します。 <br>
     * min:max
     * 
     * @return このオブジェクトの文字列表現
     * @see java.lang.Object#toString()
     */
    public String toString() {
        return String.valueOf(min) + ":" + String.valueOf(max);
    }

}
