/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.core.helpers;

import java.nio.charset.Charset;
import java.nio.charset.CharsetEncoder;
import java.nio.charset.IllegalCharsetNameException;
import java.nio.charset.UnsupportedCharsetException;
import java.text.DateFormatSymbols;
import java.text.DecimalFormatSymbols;
import java.util.Collection;
import java.util.List;
import java.util.Locale;

import shohaku.core.lang.RangeInt;

/**
 * 値を妥当性を評価するヘルパーメソッド群を提供します。
 */
public class HValid {

    /** デフォルトの日時とミリ秒のフォーマット。 */
    public static final String DATETIME_FORMAT = Constants.DATETIME_FORMAT;

    /** デフォルトで有効な日付フォーマットの配列。 */
    public static final List DATETIME_FORMAT_LIST = Constants.DATETIME_FORMAT_LIST;

    /*
     * Chars
     */

    /**
     * 指定の文字のみで構成される文字シーケンスか評価します。
     * 
     * @param cs
     *            評価する文字シーケンス
     * @param chars
     *            構成文字
     * @return 指定の文字のみで構成される文字シーケンスの場合のみ true
     */
    public static boolean isCharsOnly(CharSequence cs, char[] chars) {
        if (cs == null) {
            return false;
        }
        return (0 > HSeek.orOtherIndexOf(cs, chars, 1));
    }

    /**
     * 指定の文字のみで構成される文字シーケンスか評価します。
     * 
     * @param cs
     *            評価する文字シーケンス
     * @param chars
     *            構成文字
     * @param fromIndex
     *            検索の開始位置を示すインデックス
     * @param toIndex
     *            検索の終了位置を示すインデックス
     * @return 指定の範囲内が指定の文字のみで構成される文字シーケンスの場合のみ true
     */
    public static boolean isCharsOnly(CharSequence cs, char[] chars, int fromIndex, int toIndex) {
        if (cs == null) {
            return false;
        }
        return (0 > HSeek.orOtherIndexOf(cs, toIndex, chars, fromIndex));
    }

    /**
     * 指定の範囲内の文字のみで構成される文字シーケンスか評価します。
     * 
     * @param cs
     *            評価する文字シーケンス
     * @param rangeChar
     *            有効な文字の範囲
     * @return 文字シーケンスの全要素が指定の範囲内の場合のみ true
     */
    public static boolean isCharsRange(CharSequence cs, RangeInt rangeChar) {
        if (cs == null) {
            return false;
        }
        int len = cs.length();
        for (int i = 0; i < len; i++) {
            if (!isRange(cs.charAt(i), rangeChar.getMin(), rangeChar.getMax())) {
                return false;
            }
        }
        return true;
    }

    /**
     * 指定の範囲内の文字のみで構成される文字シーケンスか評価します。
     * 
     * @param cs
     *            評価する文字シーケンス
     * @param rangeChars
     *            有効な文字の範囲
     * @return 文字シーケンスの全要素が指定の範囲内の場合のみ true
     */
    public static boolean isCharsRange(CharSequence cs, RangeInt[] rangeChars) {
        if (cs == null) {
            return false;
        }
        int len = cs.length();
        boolean hit = false;
        for (int i = 0; i < len; i++) {
            char c = cs.charAt(i);
            for (int j = 0; j < rangeChars.length; j++) {
                RangeInt r = rangeChars[j];
                if (isRange(c, r.getMin(), r.getMax())) {
                    hit = true;
                    break;
                }
            }
            if (!hit) {
                return false;
            }
            hit = false;
        }
        return true;
    }

    /*
     * date
     */

    /**
     * 日付文字列として認識できるか評価します。
     * 
     * @param date
     *            評価する日付文字列
     * @return 日付文字列として認識できる場合のみ true
     */
    public static boolean isDateTime(CharSequence date) {
        return isDateTime(date, DATETIME_FORMAT_LIST);
    }

    /**
     * 日付文字列として認識できるか評価します。
     * 
     * @param date
     *            評価する日付文字列
     * @param pattern
     *            書式パターン
     * @return 日付文字列として認識できる場合のみ true
     */
    public static boolean isDateTime(CharSequence date, String pattern) {
        return isDateTime(date, Locale.getDefault(), pattern);
    }

    /**
     * 日付文字列として認識できるか評価します。
     * 
     * @param date
     *            評価する日付文字列
     * @param locale
     *            ロケール
     * @param pattern
     *            書式パターン
     * @return 日付文字列として認識できる場合のみ true
     */
    public static boolean isDateTime(CharSequence date, Locale locale, String pattern) {
        return isDateTime(date, new DateFormatSymbols(locale), pattern);
    }

    /**
     * 日付文字列として認識できるか評価します。
     * 
     * @param date
     *            評価する日付文字列
     * @param symbols
     *            日付の記号セット
     * @param pattern
     *            書式パターン
     * @return 日付文字列として認識できる場合のみ true
     */
    public static boolean isDateTime(CharSequence date, DateFormatSymbols symbols, String pattern) {
        return isDateTime(date, symbols, pattern, false);
    }

    /**
     * 日付文字列として認識できるか評価します。
     * 
     * @param date
     *            評価する日付文字列
     * @param symbols
     *            日付の記号セット
     * @param pattern
     *            書式パターン
     * @param lenient
     *            日付/時刻解析を曖昧に行うか設定する、true=曖昧な解析
     * @return 日付文字列として認識できる場合のみ true
     */
    public static boolean isDateTime(CharSequence date, DateFormatSymbols symbols, String pattern, boolean lenient) {
        return (date == null) ? false : (null != HCnv.toDateTime(date, symbols, pattern, lenient));
    }

    /**
     * 一つ以上の書式パターンで日付文字列として認識できるか評価します。
     * 
     * @param date
     *            評価する日付文字列
     * @param patterns
     *            書式パターン
     * @return 一つ以上の書式パターンで日付文字列として認識できる場合のみ true
     */
    public static boolean isDateTime(CharSequence date, Collection patterns) {
        return isDateTime(date, Locale.getDefault(), patterns);
    }

    /**
     * 一つ以上の書式パターンで日付文字列として認識できるか評価します。
     * 
     * @param date
     *            評価する日付文字列
     * @param locale
     *            ロケール
     * @param patterns
     *            書式パターン
     * @return 一つ以上の書式パターンで日付文字列として認識できる場合のみ true
     */
    public static boolean isDateTime(CharSequence date, Locale locale, Collection patterns) {
        return isDateTime(date, new DateFormatSymbols(locale), patterns);
    }

    /**
     * 一つ以上の書式パターンで日付文字列として認識できるか評価します。
     * 
     * @param date
     *            評価する日付文字列
     * @param symbols
     *            日付の記号セット
     * @param patterns
     *            書式パターン
     * @return 一つ以上の書式パターンで日付文字列として認識できる場合のみ true
     */
    public static boolean isDateTime(CharSequence date, DateFormatSymbols symbols, Collection patterns) {
        return isDateTime(date, symbols, patterns, false);
    }

    /**
     * 一つ以上の書式パターンで日付文字列として認識できるか評価します。
     * 
     * @param date
     *            評価する日付文字列
     * @param symbols
     *            日付の記号セット
     * @param patterns
     *            書式パターン
     * @param lenient
     *            日付/時刻解析を曖昧に行うか設定する、true=曖昧な解析
     * @return 一つ以上の書式パターンで日付文字列として認識できる場合のみ true
     */
    public static boolean isDateTime(CharSequence date, DateFormatSymbols symbols, Collection patterns, boolean lenient) {
        return (date == null) ? false : (null != HCnv.toDateTime(date, symbols, patterns, lenient));
    }

    /*
     * Decimal
     */

    /**
     * 数値文字列として認識できるか評価します。
     * 
     * @param num
     *            評価する数値文字列
     * @param pattern
     *            書式パターン
     * @return 数値文字列として認識できる場合のみ true
     */
    public static boolean isDecimal(CharSequence num, String pattern) {
        return isDecimal(num, Locale.getDefault(), pattern);
    }

    /**
     * 数値文字列として認識できるか評価します。
     * 
     * @param num
     *            評価する数値文字列
     * @param locale
     *            ロケール
     * @param pattern
     *            書式パターン
     * @return 数値文字列として認識できる場合のみ true
     */
    public static boolean isDecimal(CharSequence num, Locale locale, String pattern) {
        return isDecimal(num, new DecimalFormatSymbols(locale), pattern);
    }

    /**
     * 数値文字列として認識できるか評価します。
     * 
     * @param num
     *            評価する数値文字列
     * @param symbols
     *            数値変換の記号セット
     * @param pattern
     *            書式パターン
     * @return 数値文字列として認識できる場合のみ true
     */
    public static boolean isDecimal(CharSequence num, DecimalFormatSymbols symbols, String pattern) {
        return (num == null) ? false : (null != HCnv.toDecimal(num, symbols, pattern));
    }

    /**
     * 一つ以上の書式パターンで数値文字列として認識できるか評価します。
     * 
     * @param num
     *            評価する数値文字列
     * @param patterns
     *            書式パターン
     * @return 一つ以上の書式パターンで数値文字列として認識できる場合のみ true
     */
    public static boolean isDecimal(CharSequence num, Collection patterns) {
        return isDecimal(num, Locale.getDefault(), patterns);
    }

    /**
     * 一つ以上の書式パターンで数値文字列として認識できるか評価します。
     * 
     * @param num
     *            評価する数値文字列
     * @param locale
     *            ロケール
     * @param patterns
     *            書式パターン
     * @return 一つ以上の書式パターンで数値文字列として認識できる場合のみ true
     */
    public static boolean isDecimal(CharSequence num, Locale locale, Collection patterns) {
        return isDecimal(num, new DecimalFormatSymbols(locale), patterns);
    }

    /**
     * 一つ以上の書式パターンで数値文字列として認識できるか評価します。
     * 
     * @param num
     *            評価する数値文字列
     * @param symbols
     *            数値変換の記号セット
     * @param patterns
     *            書式パターン
     * @return 一つ以上の書式パターンで数値文字列として認識できる場合のみ true
     */
    public static boolean isDecimal(CharSequence num, DecimalFormatSymbols symbols, Collection patterns) {
        return (num == null) ? false : (null != HCnv.toDecimal(num, symbols, patterns));
    }

    /*
     * Encode
     */

    /**
     * 指定の文字セットで指定された文字をエンコードできるかを評価します。
     * 
     * @param c
     *            評価する文字
     * @param charsetName
     *            要求された文字セットの名前 (標準名または別名)
     * @return 指定の文字セットで指定された文字をエンコードできる場合にかぎり true
     * @throws UnsupportedCharsetException
     *             指定された文字セット名が不当である場合
     * @throws IllegalCharsetNameException
     *             指定された文字セットを現在の Java 仮想マシンでは利用できない場合
     * @throws UnsupportedOperationException
     *             この文字セットがエンコードをサポートしない場合
     */
    public static boolean isEncode(char c, String charsetName) throws UnsupportedCharsetException, IllegalCharsetNameException, UnsupportedOperationException {
        return isEncode(c, Charset.forName(charsetName));
    }

    /**
     * 指定の文字セットで指定された文字をエンコードできるかを評価します。
     * 
     * @param c
     *            評価する文字
     * @param charset
     *            文字セット
     * @return 指定の文字セットで指定された文字をエンコードできる場合にかぎり true
     * @throws UnsupportedOperationException
     *             この文字セットがエンコードをサポートしない場合
     */
    public static boolean isEncode(char c, Charset charset) throws UnsupportedOperationException {
        CharsetEncoder cEncoder = charset.newEncoder();
        return cEncoder.canEncode(c);
    }

    /**
     * 指定の文字セットで指定された文字シーケンスをエンコードできるかを評価します。
     * 
     * @param cs
     *            評価する文字シーケンス
     * @param charsetName
     *            要求された文字セットの名前 (標準名または別名)
     * @return 指定の文字セットで指定された文字をエンコードできる場合にかぎり true
     * @throws UnsupportedCharsetException
     *             指定された文字セット名が不当である場合
     * @throws IllegalCharsetNameException
     *             指定された文字セットを現在の Java 仮想マシンでは利用できない場合
     * @throws UnsupportedOperationException
     *             この文字セットがエンコードをサポートしない場合
     */
    public static boolean isEncode(CharSequence cs, String charsetName) throws UnsupportedCharsetException, IllegalCharsetNameException,
            UnsupportedOperationException {
        return isEncode(cs, Charset.forName(charsetName));
    }

    /**
     * 指定の文字セットで指定された文字シーケンスをエンコードできるかを評価します。
     * 
     * @param cs
     *            評価する文字シーケンス
     * @param charset
     *            文字セット
     * @return 指定の文字セットで指定された文字をエンコードできる場合にかぎり true
     * @throws UnsupportedOperationException
     *             この文字セットがエンコードをサポートしない場合
     */
    public static boolean isEncode(CharSequence cs, Charset charset) throws UnsupportedOperationException {
        CharsetEncoder cEncoder = charset.newEncoder();
        return cEncoder.canEncode(cs);
    }

    /*
     * Range
     */

    /**
     * 指定された数値が指定の範囲内か評価します。
     * 
     * @param n
     *            評価する数値
     * @param from
     *            範囲の最小値
     * @param to
     *            範囲の最大値
     * @return 指定の範囲内の場合のみ true
     */
    public static boolean isRange(int n, int from, int to) {
        if (from > to) {
            throw new IllegalArgumentException("from > to (" + from + " > " + to + ")");
        }
        return (from <= n && n <= to);
    }

    /**
     * 指定された数値が指定の範囲内か評価します。
     * 
     * @param n
     *            評価する数値
     * @param from
     *            範囲の最小値
     * @param to
     *            範囲の最大値
     * @return 指定の範囲内の場合のみ true
     */
    public static boolean isRange(long n, long from, long to) {
        if (from > to) {
            throw new IllegalArgumentException("from > to (" + from + " > " + to + ")");
        }
        return (from <= n && n <= to);
    }

    /**
     * 指定された数値が指定の範囲内か、数値比較演算子の比較基準で評価します。<br>
     * この比較基準では正のゼロと負のゼロを同値として、また片方または双方が非数の場合は常に false と評価します。<br>
     * つまり (+0.0f == -0.0f) = true、(NaN <==> NaN) = false、(NaN <==> !NaN) = false、(!NaN <==> NaN) = false と評価します。<br>
     * java.lang.Float と同一基準で比較する場合は isLongBitsRange(float, float, float) を使用してください。
     * 
     * @param n
     *            評価する数値
     * @param from
     *            範囲の最小値
     * @param to
     *            範囲の最大値
     * @return 指定の範囲内の場合のみ true
     */
    public static boolean isRange(float n, float from, float to) {
        if (from > to) {
            throw new IllegalArgumentException("from > to (" + from + " > " + to + ")");
        }
        return (isMin(n, from) && isMax(n, to));
    }

    /**
     * 指定された数値が指定の範囲内か、数値比較演算子の比較基準で評価します。<br>
     * この比較基準では正のゼロと負のゼロを同値として、また片方または双方が非数の場合は常に false と評価します。<br>
     * つまり (+0.0d == -0.0d) = true、(NaN <==> NaN) = false、(NaN <==> !NaN) = false、(!NaN <==> NaN) = false と評価します。<br>
     * java.lang.Double と同一基準で比較する場合は isLongBitsRange(double, double, double) を使用してください。
     * 
     * @param n
     *            評価する数値
     * @param from
     *            範囲の最小値
     * @param to
     *            範囲の最大値
     * @return 指定の範囲内の場合のみ true
     */
    public static boolean isRange(double n, double from, double to) {
        if (from > to) {
            throw new IllegalArgumentException("from > to (" + from + " > " + to + ")");
        }
        return (isMin(n, from) && isMax(n, to));
    }

    /**
     * 指定された数値が指定の範囲内か、Float.compare と同様の比較基準で評価します。<br>
     * この比較基準では正のゼロと負のゼロを区別し、双方が非数の場合は同値、非数を正の無限大よりも大きいと見做します。<br>
     * また比較基準は、すべての NaN 値を単一の正規 NaN 値に収納します。<br>
     * つまり +0.0f > -0.0f、Float.NaN == Float.NaN、Float.NaN > Float.POSITIVE_INFINITY と評価します。 <br>
     * 比較演算子と同一基準で比較する場合は isRange(float, float, float) を使用してください。
     * 
     * @param n
     *            評価する数値
     * @param from
     *            範囲の最小値
     * @param to
     *            範囲の最大値
     * @return 指定の範囲内の場合のみ true
     */
    public static boolean isLongBitsRange(float n, float from, float to) {
        if (Float.compare(from, to) > 0) {
            throw new IllegalArgumentException("from > to (" + from + " > " + to + ")");
        }
        return (isLongBitsMin(n, from) && isLongBitsMax(n, to));
    }

    /**
     * 指定された数値が指定の範囲内か、Double.compare と同様の比較基準で評価します。<br>
     * この比較基準では正のゼロと負のゼロを区別し、双方が非数の場合は同値、非数を正の無限大よりも大きいと見做します。<br>
     * また比較基準は、すべての NaN 値を単一の正規 NaN 値に収納します。<br>
     * つまり +0.0d > -0.0d、Double.NaN == Double.NaN、Double.NaN > Double.POSITIVE_INFINITY と評価します。 <br>
     * 比較演算子と同一基準で比較する場合は isRange(double, double, double) を使用してください。
     * 
     * @param n
     *            評価する数値
     * @param from
     *            範囲の最小値
     * @param to
     *            範囲の最大値
     * @return 指定の範囲内の場合のみ true
     */
    public static boolean isLongBitsRange(double n, double from, double to) {
        if (Double.compare(from, to) > 0) {
            throw new IllegalArgumentException("from > to (" + from + " > " + to + ")");
        }
        return (isLongBitsMin(n, from) && isLongBitsMax(n, to));
    }

    /**
     * 指定された数値が指定の最小値の範囲か評価します。
     * 
     * @param n
     *            評価する数値
     * @param min
     *            範囲の最小値
     * @return 指定の最小値の範囲の場合のみ true
     */
    public static boolean isMin(int n, int min) {
        return (min <= n);
    }

    /**
     * 指定された数値が指定の最小値の範囲か評価します。
     * 
     * @param n
     *            評価する数値
     * @param min
     *            範囲の最小値
     * @return 指定の最小値の範囲の場合のみ true
     */
    public static boolean isMin(long n, long min) {
        return (min <= n);
    }

    /**
     * 指定された数値が指定の最小値の範囲か、数値比較演算子の比較基準で評価します。<br>
     * この比較基準では正のゼロと負のゼロを同値として、また片方または双方が非数の場合は常に false と評価します。<br>
     * つまり (+0.0f == -0.0f) = true、(NaN <==> NaN) = false、(NaN <==> !NaN) = false、(!NaN <==> NaN) = false と評価します。<br>
     * java.lang.Float と同一基準で比較する場合は isLongBitsMin(float, float) を使用してください。
     * 
     * @param n
     *            評価する数値
     * @param min
     *            範囲の最小値
     * @return 指定の最小値の範囲の場合のみ true
     */
    public static boolean isMin(float n, float min) {
        return (min <= n);
    }

    /**
     * 指定された数値が指定の最小値の範囲か、数値比較演算子の比較基準で評価します。<br>
     * この比較基準では正のゼロと負のゼロを同値として、また片方または双方が非数の場合は常に false と評価します。<br>
     * つまり (+0.0d == -0.0d) = true、(NaN <==> NaN) = false、(NaN <==> !NaN) = false、(!NaN <==> NaN) = false と評価します。<br>
     * java.lang.Double と同一基準で比較する場合は isLongBitsMin(double, double) を使用してください。
     * 
     * @param n
     *            評価する数値
     * @param min
     *            範囲の最小値
     * @return 指定の最小値の範囲の場合のみ true
     */
    public static boolean isMin(double n, double min) {
        return (min <= n);
    }

    /**
     * 指定された数値が指定の最小値の範囲か、Float.compare と同様の比較基準で評価します。<br>
     * この比較基準では正のゼロと負のゼロを区別し、双方が非数の場合は同値、非数を正の無限大よりも大きいと見做します。<br>
     * また比較基準は、すべての NaN 値を単一の正規 NaN 値に収納します。<br>
     * つまり +0.0f > -0.0f、Float.NaN == Float.NaN、Float.NaN > Float.POSITIVE_INFINITY と評価します。 <br>
     * 比較演算子と同一基準で比較する場合は isMin(float, float) を使用してください。
     * 
     * @param n
     *            評価する数値
     * @param min
     *            範囲の最小値
     * @return 指定の最小値の範囲の場合のみ true
     */
    public static boolean isLongBitsMin(float n, float min) {
        return (Float.compare(n, min) >= 0);
    }

    /**
     * 指定された数値が指定の最小値の範囲か、Double.compare と同様の比較基準で評価します。<br>
     * この比較基準では正のゼロと負のゼロを区別し、双方が非数の場合は同値、非数を正の無限大よりも大きいと見做します。<br>
     * また比較基準は、すべての NaN 値を単一の正規 NaN 値に収納します。<br>
     * つまり +0.0d > -0.0d、Double.NaN == Double.NaN、Double.NaN > Double.POSITIVE_INFINITY と評価します。 <br>
     * 比較演算子と同一基準で比較する場合は isMin(double, double) を使用してください。
     * 
     * @param n
     *            評価する数値
     * @param min
     *            範囲の最小値
     * @return 指定の最小値の範囲の場合のみ true
     */
    public static boolean isLongBitsMin(double n, double min) {
        return (Double.compare(n, min) >= 0);
    }

    /**
     * 指定された数値が指定の最大値の範囲か評価します。
     * 
     * @param n
     *            評価する数値
     * @param max
     *            範囲の最大値
     * @return 指定の最大値の範囲の場合のみ true
     */
    public static boolean isMax(int n, int max) {
        return (n <= max);
    }

    /**
     * 指定された数値が指定の最大値の範囲か評価します。
     * 
     * @param n
     *            評価する数値
     * @param max
     *            範囲の最大値
     * @return 指定の最大値の範囲の場合のみ true
     */
    public static boolean isMax(long n, long max) {
        return (n <= max);
    }

    /**
     * 指定された数値が指定の最大値の範囲か、数値比較演算子の比較基準で評価します。<br>
     * この比較基準では正のゼロと負のゼロを同値として、また片方または双方が非数の場合は常に false と評価します。<br>
     * つまり (+0.0f == -0.0f) = true、(NaN <==> NaN) = false、(NaN <==> !NaN) = false、(!NaN <==> NaN) = false と評価します。<br>
     * java.lang.Float と同一基準で比較する場合は isLongBitsMax(float, float) を使用してください。
     * 
     * @param n
     *            評価する数値
     * @param max
     *            範囲の最大値
     * @return 指定の最大値の範囲の場合のみ true
     */
    public static boolean isMax(float n, float max) {
        return (n <= max);
    }

    /**
     * 指定された数値が指定の最大値の範囲か、数値比較演算子の比較基準で評価します。<br>
     * この比較基準では正のゼロと負のゼロを同値として、また片方または双方が非数の場合は常に false と評価します。<br>
     * つまり (+0.0d == -0.0d) = true、(NaN <==> NaN) = false、(NaN <==> !NaN) = false、(!NaN <==> NaN) = false と評価します。<br>
     * java.lang.Double と同一基準で比較する場合は isLongBitsMax(double, double) を使用してください。
     * 
     * @param n
     *            評価する数値
     * @param max
     *            範囲の最大値
     * @return 指定の最大値の範囲の場合のみ true
     */
    public static boolean isMax(double n, double max) {
        return (n <= max);
    }

    /**
     * 指定された数値が指定の最大値の範囲か、Float.compare と同様の比較基準で評価します。<br>
     * この比較基準では正のゼロと負のゼロを区別し、双方が非数の場合は同値、非数を正の無限大よりも大きいと見做します。<br>
     * また比較基準は、すべての NaN 値を単一の正規 NaN 値に収納します。<br>
     * つまり +0.0f > -0.0f、Float.NaN == Float.NaN、Float.NaN > Float.POSITIVE_INFINITY と評価します。 <br>
     * 比較演算子と同一基準で比較する場合は isMax(float, float) を使用してください。
     * 
     * @param n
     *            評価する数値
     * @param max
     *            範囲の最大値
     * @return 指定の最大値の範囲の場合のみ true
     */
    public static boolean isLongBitsMax(float n, float max) {
        return (Float.compare(n, max) <= 0);
    }

    /**
     * 指定された数値が指定の最大値の範囲か、Double.compare と同様の比較基準で評価します。<br>
     * この比較基準では正のゼロと負のゼロを区別し、双方が非数の場合は同値、非数を正の無限大よりも大きいと見做します。<br>
     * また比較基準は、すべての NaN 値を単一の正規 NaN 値に収納します。<br>
     * つまり +0.0d > -0.0d、Double.NaN == Double.NaN、Double.NaN > Double.POSITIVE_INFINITY と評価します。 <br>
     * 比較演算子と同一基準で比較する場合は isMax(double, double) を使用してください。
     * 
     * @param n
     *            評価する数値
     * @param max
     *            範囲の最大値
     * @return 指定の最大値の範囲の場合のみ true
     */
    public static boolean isLongBitsMax(double n, double max) {
        return (Double.compare(n, max) <= 0);
    }

    /*
     * char sequence size
     */

    /**
     * 指定された文字シーケンスの長さが指定の長さと同一か評価します。
     * 
     * @param cs
     *            評価する文字シーケンス
     * @param size
     *            文字シーケンスの長さ
     * @return 指定された文字シーケンスの長さが指定の長さと同一の場合のみ true
     */
    public static boolean isSize(CharSequence cs, int size) {
        return (cs.length() == size);
    }

    /**
     * 指定された文字シーケンスの長さが指定の範囲内か評価します。
     * 
     * @param cs
     *            評価する文字シーケンス
     * @param min
     *            文字数の最小値
     * @param max
     *            文字数の最大値
     * @return 文字シーケンスの長さが範囲内の場合のみ true
     */
    public static boolean isRangeSize(CharSequence cs, int min, int max) {
        return (isRange(cs.length(), min, max));
    }

    /**
     * 指定された文字シーケンスの長さが指定の最小値の範囲内か評価します。
     * 
     * @param cs
     *            評価する文字シーケンス
     * @param min
     *            文字数の最小値
     * @return 文字シーケンスの長さが最小値の範囲内の場合のみ true
     */
    public static boolean isMinSize(CharSequence cs, int min) {
        return (isMin(cs.length(), min));
    }

    /**
     * 指定された文字シーケンスの長さが指定の最大値の範囲内か評価します。
     * 
     * @param cs
     *            評価する文字シーケンス
     * @param max
     *            文字数の最大値
     * @return 文字シーケンスの長さが最大値の範囲内の場合のみ true
     */
    public static boolean isMaxSize(CharSequence cs, int max) {
        return (isMax(cs.length(), max));
    }

    /*
     * Negative
     */

    /**
     * 指定された数値が負数か評価します。
     * 
     * @param n
     *            評価する数値
     * @return 指定された数値が負数の場合のみ true
     */
    public static boolean isNegative(int n) {
        return (0 > n);
    }

    /**
     * 指定された数値が負数か評価します。
     * 
     * @param n
     *            評価する数値
     * @return 指定された数値が負数の場合のみ true
     */
    public static boolean isNegative(long n) {
        return (0 > n);
    }

    /**
     * 指定された数値が負数か、数値比較演算子の比較基準で評価します。<br>
     * この比較基準では正のゼロと負のゼロを同値として、また片方または双方が非数の場合は常に false と評価します。<br>
     * つまり (+0.0f == -0.0f) = true、(NaN <==> NaN) = false、(NaN <==> !NaN) = false、(!NaN <==> NaN) = false と評価します。<br>
     * java.lang.Float と同一基準で比較する場合は isLongBitsNegative(float) を使用してください。
     * 
     * @param n
     *            評価する数値
     * @return 指定された数値が負数の場合のみ true
     */
    public static boolean isNegative(float n) {
        return (0.0f > n);
    }

    /**
     * 指定された数値が負数か、数値比較演算子の比較基準で評価します。<br>
     * この比較基準では正のゼロと負のゼロを同値として、また片方または双方が非数の場合は常に false と評価します。<br>
     * つまり (+0.0d == -0.0d) = true、(NaN <==> NaN) = false、(NaN <==> !NaN) = false、(!NaN <==> NaN) = false と評価します。<br>
     * java.lang.Double と同一基準で比較する場合は isLongBitsNegative(double) を使用してください。
     * 
     * @param n
     *            評価する数値
     * @return 指定された数値が負数の場合のみ true
     */
    public static boolean isNegative(double n) {
        return (0.0d > n);
    }

    /**
     * 指定された数値が負数か、Float.compare と同様の比較基準で評価します。<br>
     * この比較基準では正のゼロと負のゼロを区別し、双方が非数の場合は同値、非数を正の無限大よりも大きいと見做します。<br>
     * また比較基準は、すべての NaN 値を単一の正規 NaN 値に収納します。<br>
     * つまり +0.0f > -0.0f、Float.NaN == Float.NaN、Float.NaN > Float.POSITIVE_INFINITY と評価します。 <br>
     * 比較演算子と同一基準で比較する場合は isNegative(float, float) を使用してください。
     * 
     * @param n
     *            評価する数値
     * @return 指定された数値が正数の場合のみ true
     */
    public static boolean isLongBitsNegative(float n) {
        return (Float.compare(n, 0.0f) < 0);
    }

    /**
     * 指定された数値が負数か、Double.compare と同様の比較基準で評価します。<br>
     * この比較基準では正のゼロと負のゼロを区別し、双方が非数の場合は同値、非数を正の無限大よりも大きいと見做します。<br>
     * また比較基準は、すべての NaN 値を単一の正規 NaN 値に収納します。<br>
     * つまり +0.0d > -0.0d、Double.NaN == Double.NaN、Double.NaN > Double.POSITIVE_INFINITY と評価します。 <br>
     * 比較演算子と同一基準で比較する場合は isNegative(double, double) を使用してください。
     * 
     * @param n
     *            評価する数値
     * @return 指定された数値が正数の場合のみ true
     */
    public static boolean isLongBitsNegative(double n) {
        return (Double.compare(n, 0.0d) < 0);
    }

    /**
     * 指定された数値がゼロまたは負数か評価します。
     * 
     * @param n
     *            評価する数値
     * @return 指定された数値がゼロまたは負数の場合のみ true
     */
    public static boolean isZeroOrNegative(int n) {
        return (0 >= n);
    }

    /**
     * 指定された数値がゼロまたは負数か評価します。
     * 
     * @param n
     *            評価する数値
     * @return 指定された数値がゼロまたは負数の場合のみ true
     */
    public static boolean isZeroOrNegative(long n) {
        return (0 >= n);
    }

    /**
     * 指定された数値がゼロまたは負数か、数値比較演算子の比較基準で評価します。<br>
     * この比較基準では正のゼロと負のゼロを同値として、また片方または双方が非数の場合は常に false と評価します。<br>
     * つまり (+0.0f == -0.0f) = true、(NaN <==> NaN) = false、(NaN <==> !NaN) = false、(!NaN <==> NaN) = false と評価します。<br>
     * java.lang.Float と同一基準で比較する場合は isLongBitsZeroOrNegative(float) を使用してください。
     * 
     * @param n
     *            評価する数値
     * @return 指定された数値がゼロまたは負数の場合のみ true
     */
    public static boolean isZeroOrNegative(float n) {
        return (0.0f >= n);
    }

    /**
     * 指定された数値がゼロまたは負数か、数値比較演算子の比較基準で評価します。<br>
     * この比較基準では正のゼロと負のゼロを同値として、また片方または双方が非数の場合は常に false と評価します。<br>
     * つまり (+0.0d == -0.0d) = true、(NaN <==> NaN) = false、(NaN <==> !NaN) = false、(!NaN <==> NaN) = false と評価します。<br>
     * java.lang.Double と同一基準で比較する場合は isLongBitsZeroOrNegative(double) を使用してください。
     * 
     * @param n
     *            評価する数値
     * @return 指定された数値がゼロまたは負数の場合のみ true
     */
    public static boolean isZeroOrNegative(double n) {
        return (0.0d >= n);
    }

    /**
     * 指定された数値がゼロまたは負数か、Float.compare と同様の比較基準で評価します。<br>
     * この比較基準では正のゼロと負のゼロを区別し、双方が非数の場合は同値、非数を正の無限大よりも大きいと見做します。<br>
     * また比較基準は、すべての NaN 値を単一の正規 NaN 値に収納します。<br>
     * つまり +0.0f > -0.0f、Float.NaN == Float.NaN、Float.NaN > Float.POSITIVE_INFINITY と評価します。 <br>
     * 比較演算子と同一基準で比較する場合は isZeroOrNegative(float, float) を使用してください。
     * 
     * @param n
     *            評価する数値
     * @return 指定された数値がゼロまたは負数の場合のみ true
     */
    public static boolean isLongBitsZeroOrNegative(float n) {
        return (Float.compare(n, 0.0f) <= 0);
    }

    /**
     * 指定された数値がゼロまたは負数か、Double.compare と同様の比較基準で評価します。<br>
     * この比較基準では正のゼロと負のゼロを区別し、双方が非数の場合は同値、非数を正の無限大よりも大きいと見做します。<br>
     * また比較基準は、すべての NaN 値を単一の正規 NaN 値に収納します。<br>
     * つまり +0.0d > -0.0d、Double.NaN == Double.NaN、Double.NaN > Double.POSITIVE_INFINITY と評価します。 <br>
     * 比較演算子と同一基準で比較する場合は isZeroOrNegative(double, double) を使用してください。
     * 
     * @param n
     *            評価する数値
     * @return 指定された数値がゼロまたは負数の場合のみ true
     */
    public static boolean isLongBitsZeroOrNegative(double n) {
        return (Double.compare(n, 0.0d) <= 0);
    }

    /*
     * Positive
     */

    /**
     * 指定された数値が正数か評価します。
     * 
     * @param n
     *            評価する数値
     * @return 指定された数値が正数の場合のみ true
     */
    public static boolean isPositive(int n) {
        return (0 < n);
    }

    /**
     * 指定された数値が正数か評価します。
     * 
     * @param n
     *            評価する数値
     * @return 指定された数値が正数の場合のみ true
     */
    public static boolean isPositive(long n) {
        return (0 < n);
    }

    /**
     * 指定された数値が正数か、数値比較演算子の比較基準で評価します。<br>
     * この比較基準では正のゼロと負のゼロを同値として、また片方または双方が非数の場合は常に false と評価します。<br>
     * つまり (+0.0f == -0.0f) = true、(NaN <==> NaN) = false、(NaN <==> !NaN) = false、(!NaN <==> NaN) = false と評価します。<br>
     * java.lang.Float と同一基準で比較する場合は isLongBitsPositive(float) を使用してください。
     * 
     * @param n
     *            評価する数値
     * @return 指定された数値が正数の場合のみ true
     */
    public static boolean isPositive(float n) {
        return (0.0f < n);
    }

    /**
     * 指定された数値が正数か、数値比較演算子の比較基準で評価します。<br>
     * この比較基準では正のゼロと負のゼロを同値として、また片方または双方が非数の場合は常に false と評価します。<br>
     * つまり (+0.0d == -0.0d) = true、(NaN <==> NaN) = false、(NaN <==> !NaN) = false、(!NaN <==> NaN) = false と評価します。<br>
     * java.lang.Double と同一基準で比較する場合は isLongBitsPositive(double) を使用してください。
     * 
     * @param n
     *            評価する数値
     * @return 指定された数値が正数の場合のみ true
     */
    public static boolean isPositive(double n) {
        return (0.0d < n);
    }

    /**
     * 指定された数値が正数か、Float.compare と同様の比較基準で評価します。<br>
     * この比較基準では正のゼロと負のゼロを区別し、双方が非数の場合は同値、非数を正の無限大よりも大きいと見做します。<br>
     * また比較基準は、すべての NaN 値を単一の正規 NaN 値に収納します。<br>
     * つまり +0.0f > -0.0f、Float.NaN == Float.NaN、Float.NaN > Float.POSITIVE_INFINITY と評価します。 <br>
     * 比較演算子と同一基準で比較する場合は isPositive(float, float) を使用してください。
     * 
     * @param n
     *            評価する数値
     * @return 指定された数値が正数の場合のみ true
     */
    public static boolean isLongBitsPositive(float n) {
        return (Float.compare(n, -0.0f) > 0);
    }

    /**
     * 指定された数値が正数か、Double.compare と同様の比較基準で評価します。<br>
     * この比較基準では正のゼロと負のゼロを区別し、双方が非数の場合は同値、非数を正の無限大よりも大きいと見做します。<br>
     * また比較基準は、すべての NaN 値を単一の正規 NaN 値に収納します。<br>
     * つまり +0.0d > -0.0d、Double.NaN == Double.NaN、Double.NaN > Double.POSITIVE_INFINITY と評価します。 <br>
     * 比較演算子と同一基準で比較する場合は isPositive(double, double) を使用してください。
     * 
     * @param n
     *            評価する数値
     * @return 指定された数値が正数の場合のみ true
     */
    public static boolean isLongBitsPositive(double n) {
        return (Double.compare(n, -0.0d) > 0);
    }

    /**
     * 指定された数値がゼロまたは正数か評価します。
     * 
     * @param n
     *            評価する数値
     * @return 指定された数値がゼロまたは正数の場合のみ true
     */
    public static boolean isZeroOrPositive(int n) {
        return (0 <= n);
    }

    /**
     * 指定された数値がゼロまたは正数か評価します。
     * 
     * @param n
     *            評価する数値
     * @return 指定された数値がゼロまたは正数の場合のみ true
     */
    public static boolean isZeroOrPositive(long n) {
        return (0 <= n);
    }

    /**
     * 指定された数値がゼロまたは正数か、数値比較演算子の比較基準で評価します。<br>
     * この比較基準では正のゼロと負のゼロを同値として、また片方または双方が非数の場合は常に false と評価します。<br>
     * つまり (+0.0f == -0.0f) = true、(NaN <==> NaN) = false、(NaN <==> !NaN) = false、(!NaN <==> NaN) = false と評価します。<br>
     * java.lang.Float と同一基準で比較する場合は isLongBitsZeroOrPositive(float) を使用してください。
     * 
     * @param n
     *            評価する数値
     * @return 指定された数値がゼロまたは正数の場合のみ true
     */
    public static boolean isZeroOrPositive(float n) {
        return (0.0f <= n);
    }

    /**
     * 指定された数値がゼロまたは正数か、数値比較演算子の比較基準で評価します。<br>
     * この比較基準では正のゼロと負のゼロを同値として、また片方または双方が非数の場合は常に false と評価します。<br>
     * つまり (+0.0d == -0.0d) = true、(NaN <==> NaN) = false、(NaN <==> !NaN) = false、(!NaN <==> NaN) = false と評価します。<br>
     * java.lang.Double と同一基準で比較する場合は isLongBitsZeroOrPositive(double) を使用してください。
     * 
     * @param n
     *            評価する数値
     * @return 指定された数値がゼロまたは正数の場合のみ true
     */
    public static boolean isZeroOrPositive(double n) {
        return (0.0d <= n);
    }

    /**
     * 指定された数値がゼロまたは正数か、Float.compare と同様の比較基準で評価します。<br>
     * この比較基準では正のゼロと負のゼロを区別し、双方が非数の場合は同値、非数を正の無限大よりも大きいと見做します。<br>
     * また比較基準は、すべての NaN 値を単一の正規 NaN 値に収納します。<br>
     * つまり +0.0f > -0.0f、Float.NaN == Float.NaN、Float.NaN > Float.POSITIVE_INFINITY と評価します。 <br>
     * 比較演算子と同一基準で比較する場合は isZeroOrPositive(float, float) を使用してください。
     * 
     * @param n
     *            評価する数値
     * @return 指定された数値がゼロまたは正数の場合のみ true
     */
    public static boolean isLongBitsZeroOrPositive(float n) {
        return (Float.compare(n, -0.0f) >= 0);
    }

    /**
     * 指定された数値がゼロまたは正数か、Double.compare と同様の比較基準で評価します。<br>
     * この比較基準では正のゼロと負のゼロを区別し、双方が非数の場合は同値、非数を正の無限大よりも大きいと見做します。<br>
     * また比較基準は、すべての NaN 値を単一の正規 NaN 値に収納します。<br>
     * つまり +0.0d > -0.0d、Double.NaN == Double.NaN、Double.NaN > Double.POSITIVE_INFINITY と評価します。 <br>
     * 比較演算子と同一基準で比較する場合は isZeroOrPositive(double, double) を使用してください。
     * 
     * @param n
     *            評価する数値
     * @return 指定された数値がゼロまたは正数の場合のみ true
     */
    public static boolean isLongBitsZeroOrPositive(double n) {
        return (Double.compare(n, -0.0d) >= 0);
    }

    /*
     * Infinite Or NaN
     */

    /**
     * 数値の絶対値が無限量か、または非数 (NaN) であるか評価します。
     * 
     * @param val
     *            評価する数値
     * @return 数値の絶対値が無限量か、または非数 (NaN) の場合のみ true
     */
    public static boolean isInfiniteOrNaN(Number val) {
        if (val instanceof Float) {
            return isInfiniteOrNaN((Float) val);
        }
        if (val instanceof Double) {
            return isInfiniteOrNaN((Double) val);
        }
        return false;
    }

    /**
     * 数値の絶対値が無限量か、または非数 (NaN) であるか評価します。
     * 
     * @param val
     *            評価する数値
     * @return 数値の絶対値が無限量か、または非数 (NaN) の場合のみ true
     */
    public static boolean isInfiniteOrNaN(Float val) {
        return (val.isInfinite() || val.isNaN());
    }

    /**
     * 数値の絶対値が無限量か、または非数 (NaN) であるか評価します。
     * 
     * @param val
     *            評価する数値
     * @return 数値の絶対値が無限量か、または非数 (NaN) の場合のみ true
     */
    public static boolean isInfiniteOrNaN(float val) {
        return (Float.isInfinite(val) || Float.isNaN(val));
    }

    /**
     * 数値の絶対値が無限量か、または非数 (NaN) であるか評価します。
     * 
     * @param val
     *            評価する数値
     * @return 数値の絶対値が無限量か、または非数 (NaN) の場合のみ true
     */
    public static boolean isInfiniteOrNaN(Double val) {
        return (val.isInfinite() || val.isNaN());
    }

    /**
     * 数値の絶対値が無限量か、または非数 (NaN) であるか評価します。
     * 
     * @param val
     *            評価する数値
     * @return 数値の絶対値が無限量か、または非数 (NaN) の場合のみ true
     */
    public static boolean isInfiniteOrNaN(double val) {
        return (Double.isInfinite(val) || Double.isNaN(val));
    }

    /*
     * Range All
     */

    /**
     * 指定された全ての数値が指定の範囲内か評価します。
     * 
     * @param a
     *            評価する数値
     * @param from
     *            範囲の最小値
     * @param to
     *            範囲の最大値
     * @return 指定の範囲内の場合のみ true
     */
    public static boolean isRangeAll(byte[] a, byte from, byte to) {
        for (int i = 0; i < a.length; i++) {
            if (!isRange(a[i], from, to)) {
                return false;
            }
        }
        return true;
    }

    /**
     * 指定された全ての数値が指定の範囲内か評価します。
     * 
     * @param a
     *            評価する数値
     * @param from
     *            範囲の最小値
     * @param to
     *            範囲の最大値
     * @return 指定の範囲内の場合のみ true
     */
    public static boolean isRangeAll(short[] a, short from, short to) {
        for (int i = 0; i < a.length; i++) {
            if (!isRange(a[i], from, to)) {
                return false;
            }
        }
        return true;
    }

    /**
     * 指定された全ての数値が指定の範囲内か評価します。
     * 
     * @param a
     *            評価する数値
     * @param from
     *            範囲の最小値
     * @param to
     *            範囲の最大値
     * @return 指定の範囲内の場合のみ true
     */
    public static boolean isRangeAll(int[] a, int from, int to) {
        for (int i = 0; i < a.length; i++) {
            if (!isRange(a[i], from, to)) {
                return false;
            }
        }
        return true;
    }

    /**
     * 指定された全ての数値が指定の範囲内か評価します。
     * 
     * @param a
     *            評価する数値
     * @param from
     *            範囲の最小値
     * @param to
     *            範囲の最大値
     * @return 指定の範囲内の場合のみ true
     */
    public static boolean isRangeAll(long[] a, long from, long to) {
        for (int i = 0; i < a.length; i++) {
            if (!isRange(a[i], from, to)) {
                return false;
            }
        }
        return true;
    }

    /**
     * 指定された全ての数値が指定の範囲内か、数値比較演算子の比較基準で評価します。<br>
     * この比較基準では正のゼロと負のゼロを同値として、また片方または双方が非数の場合は常に false と評価します。<br>
     * つまり (+0.0f == -0.0f) = true、(NaN <==> NaN) = false、(NaN <==> !NaN) = false、(!NaN <==> NaN) = false と評価します。<br>
     * java.lang.Float と同一基準で比較する場合は isLongBitsRangeAll(float, float, float) を使用してください。
     * 
     * @param a
     *            評価する数値
     * @param from
     *            範囲の最小値
     * @param to
     *            範囲の最大値
     * @return 指定の範囲内の場合のみ true
     */
    public static boolean isRangeAll(float[] a, float from, float to) {
        for (int i = 0; i < a.length; i++) {
            if (!isRange(a[i], from, to)) {
                return false;
            }
        }
        return true;
    }

    /**
     * 指定された全ての数値が指定の範囲内か、数値比較演算子の比較基準で評価します。<br>
     * この比較基準では正のゼロと負のゼロを同値として、また片方または双方が非数の場合は常に false と評価します。<br>
     * つまり (+0.0d == -0.0d) = true、(NaN <==> NaN) = false、(NaN <==> !NaN) = false、(!NaN <==> NaN) = false と評価します。<br>
     * java.lang.Double と同一基準で比較する場合は isLongBitsRangeAll(double, double, double) を使用してください。
     * 
     * @param a
     *            評価する数値
     * @param from
     *            範囲の最小値
     * @param to
     *            範囲の最大値
     * @return 指定の範囲内の場合のみ true
     */
    public static boolean isRangeAll(double[] a, double from, double to) {
        for (int i = 0; i < a.length; i++) {
            if (!isRange(a[i], from, to)) {
                return false;
            }
        }
        return true;
    }

    /**
     * 指定された全ての数値が指定の範囲内か、Float.compare と同様の比較基準で評価します。<br>
     * この比較基準では正のゼロと負のゼロを区別し、双方が非数の場合は同値、非数を正の無限大よりも大きいと見做します。<br>
     * また比較基準は、すべての NaN 値を単一の正規 NaN 値に収納します。<br>
     * つまり +0.0f > -0.0f、Float.NaN == Float.NaN、Float.NaN > Float.POSITIVE_INFINITY と評価します。 <br>
     * 比較演算子と同一基準で比較する場合は isRangeAll(float, float, float) を使用してください。
     * 
     * @param a
     *            評価する数値
     * @param from
     *            範囲の最小値
     * @param to
     *            範囲の最大値
     * @return 指定の範囲内の場合のみ true
     */
    public static boolean isLongBitsRangeAll(float[] a, float from, float to) {
        for (int i = 0; i < a.length; i++) {
            if (!isLongBitsRange(a[i], from, to)) {
                return false;
            }
        }
        return true;
    }

    /**
     * 指定された全ての数値が指定の範囲内か、Double.compare と同様の比較基準で評価します。<br>
     * この比較基準では正のゼロと負のゼロを区別し、双方が非数の場合は同値、非数を正の無限大よりも大きいと見做します。<br>
     * また比較基準は、すべての NaN 値を単一の正規 NaN 値に収納します。<br>
     * つまり +0.0d > -0.0d、Double.NaN == Double.NaN、Double.NaN > Double.POSITIVE_INFINITY と評価します。 <br>
     * 比較演算子と同一基準で比較する場合は isRangeAll(double, double, double) を使用してください。
     * 
     * @param a
     *            評価する数値
     * @param from
     *            範囲の最小値
     * @param to
     *            範囲の最大値
     * @return 指定の範囲内の場合のみ true
     */
    public static boolean isLongBitsRangeAll(double[] a, double from, double to) {
        for (int i = 0; i < a.length; i++) {
            if (!isLongBitsRange(a[i], from, to)) {
                return false;
            }
        }
        return true;
    }

}
