/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.core.helpers;

/**
 * ハッシュコードを算出するヘルパーメソッド群を提供します。
 */
public class HHash {

    /**
     * Byte.hashCode() と同一の方法で算出して返します。
     * 
     * @param src
     *            算出基
     * @return ハッシュコード
     */
    public static int hash(byte src) {
        return src;
    }

    /**
     * Short.hashCode() と同一の方法で算出して返します。
     * 
     * @param src
     *            算出基
     * @return ハッシュコード
     */
    public static int hash(short src) {
        return src;
    }

    /**
     * Integer.hashCode() と同一の方法で算出して返します。
     * 
     * @param src
     *            算出基
     * @return ハッシュコード
     */
    public static int hash(int src) {
        return src;
    }

    /**
     * Long.hashCode() と同一の方法で算出して返します。
     * 
     * @param src
     *            算出基
     * @return ハッシュコード
     */
    public static int hash(long src) {
        return (int) (src ^ (src >>> 32));
    }

    /**
     * Float.hashCode() と同一の方法で算出して返します。
     * 
     * @param src
     *            算出基
     * @return ハッシュコード
     */
    public static int hash(float src) {
        return Float.floatToIntBits(src);
    }

    /**
     * Double.hashCode() と同一の方法で算出して返します。
     * 
     * @param src
     *            算出基
     * @return ハッシュコード
     */
    public static int hash(double src) {
        long bits = Double.doubleToLongBits(src);
        return (int) (bits ^ (bits >>> 32));
    }

    /**
     * Boolean.hashCode() と同一の方法で算出して返します。
     * 
     * @param src
     *            算出基
     * @return ハッシュコード
     */
    public static int hash(boolean src) {
        return (src ? 1231 : 1237);
    }

    /**
     * Character.hashCode() と同一の方法で算出して返します。
     * 
     * @param src
     *            算出基
     * @return ハッシュコード
     */
    public static int hash(char src) {
        return src;
    }

    /**
     * ハッシュコードを (src == null) &#x3F; 0 : src.hashCode() で算出して返します。
     * 
     * @param src
     *            算出基
     * @return ハッシュコード
     */
    public static int hash(Object src) {
        return (src == null) ? 0 : src.hashCode();
    }

    /**
     * デフォルトのハッシュコードを (src == null) &#x3F; 0 : System.identityHashCode(src)）で算出して返します。
     * 
     * @param src
     *            算出基
     * @return ハッシュコード
     */
    public static int identityHash(Object src) {
        return (src == null) ? 0 : System.identityHashCode(src);
    }

    /**
     * 配列の要素のハッシュコードの総数を算出して返します。<br>
     * 引数が null の場合は 0 を返します。
     * 
     * @param src
     *            算出基の配列
     * @return ハッシュコードの総数
     */
    public static int hash(byte[] src) {
        if (src == null) {
            return 0;
        }
        int hash = 0;
        for (int i = 0; i < src.length; i++) {
            hash += src[i];
        }
        return hash;
    }

    /**
     * 配列の要素のハッシュコードの総数を算出して返します。<br>
     * 引数が null の場合は 0 を返します。
     * 
     * @param src
     *            算出基の配列
     * @return ハッシュコードの総数
     */
    public static int hash(short[] src) {
        if (src == null) {
            return 0;
        }
        int hash = 0;
        for (int i = 0; i < src.length; i++) {
            hash += src[i];
        }
        return hash;
    }

    /**
     * 配列の要素のハッシュコードの総数を算出して返します。<br>
     * 引数が null の場合は 0 を返します。
     * 
     * @param src
     *            算出基の配列
     * @return ハッシュコードの総数
     */
    public static int hash(int[] src) {
        if (src == null) {
            return 0;
        }
        int hash = 0;
        for (int i = 0; i < src.length; i++) {
            hash += src[i];
        }
        return hash;
    }

    /**
     * 配列の要素のハッシュコードの総数を算出して返します。<br>
     * 引数が null の場合は 0 を返します。
     * 
     * @param src
     *            算出基の配列
     * @return ハッシュコードの総数
     */
    public static int hash(long[] src) {
        if (src == null) {
            return 0;
        }
        int hash = 0;
        for (int i = 0; i < src.length; i++) {
            hash += hash(src[i]);
        }
        return hash;
    }

    /**
     * 配列の要素のハッシュコードの総数を算出して返します。<br>
     * 引数が null の場合は 0 を返します。
     * 
     * @param src
     *            算出基の配列
     * @return ハッシュコードの総数
     */
    public static int hash(float[] src) {
        if (src == null) {
            return 0;
        }
        int hash = 0;
        for (int i = 0; i < src.length; i++) {
            hash += hash(src[i]);
        }
        return hash;
    }

    /**
     * 配列の要素のハッシュコードの総数を算出して返します。<br>
     * 引数が null の場合は 0 を返します。
     * 
     * @param src
     *            算出基の配列
     * @return ハッシュコードの総数
     */
    public static int hash(double[] src) {
        if (src == null) {
            return 0;
        }
        int hash = 0;
        for (int i = 0; i < src.length; i++) {
            hash += hash(src[i]);
        }
        return hash;
    }

    /**
     * 配列の要素のハッシュコードの総数を算出して返します。<br>
     * 引数が null の場合は 0 を返します。
     * 
     * @param src
     *            算出基の配列
     * @return ハッシュコードの総数
     */
    public static int hash(boolean[] src) {
        if (src == null) {
            return 0;
        }
        int hash = 0;
        for (int i = 0; i < src.length; i++) {
            hash += hash(src[i]);
        }
        return hash;
    }

    /**
     * 配列の要素のハッシュコードの総数を算出して返します。<br>
     * 引数が null の場合は 0 を返します。
     * 
     * @param src
     *            算出基の配列
     * @return ハッシュコードの総数
     */
    public static int hash(char[] src) {
        if (src == null) {
            return 0;
        }
        int hash = 0;
        for (int i = 0; i < src.length; i++) {
            hash += src[i];
        }
        return hash;
    }

    /**
     * 配列の要素のハッシュコードを (src == null) &#x3F; 0 : src.hashCode() で算出して総数を返します。<br>
     * 引数が null の場合は 0 を返します。
     * 
     * @param src
     *            算出基の配列
     * @return ハッシュコードの総数
     */
    public static int hash(Object[] src) {
        if (src == null) {
            return 0;
        }
        int hash = 0;
        for (int i = 0; i < src.length; i++) {
            hash += hash(src[i]);
        }
        return hash;
    }

    /**
     * 配列の要素のデフォルトのハッシュコードを (src == null) &#x3F; 0 : System.identityHashCode(src）で算出して総数を返します。<br>
     * 引数が null の場合は 0 を返します。
     * 
     * @param src
     *            算出基のオブジェクト配列
     * @return デフォルトのハッシュコードの総数
     */
    public static int identityHash(Object[] src) {
        if (src == null) {
            return 0;
        }
        int hash = 0;
        for (int i = 0; i < src.length; i++) {
            hash += identityHash(src[i]);
        }
        return hash;
    }

}
