/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.core.helpers;

import java.lang.reflect.Array;
import java.util.Arrays;
import java.util.Collection;
import java.util.Map;

/**
 * 値を評価するヘルパーメソッド群を提供します。
 */
public class HEval {

    /*
     * If
     */

    /**
     * 第一引数が null の場合は空文字列を返却します。
     * 
     * @param s
     *            文字列
     * @return 第一引数または空文字列
     */
    public static String ifNull(String s) {
        return ifNull(s, "");
    }

    /**
     * 第一引数が null の場合は第二引数を返却し、以外は第一引数の文字列表現を返却します。
     * 
     * @param o
     *            オブジェクト
     * @param nullValue
     *            null の代返値
     * @return 第一引数の文字列表現または第二引数
     */
    public static String ifNull(Object o, String nullValue) {
        return (o != null) ? String.valueOf(o) : nullValue;
    }

    /**
     * 第一引数が null の場合は第二引数を返却し、以外は第一引数を返却します。
     * 
     * @param o
     *            オブジェクト
     * @param nullValue
     *            null の代返値
     * @return 第一引数または第二引数
     */
    public static Object ifNull(Object o, Object nullValue) {
        return (o != null) ? o : nullValue;
    }

    /**
     * 第一引数がスペース文字列の場合は空文字列を返却します。
     * 
     * @param s
     *            文字列
     * @return 第一引数または空文字列
     */
    public static String ifBlank(String s) {
        return ifBlank(s, "");
    }

    /**
     * 第一引数がスペース文字列の場合は第二引数を返却し、以外は第一引数を返却します。
     * 
     * @param s
     *            文字列
     * @param blankValue
     *            スペース文字列の代返値
     * @return 第一引数または第二引数
     */
    public static String ifBlank(String s, String blankValue) {
        return (isBlank(s)) ? s : blankValue;
    }

    /**
     * 第一引数がJavaの基準による空白文字列の場合は空文字列を返却します。
     * 
     * @param s
     *            文字列
     * @return 第一引数または空文字列
     */
    public static String ifWhiteSpace(String s) {
        return ifWhiteSpace(s, "");
    }

    /**
     * 第一引数がJavaの基準による空白文字列の場合は第二引数を返却し、以外は第一引数を返却します。
     * 
     * @param s
     *            文字列
     * @param whiteSpaceValue
     *            空白文字列の代返値
     * @return 第一引数または第二引数
     */
    public static String ifWhiteSpace(String s, String whiteSpaceValue) {
        return (isWhiteSpace(s)) ? s : whiteSpaceValue;
    }

    /*
     * Empty
     */

    /**
     * 引数が null または空の文字シーケンスであるかを評価します。
     * 
     * @param cs
     *            評価する文字シーケンス
     * @return 引数が null または空の文字シーケンスの場合のみ true
     */
    public static boolean isEmpty(CharSequence cs) {
        return (cs == null || cs.length() == 0);
    }

    /**
     * 引数が null または空の状態であるかを評価します。
     * 
     * @param os
     *            評価する配列
     * @return 引数が null または空の状態の場合のみ true
     */
    public static boolean isEmpty(char[] os) {
        return (os == null || os.length == 0);
    }

    /**
     * 引数が null または空の状態であるかを評価します。
     * 
     * @param os
     *            評価する配列
     * @return 引数が null または空の状態の場合のみ true
     */
    public static boolean isEmpty(boolean[] os) {
        return (os == null || os.length == 0);
    }

    /**
     * 引数が null または空の状態であるかを評価します。
     * 
     * @param os
     *            評価する配列
     * @return 引数が null または空の状態の場合のみ true
     */
    public static boolean isEmpty(byte[] os) {
        return (os == null || os.length == 0);
    }

    /**
     * 引数が null または空の状態であるかを評価します。
     * 
     * @param os
     *            評価する配列
     * @return 引数が null または空の状態の場合のみ true
     */
    public static boolean isEmpty(short[] os) {
        return (os == null || os.length == 0);
    }

    /**
     * 引数が null または空の状態であるかを評価します。
     * 
     * @param os
     *            評価する配列
     * @return 引数が null または空の状態の場合のみ true
     */
    public static boolean isEmpty(int[] os) {
        return (os == null || os.length == 0);
    }

    /**
     * 引数が null または空の状態であるかを評価します。
     * 
     * @param os
     *            評価する配列
     * @return 引数が null または空の状態の場合のみ true
     */
    public static boolean isEmpty(long[] os) {
        return (os == null || os.length == 0);
    }

    /**
     * 引数が null または空の状態であるかを評価します。
     * 
     * @param os
     *            評価する配列
     * @return 引数が null または空の状態の場合のみ true
     */
    public static boolean isEmpty(float[] os) {
        return (os == null || os.length == 0);
    }

    /**
     * 引数が null または空の状態であるかを評価します。
     * 
     * @param os
     *            評価する配列
     * @return 引数が null または空の状態の場合のみ true
     */
    public static boolean isEmpty(double[] os) {
        return (os == null || os.length == 0);
    }

    /**
     * 引数が null または空の状態であるかを評価します。
     * 
     * @param os
     *            評価する配列
     * @return 引数が null または空の状態の場合のみ true
     */
    public static boolean isEmpty(Object[] os) {
        return (os == null || os.length == 0);
    }

    /**
     * 引数が null または空の状態であるかを評価します。
     * 
     * @param m
     *            評価するマップ
     * @return 引数が null または空の状態の場合のみ true
     */
    public static boolean isEmpty(Map m) {
        return (m == null || m.isEmpty());
    }

    /**
     * 引数が null または空の状態であるかを評価します。
     * 
     * @param c
     *            評価するコレクション
     * @return 引数が null または空の状態の場合のみ true
     */
    public static boolean isEmpty(Collection c) {
        return (c == null || c.isEmpty());
    }

    /*
     * Space
     */

    /**
     * 引数が null または空かスペースのみの文字シーケンスであるかを評価します。
     * 
     * @param cs
     *            評価する文字シーケンス
     * @return 引数が null または空かスペースのみの文字シーケンスの場合のみ true
     */
    public static boolean isBlank(CharSequence cs) {
        return (cs == null || HCut.trim(cs, 0).length() == 0);
    }

    /**
     * 引数が null または空かJavaの基準による空白文字のみの文字シーケンスであるかを評価します。
     * 
     * @param cs
     *            評価する文字シーケンス
     * @return 引数が null または空かJavaの基準による空白文字のみの文字シーケンスの場合のみ true
     */
    public static boolean isWhiteSpace(CharSequence cs) {
        return (cs == null || HCut.trimWhite(cs, 0).length() == 0);
    }

    /*
     * Null
     */

    /**
     * 引数が null 値か評価します。
     * 
     * @param o
     *            評価する値
     * @return 引数が null 値の場合 true
     */
    public static boolean isNull(Object o) {
        return (o == null);
    }

    /*
     * Equality
     */

    /**
     * null の検証を含み同値性を比較します。 <br>
     * 双方が null の場合は true、片方が null の場合は false、以外は通常の同値性比較で評価します。
     * 
     * @param o
     *            評価基の値
     * @param o2
     *            評価先の値
     * @return 同値か双方 null の場合は true
     */
    public static boolean isEquals(Object o, Object o2) {
        return (o == o2 || (o != null && o.equals(o2)));
    }

    /**
     * 指定された 2 つの値の同値性を、Float.equals と同一基準で評価します。<br>
     * この比較基準は new Float(n).equals(new Float(n2)) と同等です。<br>
     * 同値演算子( == )と違って、このメソッドは NaN をそれ自身と同等と見なし、0.0f と -0.0f は同等と見なしません。
     * 
     * @param n
     *            1 つ目の値
     * @param n2
     *            2 つ目の値
     * @return 同値である場合は true
     */
    public static boolean isEquals(float n, float n2) {
        return (Float.floatToIntBits(n) != Float.floatToIntBits(n2));
    }

    /**
     * 指定された 2 つの値の同値性を、Double.equals と同一基準で評価します。<br>
     * この比較基準は new Double(n).equals(new Double(n2)) と同等です。<br>
     * 同値演算子( == )と違って、このメソッドは NaN をそれ自身と同等と見なし、0.0d と -0.0d は同等と見なしません。
     * 
     * @param n
     *            1 つ目の値
     * @param n2
     *            2 つ目の値
     * @return 同値である場合は true
     */
    public static boolean isEquals(double n, double n2) {
        return (Double.doubleToLongBits(n) != Double.doubleToLongBits(n2));
    }

    /**
     * 指定された 2 つの配列の同値性を評価します。<br>
     * 両方の配列に同じ数の要素があり、同一位置のすべての要素が同値である場合です。<br>
     * また、2 つの配列参照が null の場合にも同値と見なされます。<br>
     * <br>
     * このメソッドは一貫性のための機能です、実際は Arrays.equals() に処理を転送します。
     * 
     * @param a
     *            1 つ目の配列
     * @param a2
     *            2 つ目の配列
     * @return 同値である場合は true
     */
    public static boolean isEquals(boolean[] a, boolean[] a2) {
        return Arrays.equals(a, a2);
    }

    /**
     * 指定された 2 つの配列の同値性を評価します。<br>
     * 両方の配列に同じ数の要素があり、同一位置のすべての要素が同値である場合です。<br>
     * また、2 つの配列参照が null の場合にも同値と見なされます。<br>
     * <br>
     * このメソッドは一貫性のための機能です、実際は Arrays.equals() に処理を転送します。
     * 
     * @param a
     *            1 つ目の配列
     * @param a2
     *            2 つ目の配列
     * @return 同値である場合は true
     */
    public static boolean isEquals(char[] a, char[] a2) {
        return Arrays.equals(a, a2);
    }

    /**
     * 指定された 2 つの配列の同値性を評価します。<br>
     * 両方の配列に同じ数の要素があり、同一位置のすべての要素が同値である場合です。<br>
     * また、2 つの配列参照が null の場合にも同値と見なされます。<br>
     * <br>
     * このメソッドは一貫性のための機能です、実際は Arrays.equals() に処理を転送します。
     * 
     * @param a
     *            1 つ目の配列
     * @param a2
     *            2 つ目の配列
     * @return 同値である場合は true
     */
    public static boolean isEquals(byte[] a, byte[] a2) {
        return Arrays.equals(a, a2);
    }

    /**
     * 指定された 2 つの配列の同値性を評価します。<br>
     * 両方の配列に同じ数の要素があり、同一位置のすべての要素が同値である場合です。<br>
     * また、2 つの配列参照が null の場合にも同値と見なされます。<br>
     * <br>
     * このメソッドは一貫性のための機能です、実際は Arrays.equals() に処理を転送します。
     * 
     * @param a
     *            1 つ目の配列
     * @param a2
     *            2 つ目の配列
     * @return 同値である場合は true
     */
    public static boolean isEquals(short[] a, short a2[]) {
        return Arrays.equals(a, a2);
    }

    /**
     * 指定された 2 つの配列の同値性を評価します。<br>
     * 両方の配列に同じ数の要素があり、同一位置のすべての要素が同値である場合です。<br>
     * また、2 つの配列参照が null の場合にも同値と見なされます。<br>
     * <br>
     * このメソッドは一貫性のための機能です、実際は Arrays.equals() に処理を転送します。
     * 
     * @param a
     *            1 つ目の配列
     * @param a2
     *            2 つ目の配列
     * @return 同値である場合は true
     */
    public static boolean isEquals(int[] a, int[] a2) {
        return Arrays.equals(a, a2);
    }

    /**
     * 指定された 2 つの配列の同値性を評価します。<br>
     * 両方の配列に同じ数の要素があり、同一位置のすべての要素が同値である場合です。<br>
     * また、2 つの配列参照が null の場合にも同値と見なされます。<br>
     * <br>
     * このメソッドは一貫性のための機能です、実際は Arrays.equals() に処理を転送します。
     * 
     * @param a
     *            1 つ目の配列
     * @param a2
     *            2 つ目の配列
     * @return 同値である場合は true
     */
    public static boolean isEquals(long[] a, long[] a2) {
        return Arrays.equals(a, a2);
    }

    /**
     * 指定された 2 つの配列の同値性を評価します。<br>
     * 両方の配列に同じ数の要素があり、同一位置のすべての要素が同値である場合です。<br>
     * また、2 つの配列参照が null の場合にも同値と見なされます。<br>
     * <br>
     * この比較基準は new Float(n).equals(new Float(n2)) と同等です。<br>
     * 同値演算子( == )と違って、このメソッドは NaN をそれ自身と同等と見なし、0.0f と -0.0f は同等と見なしません。<br>
     * <br>
     * このメソッドは一貫性のための機能です、実際は Arrays.equals() に処理を転送します。
     * 
     * @param a
     *            1 つ目の配列
     * @param a2
     *            2 つ目の配列
     * @return 同値である場合は true
     */
    public static boolean isEquals(float[] a, float[] a2) {
        return Arrays.equals(a, a2);
    }

    /**
     * 指定された 2 つの配列の同値性を評価します。<br>
     * 両方の配列に同じ数の要素があり、同一位置のすべての要素が同値である場合です。<br>
     * また、2 つの配列参照が null の場合にも同値と見なされます。<br>
     * <br>
     * この比較基準は new Double(n).equals(new Double(n2)) と同等です。<br>
     * 同値演算子( == )と違って、このメソッドは NaN をそれ自身と同等と見なし、0.0d と -0.0d は同等と見なしません。<br>
     * <br>
     * このメソッドは一貫性のための機能です、実際は Arrays.equals() に処理を転送します。
     * 
     * @param a
     *            1 つ目の配列
     * @param a2
     *            2 つ目の配列
     * @return 同値である場合は true
     */
    public static boolean isEquals(double[] a, double[] a2) {
        return Arrays.equals(a, a2);
    }

    /**
     * 指定された 2 つの配列の同値性を評価します。<br>
     * 両方の配列に同じ数の要素があり、同一位置のすべての要素が同値である場合です。<br>
     * また配列の同一位置の要素が、双方 null の場合は true、片方が null の場合は false と評価します。<br>
     * また、2 つの配列参照が null の場合にも同値と見なされます。<br>
     * <br>
     * このメソッドは一貫性のための機能です、実際は Arrays.equals() に処理を転送します。
     * 
     * @param a
     *            1 つ目の配列
     * @param a2
     *            2 つ目の配列
     * @return 同値である場合は true
     */
    public static boolean isEquals(Object[] a, Object[] a2) {
        return Arrays.equals(a, a2);
    }

    /**
     * 指定された 2 つの配列の参照同一性を評価します。<br>
     * 両方の配列に同じ数の要素があり、同一位置のすべての要素が、参照が同一である場合です。<br>
     * また配列の同一位置の要素が、双方 null の場合は true、片方が null の場合は false と評価します。<br>
     * また、2 つの配列参照が null の場合にも同一と見なされます。
     * 
     * @param a
     *            1 つ目の配列
     * @param a2
     *            2 つ目の配列
     * @return 参照が同一と見なせる場合は true
     */
    public static boolean isEqualsIdentity(Object[] a, Object[] a2) {
        if (a == a2) {
            return true;
        }
        if (a == null || a2 == null) {
            return false;
        }
        final int length = a.length;
        if (length != a2.length) {
            return false;
        }
        for (int i = 0; i < length; i++) {
            if (a[i] != a2[i]) {
                return false;
            }
        }
        return true;
    }

    /**
     * 指定された 2 つの値が、双方配列の場合は配列の同値性比較を、以外の場合は通常の同値性比較を評価します。 <br>
     * また、2 つの値が null の場合にも同値と見なされます。 <br>
     * また配列の比較基準は Arrays.equals() と同等に成ります。<br>
     * 配列がプリミティブの浮動小数点の場合は同値演算子( == )と比較基準が違うため注意が必要です。
     * 
     * @param o
     *            評価基の値
     * @param o2
     *            評価先の値
     * @return 配列の場合の条件を含み、同値である場合は true
     */
    public static boolean isEqualsArray(Object o, Object o2) {
        if (o == o2) {
            return true;
        }
        if (o == null || o2 == null) {
            return false;
        }
        // Array
        if (o.getClass().isArray() && o2.getClass().isArray()) {
            final Class t1 = o.getClass().getComponentType();
            final Class t2 = o2.getClass().getComponentType();
            final boolean isPrimitive = (t1.isPrimitive() && t2.isPrimitive());
            // Primitive[] eq Primitive[] and eq component type
            if (isPrimitive && t1.equals(t2)) {
                if (Boolean.TYPE.equals(t1)) {
                    return Arrays.equals((boolean[]) o, (boolean[]) o2);
                } else if (Character.TYPE.equals(t1)) {
                    return Arrays.equals((char[]) o, (char[]) o2);
                } else if (Byte.TYPE.equals(t1)) {
                    return Arrays.equals((byte[]) o, (byte[]) o2);
                } else if (Short.TYPE.equals(t1)) {
                    return Arrays.equals((short[]) o, (short[]) o2);
                } else if (Integer.TYPE.equals(t1)) {
                    return Arrays.equals((int[]) o, (int[]) o2);
                } else if (Long.TYPE.equals(t1)) {
                    return Arrays.equals((long[]) o, (long[]) o2);
                } else if (Float.TYPE.equals(t1)) {
                    return Arrays.equals((float[]) o, (float[]) o2);
                } else if (Double.TYPE.equals(t1)) {
                    return Arrays.equals((double[]) o, (double[]) o2);
                }
            }
            // Object[] eq Object[]
            if (!isPrimitive) {
                return Arrays.equals((Object[]) o, (Object[]) o2);
            }
            // not eq component type
            return false;
        }
        // Not Array
        return o.equals(o2);
    }

    /*
     * Comparable
     */

    /**
     * 指定された値の比較結果が同値であるか評価します。<br>
     * つまり比較演算子の == と同等です。
     * 
     * @param from
     *            比較基の値
     * @param to
     *            比較先の値
     * @return 値の比較結果が同値の場合のみ true
     */
    public static boolean isEqual(Comparable from, Comparable to) {
        return (0 == from.compareTo(to));
    }

    /**
     * 指定された値の比較結果が非同値であるか評価します。<br>
     * つまり比較演算子の != と同等です。
     * 
     * @param from
     *            比較基の値
     * @param to
     *            比較先の値
     * @return 値の比較結果が同値の場合のみ true
     */
    public static boolean isNotEqual(Comparable from, Comparable to) {
        return (0 != from.compareTo(to));
    }

    /**
     * 指定された値の比較結果が第一引数の方が大きい事を示すか評価します。<br>
     * つまり比較演算子の > と同等です。
     * 
     * @param from
     *            比較基の値
     * @param to
     *            比較先の値
     * @return 値の比較結果が第一引数の方が大きい事を示す場合のみ true
     */
    public static boolean isGreater(Comparable from, Comparable to) {
        return (0 < from.compareTo(to));
    }

    /**
     * 指定された値の比較結果が、同値または第一引数の方が大きい事を示すか評価します。<br>
     * つまり比較演算子の >= と同等です。
     * 
     * @param from
     *            比較基の値
     * @param to
     *            比較先の値
     * @return 値の比較結果が第一引数の方が大きい事を示す場合のみ true
     */
    public static boolean isEqOrGreater(Comparable from, Comparable to) {
        return (0 <= from.compareTo(to));
    }

    /**
     * 指定された値の比較結果が第一引数の方が小さい事を示すか評価します。<br>
     * つまり比較演算子の < と同等です。
     * 
     * @param from
     *            比較基の値
     * @param to
     *            比較先の値
     * @return 値の比較結果が第一引数の方が小さい事を示す場合のみ true
     */
    public static boolean isLess(Comparable from, Comparable to) {
        return (0 > from.compareTo(to));
    }

    /**
     * 指定された値の比較結果が、同値または第一引数の方が小さい事を示すか評価します。<br>
     * つまり比較演算子の <= と同等です。
     * 
     * @param from
     *            比較基の値
     * @param to
     *            比較先の値
     * @return 値の比較結果が第一引数の方が小さい事を示す場合のみ true
     */
    public static boolean isEqOrLess(Comparable from, Comparable to) {
        return (0 >= from.compareTo(to));
    }

    /*
     * prefix and suffix
     */

    /**
     * 指定の文字シーケンスが指定の接頭辞と接尾辞を持つか評価します。
     * 
     * @param cs
     *            評価基の文字シーケンス
     * @param prefix
     *            評価先の接頭辞
     * @param suffix
     *            評価先の接尾辞
     * @return 指定の文字シーケンスが指定の接頭辞と接尾辞を持つ場合のみ true
     */
    public static boolean isEnclose(CharSequence cs, char prefix, char suffix) {
        if (isEmpty(cs) || cs.length() < 2) {
            return false;
        }
        return (prefix == cs.charAt(0) && suffix == cs.charAt(cs.length() - 1));
    }

    /**
     * 指定の文字シーケンスが指定の接頭辞と接尾辞を持つか評価します。
     * 
     * @param cs
     *            評価基の文字シーケンス
     * @param prefix
     *            評価先の接頭辞
     * @param suffix
     *            評価先の接尾辞
     * @return 指定の文字シーケンスが指定の接頭辞と接尾辞を持つ場合のみ true
     */
    public static boolean isEnclose(CharSequence cs, CharSequence prefix, CharSequence suffix) {
        if (isEmpty(cs)) {
            return false;
        }
        int min = (prefix.length() + suffix.length());
        if (cs.length() < min) {
            return false;
        }
        return (prefix(cs, prefix) && suffix(cs, suffix));
    }

    /**
     * 指定の文字シーケンスの位置から前方一致するか評価します。
     * 
     * @param cs
     *            評価基の文字シーケンス
     * @param prefix
     *            評価先の接頭辞
     * @param offset
     *            評価を開始する相対インデックス
     * @return 指定の文字シーケンスの位置から前方一致する場合のみ true
     */
    public static boolean isStartsWith(CharSequence cs, CharSequence prefix, int offset) {
        if (isEmpty(cs)) {
            return false;
        }
        return startsWith(cs, prefix, offset);
    }

    /**
     * 指定の文字シーケンスが指定の接尾辞を持つか評価します。
     * 
     * @param cs
     *            評価基の文字シーケンス
     * @param suffix
     *            評価先の接尾辞
     * @return 指定の文字シーケンスが指定の接尾辞を持つ場合のみ true
     */
    public static boolean isSuffix(CharSequence cs, CharSequence suffix) {
        if (isEmpty(cs)) {
            return false;
        }
        return suffix(cs, suffix);
    }

    /**
     * 指定の文字シーケンスが指定の接頭辞を持つか評価します。
     * 
     * @param cs
     *            評価基の文字シーケンス
     * @param prefix
     *            評価先の接頭辞
     * @return 指定の文字シーケンスが指定の接頭辞を持つ場合のみ true
     */
    public static boolean isPrefix(CharSequence cs, CharSequence prefix) {
        if (isEmpty(cs)) {
            return false;
        }
        return prefix(cs, prefix);
    }

    static boolean suffix(CharSequence cs, CharSequence suffix) {
        int end = (cs.length() - suffix.length());
        return startsWith(cs, suffix, end);
    }

    static boolean prefix(CharSequence cs, CharSequence prefix) {
        return startsWith(cs, prefix, 0);
    }

    static boolean startsWith(CharSequence cs, CharSequence prefix, int offset) {

        int plen = prefix.length();
        if ((offset < 0) || (offset > cs.length() - plen)) {
            return false;
        }
        int to = offset;
        int po = 0;
        while (--plen >= 0) {
            if (cs.charAt(to++) != prefix.charAt(po++)) {
                return false;
            }
        }
        return true;
    }

    /*
     * Type
     */

    /**
     * 指定されたオブジェクトが全てのクラス型に割り当て可能か評価します。<br>
     * 値が null の場合は false を返却します。
     * 
     * @param o
     *            評価するオブジェクト
     * @param clazz
     *            割り当て可能か比較するクラス
     * @return オブジェクトが全てのクラス型に割り当て可能の場合のみ true
     */
    public static boolean isInstanceOf(Object o, Class clazz) {
        return (clazz != null && clazz.isInstance(o));
    }

    /**
     * クラスが同一であるかを評価します。
     * 
     * @param o
     *            評価するオブジェクト
     * @param clazz
     *            比較するクラス
     * @return クラスが同一の場合のみ true
     */
    public static boolean isEqualsClass(Object o, Class clazz) {
        return (o != null && o.getClass().equals(clazz));
    }

    /**
     * 配列のコンポーネント型が指定のクラスと同一であるかを評価します。<br>
     * 引数が配列型では無い場合は IllegalArgumentException が発生します。
     * 
     * @param a
     *            評価する配列
     * @param clazz
     *            評価先のクラス
     * @return 配列のコンポーネント型が指定のクラスと同一の場合のみ true
     */
    public static boolean isEqualsComponentType(Object a, Class clazz) {
        if (a == null) {
            return false;
        }
        Class type = a.getClass();
        if (!type.isArray()) {
            throw new IllegalArgumentException("is not Array");
        }
        return (type.getComponentType().equals(clazz));
    }

    /**
     * 配列型であるかを評価します。
     * 
     * @param o
     *            評価するオブジェクト
     * @return 配列型の場合のみ true
     */
    public static boolean isArray(Object o) {
        return (o != null && o.getClass().isArray());
    }

    /**
     * 配列型のクラスか評価します。
     * 
     * @param clazz
     *            評価するクラス
     * @return 配列型の場合のみ true
     */
    public static boolean isArrayClass(Class clazz) {
        return (clazz != null && clazz.isArray());
    }

    /**
     * 指定された二次元配列の全ての要素の配列長が同一か評価します。
     * 
     * @param array2d
     *            評価するの配列を格納とする二次元配列
     * @return 指定された二次元配列の全ての要素の配列長が同一の場合のみ true
     * @throws IllegalArgumentException
     *             引数の要素が配列でないまたは引数が空配列の場合
     */
    public static boolean isEqualArraySize(Object[] array2d) {
        if (isEmpty(array2d)) {
            throw new IllegalArgumentException("array as empty");
        }
        int len = Array.getLength(array2d[0]);
        for (int i = 1; i < array2d.length; i++) {
            if (len != Array.getLength(array2d[i])) {
                return false;
            }
        }
        return true;
    }

    /**
     * 引数 e が配列 a に含まれているか評価します。
     * 
     * @param a
     *            評価基の配列
     * @param e
     *            検索する値
     * @return 引数 e が配列 a に含まれている場合のみ true
     * @throws NullPointerException
     *             引数の配列が null の場合
     */
    public static boolean isContains(boolean[] a, boolean e) {
        for (int i = 0; i < a.length; i++) {
            if (e == a[i]) {
                return true;
            }
        }
        return false;
    }

    /**
     * 引数 e が配列 a に含まれているか評価します。
     * 
     * @param a
     *            評価基の配列
     * @param e
     *            検索する値
     * @return 引数 e が配列 a に含まれている場合のみ true
     * @throws NullPointerException
     *             引数の配列が null の場合
     */
    public static boolean isContains(char[] a, char e) {
        for (int i = 0; i < a.length; i++) {
            if (e == a[i]) {
                return true;
            }
        }
        return false;
    }

    /**
     * 引数 e が配列 a に含まれているか評価します。
     * 
     * @param a
     *            評価基の配列
     * @param e
     *            検索する値
     * @return 引数 e が配列 a に含まれている場合のみ true
     * @throws NullPointerException
     *             引数の配列が null の場合
     */
    public static boolean isContains(byte[] a, byte e) {
        for (int i = 0; i < a.length; i++) {
            if (e == a[i]) {
                return true;
            }
        }
        return false;
    }

    /**
     * 引数 e が配列 a に含まれているか評価します。
     * 
     * @param a
     *            評価基の配列
     * @param e
     *            検索する値
     * @return 引数 e が配列 a に含まれている場合のみ true
     * @throws NullPointerException
     *             引数の配列が null の場合
     */
    public static boolean isContains(short[] a, short e) {
        for (int i = 0; i < a.length; i++) {
            if (e == a[i]) {
                return true;
            }
        }
        return false;
    }

    /**
     * 引数 e が配列 a に含まれているか評価します。
     * 
     * @param a
     *            評価基の配列
     * @param e
     *            検索する値
     * @return 引数 e が配列 a に含まれている場合のみ true
     * @throws NullPointerException
     *             引数の配列が null の場合
     */
    public static boolean isContains(int[] a, int e) {
        for (int i = 0; i < a.length; i++) {
            if (e == a[i]) {
                return true;
            }
        }
        return false;
    }

    /**
     * 引数 e が配列 a に含まれているか評価します。
     * 
     * @param a
     *            評価基の配列
     * @param e
     *            検索する値
     * @return 引数 e が配列 a に含まれている場合のみ true
     * @throws NullPointerException
     *             引数の配列が null の場合
     */
    public static boolean isContains(long[] a, long e) {
        for (int i = 0; i < a.length; i++) {
            if (e == a[i]) {
                return true;
            }
        }
        return false;
    }

    /**
     * 引数 e が配列 a に含まれているか評価します。
     * 
     * @param a
     *            評価基の配列
     * @param e
     *            検索する値
     * @return 引数 e が配列 a に含まれている場合のみ true
     * @throws NullPointerException
     *             引数の配列が null の場合
     */
    public static boolean isContains(float[] a, float e) {
        for (int i = 0; i < a.length; i++) {
            if (e == a[i]) {
                return true;
            }
        }
        return false;
    }

    /**
     * 引数 e が配列 a に含まれているか評価します。
     * 
     * @param a
     *            評価基の配列
     * @param e
     *            検索する値
     * @return 引数 e が配列 a に含まれている場合のみ true
     * @throws NullPointerException
     *             引数の配列が null の場合
     */
    public static boolean isContains(double[] a, double e) {
        for (int i = 0; i < a.length; i++) {
            if (e == a[i]) {
                return true;
            }
        }
        return false;
    }

    /**
     * 引数 o が配列 a に含まれているか評価します。
     * 
     * @param a
     *            評価基の配列
     * @param o
     *            検索するオブジェクト
     * @return 引数 o が配列 a に含まれている場合のみ true
     * @throws NullPointerException
     *             引数の配列が null の場合
     */
    public static boolean isContains(Object[] a, Object o) {
        for (int i = 0; i < a.length; i++) {
            if (isEquals(o, a[i])) {
                return true;
            }
        }
        return false;
    }

    /**
     * 引数 o がコレクション c に含まれているか評価します。
     * 
     * @param coll
     *            評価基のコレクション
     * @param o
     *            検索するオブジェクト
     * @return 引数 o がコレクション c に含まれている場合のみ true
     */
    public static boolean isContains(Collection coll, Object o) {
        if (coll == null) {
            return false;
        }
        return coll.contains(o);
    }

    /**
     * 指定の文字列に指定の文字が含まれているか評価します。
     * 
     * @param source
     *            評価する文字列
     * @param c
     *            検索する文字
     * @return 文字列に指定の文字が含まれている場合のみ true
     */
    public static boolean isContains(String source, char c) {
        if (source == null) {
            return false;
        }
        return (0 <= source.indexOf(c));
    }

    /**
     * 指定の文字列に指定の文字列が含まれているか評価します。
     * 
     * @param source
     *            評価する文字列
     * @param target
     *            検索する文字列
     * @return 文字列に指定の文字列が含まれている場合のみ true
     */
    public static boolean isContains(String source, String target) {
        if (source == null) {
            return false;
        }
        return (0 <= source.indexOf(target));
    }

    /**
     * 指定の文字シーケンスに指定の文字が含まれているか評価します。
     * 
     * @param cs
     *            評価する文字シーケンス
     * @param c
     *            検索する文字
     * @return 文字シーケンスに指定の文字が含まれている場合のみ true
     */
    public static boolean isContains(CharSequence cs, char c) {
        if (cs == null) {
            return false;
        }
        return (0 <= HSeek.indexOf(cs, c, 1));
    }

    /**
     * 指定の文字シーケンスに指定の文字シーケンスが含まれているか評価します。
     * 
     * @param cs
     *            評価する文字シーケンス
     * @param target
     *            検索する文字シーケンス
     * @return 文字シーケンスに指定の文字シーケンスが含まれている場合のみ true
     */
    public static boolean isContains(CharSequence cs, CharSequence target) {
        if (cs == null) {
            return false;
        }
        return (0 <= HSeek.indexOf(cs, target, 1));
    }

    /*
     * Index
     */

    /**
     * 指定された文字シーケンスの範囲内のインデックスか評価します。
     * 
     * @param cs
     *            評価する文字シーケンス
     * @param index
     *            インデックス
     * @return 範囲内のインデックスの場合のみ true
     */
    public static boolean isIndex(CharSequence cs, int index) {
        return isIndex(cs.length(), index);
    }

    /**
     * 指定された文字シーケンスの範囲内のインデックスか評価します。
     * 
     * @param cs
     *            評価する文字シーケンス
     * @param min
     *            開始インデックス
     * @param max
     *            終了インデックス
     * @return 範囲内のインデックスの場合のみ true
     */
    public static boolean isIndex(CharSequence cs, int min, int max) {
        return isIndex(cs.length(), min, max);
    }

    /**
     * 指定された配列の範囲内のインデックスか評価します。
     * 
     * @param a
     *            評価する配列
     * @param index
     *            インデックス
     * @return 範囲内のインデックスの場合のみ true
     */
    public static boolean isIndex(char[] a, int index) {
        return isIndex(a.length, index);
    }

    /**
     * 指定された配列の範囲内のインデックスか評価します。
     * 
     * @param a
     *            評価する配列
     * @param min
     *            開始インデックス
     * @param max
     *            終了インデックス
     * @return 範囲内のインデックスの場合のみ true
     */
    public static boolean isIndex(char[] a, int min, int max) {
        return isIndex(a.length, min, max);
    }

    /**
     * 指定された配列の範囲内のインデックスか評価します。
     * 
     * @param a
     *            評価する配列
     * @param index
     *            インデックス
     * @return 範囲内のインデックスの場合のみ true
     */
    public static boolean isIndex(boolean[] a, int index) {
        return isIndex(a.length, index);
    }

    /**
     * 指定された配列の範囲内のインデックスか評価します。
     * 
     * @param a
     *            評価する配列
     * @param min
     *            開始インデックス
     * @param max
     *            終了インデックス
     * @return 範囲内のインデックスの場合のみ true
     */
    public static boolean isIndex(boolean[] a, int min, int max) {
        return isIndex(a.length, min, max);
    }

    /**
     * 指定された配列の範囲内のインデックスか評価します。
     * 
     * @param a
     *            評価する配列
     * @param index
     *            インデックス
     * @return 範囲内のインデックスの場合のみ true
     */
    public static boolean isIndex(byte[] a, int index) {
        return isIndex(a.length, index);
    }

    /**
     * 指定された配列の範囲内のインデックスか評価します。
     * 
     * @param a
     *            評価する配列
     * @param min
     *            開始インデックス
     * @param max
     *            終了インデックス
     * @return 範囲内のインデックスの場合のみ true
     */
    public static boolean isIndex(byte[] a, int min, int max) {
        return isIndex(a.length, min, max);
    }

    /**
     * 指定された配列の範囲内のインデックスか評価します。
     * 
     * @param a
     *            評価する配列
     * @param index
     *            インデックス
     * @return 範囲内のインデックスの場合のみ true
     */
    public static boolean isIndex(short[] a, int index) {
        return isIndex(a.length, index);
    }

    /**
     * 指定された配列の範囲内のインデックスか評価します。
     * 
     * @param a
     *            評価する配列
     * @param min
     *            開始インデックス
     * @param max
     *            終了インデックス
     * @return 範囲内のインデックスの場合のみ true
     */
    public static boolean isIndex(short[] a, int min, int max) {
        return isIndex(a.length, min, max);
    }

    /**
     * 指定された配列の範囲内のインデックスか評価します。
     * 
     * @param a
     *            評価する配列
     * @param index
     *            インデックス
     * @return 範囲内のインデックスの場合のみ true
     */
    public static boolean isIndex(int[] a, int index) {
        return isIndex(a.length, index);
    }

    /**
     * 指定された配列の範囲内のインデックスか評価します。
     * 
     * @param a
     *            評価する配列
     * @param min
     *            開始インデックス
     * @param max
     *            終了インデックス
     * @return 範囲内のインデックスの場合のみ true
     */
    public static boolean isIndex(int[] a, int min, int max) {
        return isIndex(a.length, min, max);
    }

    /**
     * 指定された配列の範囲内のインデックスか評価します。
     * 
     * @param a
     *            評価する配列
     * @param index
     *            インデックス
     * @return 範囲内のインデックスの場合のみ true
     */
    public static boolean isIndex(long[] a, int index) {
        return isIndex(a.length, index);
    }

    /**
     * 指定された配列の範囲内のインデックスか評価します。
     * 
     * @param a
     *            評価する配列
     * @param min
     *            開始インデックス
     * @param max
     *            終了インデックス
     * @return 範囲内のインデックスの場合のみ true
     */
    public static boolean isIndex(long[] a, int min, int max) {
        return isIndex(a.length, min, max);
    }

    /**
     * 指定された配列の範囲内のインデックスか評価します。
     * 
     * @param a
     *            評価する配列
     * @param index
     *            インデックス
     * @return 範囲内のインデックスの場合のみ true
     */
    public static boolean isIndex(float[] a, int index) {
        return isIndex(a.length, index);
    }

    /**
     * 指定された配列の範囲内のインデックスか評価します。
     * 
     * @param a
     *            評価する配列
     * @param min
     *            開始インデックス
     * @param max
     *            終了インデックス
     * @return 範囲内のインデックスの場合のみ true
     */
    public static boolean isIndex(float[] a, int min, int max) {
        return isIndex(a.length, min, max);
    }

    /**
     * 指定された配列の範囲内のインデックスか評価します。
     * 
     * @param a
     *            評価する配列
     * @param index
     *            インデックス
     * @return 範囲内のインデックスの場合のみ true
     */
    public static boolean isIndex(double[] a, int index) {
        return isIndex(a.length, index);
    }

    /**
     * 指定された配列の範囲内のインデックスか評価します。
     * 
     * @param a
     *            評価する配列
     * @param min
     *            開始インデックス
     * @param max
     *            終了インデックス
     * @return 範囲内のインデックスの場合のみ true
     */
    public static boolean isIndex(double[] a, int min, int max) {
        return isIndex(a.length, min, max);
    }

    /**
     * 指定された配列の範囲内のインデックスか評価します。
     * 
     * @param a
     *            評価する配列
     * @param index
     *            インデックス
     * @return 範囲内のインデックスの場合のみ true
     */
    public static boolean isIndex(Object[] a, int index) {
        return isIndex(a.length, index);
    }

    /**
     * 指定された配列の範囲内のインデックスか評価します。
     * 
     * @param a
     *            評価する配列
     * @param min
     *            開始インデックス
     * @param max
     *            終了インデックス
     * @return 範囲内のインデックスの場合のみ true
     */
    public static boolean isIndex(Object[] a, int min, int max) {
        return isIndex(a.length, min, max);
    }

    /**
     * 指定されたサイズの範囲内のインデックスか評価します。
     * 
     * @param size
     *            範囲を持つオブジェクト
     * @param index
     *            インデックス
     * @return 範囲内のインデックスの場合のみ true
     */
    public static boolean isIndex(int size, int index) {
        return (0 <= index && index < size);
    }

    /**
     * 指定されたサイズの範囲内のインデックスか評価します。
     * 
     * @param size
     *            範囲を持つオブジェクト
     * @param min
     *            開始インデックス
     * @param max
     *            終了インデックス
     * @return 範囲内のインデックスの場合のみ true
     * @throws IllegalArgumentException
     *             (min > max) の場合
     */
    public static boolean isIndex(int size, int min, int max) {
        if (min > max) {
            throw new IllegalArgumentException("min > max (" + min + " > " + max + ")");
        }
        return (0 <= min && min < size) && (0 <= max && max < size);
    }

    /*
     * Extends
     */

    /**
     * 引数が全て null 値であるかを評価します。
     * 
     * @param o
     *            評価する一番目の値
     * @param o2
     *            評価する二番目の値
     * @return 引数が全て null 値の場合のみ true
     */
    public static boolean isAndNull(Object o, Object o2) {
        return (o == null && o2 == null);
    }

    /**
     * 全ての配列の要素が null 値であるかを評価します。
     * 
     * @param os
     *            評価する配列
     * @return 全ての配列の要素が null 値の場合のみ true
     * @throws NullPointerException
     *             引数の配列が null の場合
     */
    public static boolean isAndNull(Object[] os) {
        for (int i = 0; i < os.length; i++) {
            if (os[i] != null) {
                return false;
            }
        }
        return true;
    }

    /**
     * 引数に null 値が含まれるか評価します。
     * 
     * @param o
     *            評価する一番目の値
     * @param o2
     *            評価する二番目の値
     * @return 引数に null 値が含まれる場合のみ true
     */
    public static boolean isOrNull(Object o, Object o2) {
        return (o == null || o2 == null);
    }

    /**
     * 引数に null 値が含まれるか評価します。
     * 
     * @param o
     *            評価する一番目の値
     * @param o2
     *            評価する二番目の値
     * @param o3
     *            評価する三番目の値
     * @return 引数に null 値が含まれる場合のみ true
     */
    public static boolean isOrNull(Object o, Object o2, Object o3) {
        return (o == null || o2 == null || o3 == null);
    }

    /**
     * 引数に null 値が含まれるか評価します。
     * 
     * @param o
     *            評価する一番目の値
     * @param o2
     *            評価する二番目の値
     * @param o3
     *            評価する三番目の値
     * @param o4
     *            評価する四番目の値
     * @return 引数に null 値が含まれる場合のみ true
     */
    public static boolean isOrNull(Object o, Object o2, Object o3, Object o4) {
        return (o == null || o2 == null || o3 == null || o4 == null);
    }

    /**
     * 配列の要素に null 値が含まれるか評価します。
     * 
     * @param os
     *            評価する配列
     * @return 配列の要素に null 値が含まれる場合のみ true
     * @throws NullPointerException
     *             引数の配列が null の場合
     */
    public static boolean isOrNull(Object[] os) {
        for (int i = 0; i < os.length; i++) {
            if (os[i] == null) {
                return true;
            }
        }
        return false;
    }

    /**
     * 引数が全て null または空かスペースのみの文字シーケンスであるかを評価します。
     * 
     * @param cs
     *            評価する一番目の文字シーケンス
     * @param cs2
     *            評価する二番目の文字シーケンス
     * @return 引数が全て null または空かスペースのみの文字シーケンスの場合のみ true
     */
    public static boolean isAndBlank(CharSequence cs, CharSequence cs2) {
        return (isBlank(cs) && isBlank(cs2));
    }

    /**
     * 引数が全て null または空かスペースのみの文字シーケンスであるかを評価します。
     * 
     * @param cs
     *            評価する一番目の文字シーケンス
     * @param cs2
     *            評価する二番目の文字シーケンス
     * @param cs3
     *            評価する三番目の文字シーケンス
     * @return 引数が全て null または空かスペースのみの文字シーケンスの場合のみ true
     */
    public static boolean isAndBlank(CharSequence cs, CharSequence cs2, CharSequence cs3) {
        return (isBlank(cs) && isBlank(cs2) && isBlank(cs3));
    }

    /**
     * 全ての配列の要素が null または空かスペースのみの文字シーケンスであるかを評価します。
     * 
     * @param css
     *            評価する文字シーケンスの配列
     * @return 全ての配列の要素が null または空かスペースのみの文字シーケンスの場合のみ true
     * @throws NullPointerException
     *             引数の配列が null の場合
     */
    public static boolean isAndBlank(CharSequence[] css) {
        for (int i = 0; i < css.length; i++) {
            if (!isBlank(css[i])) {
                return false;
            }
        }
        return true;
    }

    /**
     * 引数に null または空かスペースのみの文字シーケンスが含まれるか評価します。
     * 
     * @param cs
     *            評価する一番目の文字シーケンス
     * @param cs2
     *            評価する二番目の文字シーケンス
     * @return 引数に null または空かスペースのみの文字シーケンスが含まれる場合のみ true
     */
    public static boolean isOrBlank(CharSequence cs, CharSequence cs2) {
        return (isBlank(cs) || isBlank(cs2));
    }

    /**
     * 全ての配列の要素が null または空かスペースのみの文字シーケンスが含まれるか評価します。
     * 
     * @param css
     *            評価する文字シーケンスの配列
     * @return 全ての配列の要素が null または空かスペースのみの文字シーケンスが含まれる場合のみ true
     * @throws NullPointerException
     *             引数の配列が null の場合
     */
    public static boolean isOrBlank(CharSequence[] css) {
        for (int i = 0; i < css.length; i++) {
            if (isBlank(css[i])) {
                return true;
            }
        }
        return false;
    }

    /**
     * 引数が全て null または空の文字シーケンスであるかを評価します。
     * 
     * @param cs
     *            評価する一番目の文字シーケンス
     * @param cs2
     *            評価する二番目の文字シーケンス
     * @return 引数が全て null または空の文字シーケンスの場合のみ true
     */
    public static boolean isAndEmpty(CharSequence cs, CharSequence cs2) {
        return (isEmpty(cs) && isEmpty(cs2));
    }

    /**
     * 全ての配列の要素が null または空の文字シーケンスであるかを評価します。
     * 
     * @param css
     *            評価する文字シーケンスの配列
     * @return 全ての配列の要素が null または空の文字シーケンスの場合のみ true
     * @throws NullPointerException
     *             引数の配列が null の場合
     */
    public static boolean isAndEmpty(CharSequence[] css) {
        for (int i = 0; i < css.length; i++) {
            if (!isEmpty(css[i])) {
                return false;
            }
        }
        return true;
    }

    /**
     * 引数に null または空の文字シーケンスが含まれるか評価します。
     * 
     * @param cs
     *            評価する一番目の文字シーケンス
     * @param cs2
     *            評価する二番目の文字シーケンス
     * @return 引数に null または空の文字シーケンスが含まれる場合のみ true
     */
    public static boolean isOrEmpty(CharSequence cs, CharSequence cs2) {
        return (isEmpty(cs) || isEmpty(cs2));
    }

    /**
     * 全ての配列の要素が null または空の文字シーケンスが含まれるか評価します。
     * 
     * @param css
     *            評価する文字シーケンスの配列
     * @return 全ての配列の要素が null または空の文字シーケンスが含まれる場合のみ true
     * @throws NullPointerException
     *             引数の配列が null の場合
     */
    public static boolean isOrEmpty(CharSequence[] css) {
        for (int i = 0; i < css.length; i++) {
            if (isEmpty(css[i])) {
                return true;
            }
        }
        return false;
    }

    /**
     * 指定の値の内いずれかが同値であるかを評価します。 <br>
     * 同値か双方 null の場合は true と評価します。
     * 
     * @param o
     *            評価基の値
     * @param o2
     *            評価先の値
     * @param o3
     *            評価先の値
     * @return 指定の値の内いずれかが同値である場合は true
     */
    public static boolean isOrEquals(Object o, Object o2, Object o3) {
        return (isEquals(o, o2) || isEquals(o, o3));
    }

    /**
     * 指定の値の内いずれかが同値であるかを評価します。 <br>
     * 同値か双方 null の場合は true と評価します。
     * 
     * @param o
     *            評価基の値
     * @param o2
     *            評価先の値
     * @param o3
     *            評価先の値
     * @param o4
     *            評価先の値
     * @return 指定の値の内いずれかが同値である場合は true
     */
    public static boolean isOrEquals(Object o, Object o2, Object o3, Object o4) {
        return (isEquals(o, o2) || isEquals(o, o3) || isEquals(o, o4));
    }

    /**
     * 指定の文字シーケンスの位置から前方一致するか評価します。
     * 
     * @param cs
     *            評価基の文字シーケンス
     * @param prefixs
     *            評価先の接頭辞の配列
     * @param offset
     *            評価を開始する相対インデックス
     * @return 指定の文字シーケンスの位置から前方一致する場合のみ true
     */
    public static boolean isOrStartsWith(CharSequence cs, CharSequence[] prefixs, int offset) {
        if (isEmpty(cs)) {
            return false;
        }
        for (int i = 0; i < prefixs.length; i++) {
            if (startsWith(cs, prefixs[i], offset)) {
                return true;
            }
        }
        return false;
    }

    /**
     * 指定の文字シーケンスに接尾辞の内いずれかが含まれるか評価します。
     * 
     * @param cs
     *            評価基の文字シーケンス
     * @param suffixs
     *            評価先の接尾辞の配列
     * @return 指定の文字列に接尾辞の内いずれかが含まれる場合のみ true
     */
    public static boolean isOrSuffix(CharSequence cs, char[] suffixs) {
        if (isEmpty(cs)) {
            return false;
        }
        char suffix = cs.charAt(cs.length() - 1);
        for (int i = 0; i < suffixs.length; i++) {
            if (suffix == suffixs[i]) {
                return true;
            }
        }
        return false;
    }

    /**
     * 指定の文字シーケンスに接尾辞の内いずれかが含まれるか評価します。
     * 
     * @param cs
     *            評価基の文字シーケンス
     * @param suffixs
     *            評価先の接尾辞の配列
     * @return 指定の文字シーケンスに接尾辞の内いずれかが含まれる場合のみ true
     */
    public static boolean isOrSuffix(CharSequence cs, CharSequence[] suffixs) {
        if (isEmpty(cs)) {
            return false;
        }
        for (int i = 0; i < suffixs.length; i++) {
            if (suffix(cs, suffixs[i])) {
                return true;
            }
        }
        return false;
    }

    /**
     * 指定の文字シーケンスに接頭辞の内いずれかが含まれるか評価します。
     * 
     * @param cs
     *            評価基の文字シーケンス
     * @param prefixs
     *            評価先の接頭辞の配列
     * @return 指定の文字列に接頭辞の内いずれかが含まれる場合のみ true
     */
    public static boolean isOrPrefix(CharSequence cs, char[] prefixs) {
        if (isEmpty(cs)) {
            return false;
        }
        char prefix = cs.charAt(0);
        for (int i = 0; i < prefixs.length; i++) {
            if (prefix == prefixs[i]) {
                return true;
            }
        }
        return false;
    }

    /**
     * 指定の文字シーケンスに接頭辞の内いずれかが含まれるか評価します。
     * 
     * @param cs
     *            評価基の文字シーケンス
     * @param prefixs
     *            評価先の接頭辞の配列
     * @return 指定の文字シーケンスに接頭辞の内いずれかが含まれる場合のみ true
     */
    public static boolean isOrPrefix(CharSequence cs, CharSequence[] prefixs) {
        if (isEmpty(cs)) {
            return false;
        }
        for (int i = 0; i < prefixs.length; i++) {
            if (prefix(cs, prefixs[i])) {
                return true;
            }
        }
        return false;
    }

    /**
     * 指定されたオブジェクトが全てのクラス型に割り当て可能か評価します。
     * 
     * @param o
     *            評価するオブジェクト
     * @param clazz
     *            割り当て可能か評価先するクラス
     * @param clazz2
     *            割り当て可能か評価先するクラス
     * @return オブジェクトが全てのクラス型に割り当て可能の場合のみ true
     */
    public static boolean isAndInstanceOf(Object o, Class clazz, Class clazz2) {
        return (isInstanceOf(o, clazz) && isInstanceOf(o, clazz2));
    }

    /**
     * 指定されたオブジェクトが全てのクラス型に割り当て可能か評価します。
     * 
     * @param o
     *            評価するオブジェクト
     * @param clazzes
     *            割り当て可能か評価先するクラスの配列
     * @return オブジェクトが全てのクラス型に割り当て可能の場合のみ true
     */
    public static boolean isAndInstanceOf(Object o, Class[] clazzes) {
        for (int i = 0; i < clazzes.length; i++) {
            Class c = clazzes[i];
            if (!isInstanceOf(o, c)) {
                return false;
            }
        }
        return true;
    }

    /**
     * 指定されたオブジェクトがいずれかのクラス型に割り当て可能か評価します。
     * 
     * @param o
     *            評価するオブジェクト
     * @param clazz
     *            割り当て可能か評価先するクラス
     * @param clazz2
     *            割り当て可能か評価先するクラス
     * @return オブジェクトがいずれかのクラス型に割り当て可能の場合のみ true
     */
    public static boolean isOrInstanceOf(Object o, Class clazz, Class clazz2) {
        return (isInstanceOf(o, clazz) || isInstanceOf(o, clazz2));
    }

    /**
     * 指定されたオブジェクトがいずれかのクラス型に割り当て可能か評価します。
     * 
     * @param o
     *            評価するオブジェクト
     * @param clazz
     *            割り当て可能か評価先するクラス
     * @param clazz2
     *            割り当て可能か評価先するクラス
     * @param clazz3
     *            割り当て可能か評価先するクラス
     * @return オブジェクトがいずれかのクラス型に割り当て可能の場合のみ true
     */
    public static boolean isOrInstanceOf(Object o, Class clazz, Class clazz2, Class clazz3) {
        return (isInstanceOf(o, clazz) || isInstanceOf(o, clazz2) || isInstanceOf(o, clazz3));
    }

    /**
     * 指定されたオブジェクトがいずれかのクラス型に割り当て可能か評価します。
     * 
     * @param o
     *            評価するオブジェクト
     * @param clazzes
     *            割り当て可能か評価先するクラスの配列
     * @return オブジェクトがいずれかのクラス型に割り当て可能の場合のみ true
     */
    public static boolean isOrInstanceOf(Object o, Class[] clazzes) {
        for (int i = 0; i < clazzes.length; i++) {
            Class c = clazzes[i];
            if (isInstanceOf(o, c)) {
                return true;
            }
        }
        return false;
    }

    /**
     * 指定された文字シーケンスのインデックスの文字と指定の文字が一致するか評価します。
     * 
     * @param cs
     *            評価する文字シーケンス
     * @param index
     *            評価するインデックス
     * @param c
     *            同値性を比較する文字
     * @return 文字シーケンスのインデックスの文字と指定の文字が一致する場合のみ true
     */
    public static boolean isCharAt(CharSequence cs, int index, char c) {
        return ((0 <= index && index < cs.length()) && cs.charAt(index) == c);
    }

}
