/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.core.collections.decorator;

import java.io.Serializable;
import java.util.AbstractSet;
import java.util.Collection;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

/**
 * 指定されたマップエントリのセットを装飾するビューを提供します。 <br>
 * <br>
 * このクラスはマップエントリのセットの実装を隠蔽する目的や、装飾クラスの基底実装の役割を持ちます。<br>
 * 装飾されたセットが直列化可能の場合は直列化可能です。
 */
public class DecoratedMapEntrySet extends AbstractSet implements Serializable {

    /* serialVersionUID */
    private static final long serialVersionUID = 2047786192836234539L;

    /** 基となるセット。 */
    protected final Set set;

    /**
     * ラップするマップエントリのセットを指定して初期化します。
     * 
     * @param entrySet
     *            マップエントリのセット
     */
    public DecoratedMapEntrySet(Set entrySet) {
        if (entrySet == null) {
            throw new NullPointerException();
        }
        this.set = entrySet;
    }

    /**
     * 基となるコレクションの同メソッドを呼び出します。
     * 
     * @see java.util.Set#size()
     */
    public int size() {
        return set.size();
    }

    /**
     * 基となるコレクションの同メソッドを呼び出します。
     * 
     * @see java.util.Set#isEmpty()
     */
    public boolean isEmpty() {
        return set.isEmpty();
    }

    /**
     * 基となるコレクションの同メソッドを呼び出します。
     * 
     * @see java.util.Set#clear()
     */
    public void clear() {
        set.clear();
    }

    /**
     * decoratedEntry(Map.Entry) で装飾した要素を返す反復子を返却します。
     * 
     * @see java.util.AbstractCollection#iterator()
     */
    public Iterator iterator() {
        return new DecoratedIterator(set.iterator()) {
            public Object next() {
                return decorateEntry((Map.Entry) i.next());
            }
        };
    }

    /**
     * decoratedEntry(Map.Entry) で装飾して削除します。
     * 
     * @see java.util.AbstractCollection#remove(java.lang.Object)
     */
    public boolean remove(Object o) {
        return (o instanceof Map.Entry) ? set.remove(decorateEntry((Map.Entry) o)) : false;
    }

    /**
     * decoratedEntry(Map.Entry) で装飾した要素を返す反復子を返却します。
     * 
     * @see java.util.AbstractCollection#contains(java.lang.Object)
     */
    public boolean contains(Object o) {
        return (o instanceof Map.Entry) ? set.contains(decorateEntry((Map.Entry) o)) : false;
    }

    /**
     * UnsupportedOperationException を発生させます。
     * 
     * @see java.util.Set#add(java.lang.Object)
     */
    public boolean add(Object o) {
        throw new UnsupportedOperationException();
    }

    /**
     * UnsupportedOperationException を発生させます。
     * 
     * @see java.util.Set#addAll(java.util.Collection)
     */
    public boolean addAll(Collection c) {
        throw new UnsupportedOperationException();
    }

    /**
     * Map.Entry を装飾して返却します。
     * 
     * @param entry
     *            マップエントリ
     * @return 装飾されたマップエントリ
     */
    protected Map.Entry decorateEntry(Map.Entry entry) {
        return new DecoratedMapEntry(entry);
    }

}