/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.core.collections.decorator;

import java.io.Serializable;
import java.util.Collection;
import java.util.Map;
import java.util.Set;

/**
 * 指定されたマップを装飾するビューを提供します。 <br>
 * <br>
 * このクラスはマップの実装を隠蔽する目的や、装飾クラスの基底実装の役割を持ちます。<br>
 * 装飾されたマップが直列化可能の場合は直列化可能です。
 */
public class DecoratedMap implements Map, Serializable {

    /* serialVersionUID */
    private static final long serialVersionUID = 1105890205936418760L;

    /** 基となるマップ。 */
    protected final Map map;

    /**
     * ラップするマップを指定して初期化します。
     * 
     * @param m
     *            ラップするマップ
     */
    public DecoratedMap(final Map m) {
        if (m == null) {
            throw new NullPointerException();
        }
        this.map = m;
    }

    /**
     * 基となるマップの同メソッドを呼び出します。
     * 
     * @see java.util.Map#size()
     */
    public int size() {
        return map.size();
    }

    /**
     * 基となるマップの同メソッドを呼び出します。
     * 
     * @see java.util.Map#clear()
     */
    public void clear() {
        map.clear();
    }

    /**
     * 基となるマップの同メソッドを呼び出します。
     * 
     * @see java.util.Map#isEmpty()
     */
    public boolean isEmpty() {
        return map.isEmpty();
    }

    /**
     * 基となるマップの同メソッドを呼び出します。
     * 
     * @see java.util.Map#containsKey(java.lang.Object)
     */
    public boolean containsKey(Object key) {
        return map.containsKey(key);
    }

    /**
     * 基となるマップの同メソッドを呼び出します。
     * 
     * @see java.util.Map#containsValue(java.lang.Object)
     */
    public boolean containsValue(Object value) {
        return map.containsValue(value);
    }

    /**
     * 基となるマップの同メソッドを呼び出します。
     * 
     * @see java.util.Map#putAll(java.util.Map)
     */
    public void putAll(Map m) {
        map.putAll(m);
    }

    /**
     * 基となるマップの同メソッドを呼び出します。
     * 
     * @see java.util.Map#get(java.lang.Object)
     */
    public Object get(Object key) {
        return map.get(key);
    }

    /**
     * 基となるマップの同メソッドを呼び出します。
     * 
     * @see java.util.Map#remove(java.lang.Object)
     */
    public Object remove(Object key) {
        return map.remove(key);
    }

    /**
     * 基となるマップの同メソッドを呼び出します。
     * 
     * @see java.util.Map#put(java.lang.Object, java.lang.Object)
     */
    public Object put(Object key, Object value) {
        return map.put(key, value);
    }

    /**
     * 基となるマップの同メソッドを呼び出します。
     * 
     * @see java.lang.Object#equals(java.lang.Object)
     */
    public boolean equals(Object o) {
        return map.equals(o);
    }

    /**
     * 基となるマップの同メソッドを呼び出します。
     * 
     * @see java.lang.Object#hashCode()
     */
    public int hashCode() {
        return map.hashCode();
    }

    /**
     * 基となるマップの同メソッドを呼び出します。
     * 
     * @see java.lang.Object#toString()
     */
    public String toString() {
        return map.toString();
    }

    /*
     * view
     */

    /* 値コレクションビューのキャッシュです。 */
    protected transient Collection values = null;

    /* キーセットビューのキャッシュです。 */
    protected transient Set keySet = null;

    /* エントリセットビューのキャッシュです。 */
    protected transient Set entrySet = null;

    /**
     * 基となるマップの同メソッドの戻り値を DecoratedMap.DecoratedMapEntrySet で装飾したセットを返却します。
     * 
     * @see java.util.Map#entrySet()
     */
    public Set entrySet() {
        return ((entrySet != null) ? entrySet : (entrySet = new DecoratedMapEntrySet(map.entrySet())));
    }

    /**
     * 基となるマップの同メソッドの戻り値を DecoratedSet で装飾したセットを返却します。
     * 
     * @see java.util.Map#keySet()
     */
    public Set keySet() {
        return ((keySet != null) ? keySet : (keySet = new DecoratedSet(map.keySet())));
    }

    /**
     * 基となるマップの同メソッドの戻り値を DecoratedCollection で装飾したコレクションを返却します。
     * 
     * @see java.util.Map#values()
     */
    public Collection values() {
        return ((values != null) ? values : (values = new DecoratedCollection(map.values())));
    }

}