/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.core.collections.decorator;

import java.io.Serializable;
import java.util.Collection;
import java.util.Iterator;

/**
 * 指定されたコレクションを装飾するビューを提供します。 <br>
 * <br>
 * このクラスはコレクションの実装を隠蔽する目的や、装飾クラスの基底実装の役割を持ちます。<br>
 * <br>
 * コレクションのオペレーションの規約を守るため hashCode オペレーションおよび equals オペレーションを基となるコレクションに渡すことはなく、 Object の equals メソッドおよび hashCode メソッドに依存します。<br>
 * 装飾されたコレクションが直列化可能の場合は直列化可能です。
 */
public class DecoratedCollection implements Collection, Serializable {

    /* serialVersionUID */
    private static final long serialVersionUID = 6319806398267776840L;

    /** 基となるコレクション。 */
    protected final Collection coll;

    /**
     * ラップするコレクションを指定して初期化します。
     * 
     * @param c
     *            ラップするコレクション
     */
    public DecoratedCollection(Collection c) {
        if (c == null) {
            throw new NullPointerException();
        }
        this.coll = c;
    }

    /**
     * 基となるコレクションの同メソッドを呼び出します。
     * 
     * @see java.util.Collection#size()
     */
    public int size() {
        return coll.size();
    }

    /**
     * 基となるコレクションの同メソッドを呼び出します。
     * 
     * @see java.util.Collection#isEmpty()
     */
    public boolean isEmpty() {
        return coll.isEmpty();
    }

    /**
     * 基となるコレクションの同メソッドを呼び出します。
     * 
     * @see java.util.Collection#contains(java.lang.Object)
     */
    public boolean contains(Object o) {
        return coll.contains(o);
    }

    /**
     * 基となるコレクションの同メソッドを呼び出します。
     * 
     * @see java.util.Collection#toArray()
     */
    public Object[] toArray() {
        return coll.toArray();
    }

    /**
     * 基となるコレクションの同メソッドを呼び出します。
     * 
     * @see java.util.Collection#toArray(java.lang.Object[])
     */
    public Object[] toArray(Object[] a) {
        return coll.toArray(a);
    }

    /**
     * 基となるコレクションの同メソッドの戻り値を DecoratedIterator で装飾した反復子を返却します。
     * 
     * @see java.util.Collection#iterator()
     */
    public Iterator iterator() {
        return new DecoratedIterator(coll.iterator());
    }

    /**
     * 基となるコレクションの同メソッドを呼び出します。
     * 
     * @see java.util.Collection#add(java.lang.Object)
     */
    public boolean add(Object o) {
        return coll.add(o);
    }

    /**
     * 基となるコレクションの同メソッドを呼び出します。
     * 
     * @see java.util.Collection#remove(java.lang.Object)
     */
    public boolean remove(Object o) {
        return coll.remove(o);
    }

    /**
     * 基となるコレクションの同メソッドを呼び出します。
     * 
     * @see java.util.Collection#containsAll(java.util.Collection)
     */
    public boolean containsAll(Collection c) {
        return coll.containsAll(c);
    }

    /**
     * 基となるコレクションの同メソッドを呼び出します。
     * 
     * @see java.util.Collection#addAll(java.util.Collection)
     */
    public boolean addAll(Collection c) {
        return coll.addAll(c);
    }

    /**
     * 基となるコレクションの同メソッドを呼び出します。
     * 
     * @see java.util.Collection#removeAll(java.util.Collection)
     */
    public boolean removeAll(Collection c) {
        return coll.removeAll(c);
    }

    /**
     * 基となるコレクションの同メソッドを呼び出します。
     * 
     * @see java.util.Collection#retainAll(java.util.Collection)
     */
    public boolean retainAll(Collection c) {
        return coll.retainAll(c);
    }

    /**
     * 基となるコレクションの同メソッドを呼び出します。
     * 
     * @see java.util.Collection#clear()
     */
    public void clear() {
        coll.clear();
    }

    /**
     * 基となるコレクションの同メソッドを呼び出します。
     * 
     * @see java.lang.Object#toString()
     */
    public String toString() {
        return coll.toString();
    }
}