/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.core.collections.cache;

import java.io.Serializable;
import java.util.LinkedHashMap;

import shohaku.core.collections.Cache;

/**
 * 挿入順にエントリを格納するハッシュテーブル構造を持つキャッシュを提供します。<br>
 * リサイズオペレーションでは挿入の古い順にエントリを削除します。<br>
 * 識別子をキャッシュに「再挿入」する場合、挿入順は影響を受けません。<br>
 * <br>
 * このキャッシュはデータの格納に java.util.LinkedHashMap を使用するため、その特性を継承しています。<br>
 * 特性の詳細は java.util.LinkedHashMap を参照してください。
 * 
 * @see java.util.LinkedHashMap
 */
public class InsertionOrderHashCache extends AbstractCache implements Cloneable, Serializable {

    /* serialVersionUID */
    private static final long serialVersionUID = 1734981299356517097L;

    /* デフォルトの最大キャッシュサイズ */
    static final int MAXIMUM_SIZE = 1 << 10;

    /* デフォルトの初期容量 */
    static final int INITIAL_CAPACITY = 32;

    /* デフォルトの負荷係数 */
    static final float LOAD_FACTOR = 0.75f;

    /**
     * デフォルトの初期容量と負荷係数、最大キャッシュサイズで初期化します。
     */
    public InsertionOrderHashCache() {
        this(INITIAL_CAPACITY);
    }

    /**
     * 指定された初期容量とデフォルトの負荷係数、最大キャッシュサイズで初期化します。
     * 
     * @param initialCapacity
     *            初期容量
     */
    public InsertionOrderHashCache(int initialCapacity) {
        this(initialCapacity, LOAD_FACTOR);
    }

    /**
     * 指定された初期容量と負荷係数とデフォルトの最大キャッシュサイズで初期化します。
     * 
     * @param initialCapacity
     *            初期容量
     * @param loadFactor
     *            負荷係数
     */
    public InsertionOrderHashCache(int initialCapacity, float loadFactor) {
        this(initialCapacity, loadFactor, MAXIMUM_SIZE);
    }

    /**
     * 指定されたキャッシュの全てのエントリを格納して初期化します。
     * 
     * @param t
     *            キャッシュ
     */
    public InsertionOrderHashCache(Cache t) {
        this(Math.max((int) (t.size() / LOAD_FACTOR) + 1, INITIAL_CAPACITY), LOAD_FACTOR, MAXIMUM_SIZE);
    }

    /**
     * 指定された初期容量と負荷係数、最大キャッシュサイズで初期化します。
     * 
     * @param initialCapacity
     *            初期容量
     * @param loadFactor
     *            負荷係数
     * @param maxSize
     *            最大キャッシュサイズ
     */
    public InsertionOrderHashCache(int initialCapacity, float loadFactor, int maxSize) {
        this(initialCapacity, loadFactor, maxSize, null);
    }

    /**
     * 指定された初期容量と負荷係数、最大キャッシュサイズで初期化します。
     * 
     * @param initialCapacity
     *            初期容量
     * @param loadFactor
     *            負荷係数
     * @param maxSize
     *            最大キャッシュサイズ
     * @param t
     *            キャッシュ
     */
    InsertionOrderHashCache(int initialCapacity, float loadFactor, int maxSize, Cache t) {
        // LinkedHashMap accessOrder=false
        super(new LinkedHashMap(initialCapacity, loadFactor, false), maxSize, t);
    }

    /*
     * Cloneable
     */

    /**
     * クローンを生成して返却します。
     * 
     * @return このオブジェクトのクローン
     * @see java.lang.Object#clone()
     */
    public Object clone() {
        synchronized (mutex) {
            InsertionOrderHashCache result = null;
            try {
                result = (InsertionOrderHashCache) super.clone();
            } catch (CloneNotSupportedException e) {
                // assert false;
            }
            return result;
        }
    }

}
