/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.core.collections;

import java.util.ListIterator;
import java.util.NoSuchElementException;

import shohaku.core.collections.decorator.DecoratedListIterator;
import shohaku.core.collections.decorator.UnmodifiableListIterator;

/**
 * ListIterator インタフェースのユーティリティメソッドを提供します。
 */
public class ListIteratorUtils {

    /* 空要素の反復子。 */
    static class EmptyListIterator implements ListIterator {

        public int nextIndex() {
            return -1;
        }

        public boolean hasNext() {
            return false;
        }

        public Object next() {
            throw new NoSuchElementException();
        }

        public int previousIndex() {
            return -1;
        }

        public boolean hasPrevious() {
            return false;
        }

        public Object previous() {
            throw new NoSuchElementException();
        }

        public void add(Object o) {
            throw new UnsupportedOperationException();
        }

        public void remove() {
            throw new UnsupportedOperationException();
        }

        public void set(Object o) {
            throw new UnsupportedOperationException();
        }

    }

    /*
     * Empty
     */

    /**
     * 空要素のリスト反復子を生成して返却します。
     * 
     * @return 空要素のリスト反復子
     */
    public static ListIterator emptyListIterator() {
        return new EmptyListIterator();
    }

    /*
     * Wrapper
     */

    /**
     * 更新不可のリスト反復子に装飾して返却します。
     * 
     * @param listiterator
     *            ラップするリスト反復子
     * @return 更新不可のリスト反復子
     */
    public static ListIterator unmodifiableListIterator(ListIterator listiterator) {
        return new UnmodifiableListIterator(listiterator);
    }

    /**
     * 実装型に固有の機能へのアクセスを制限した反復子を返却します。
     * 
     * @param listiterator
     *            ラップするリスト反復子
     * @return 実装を装飾したリスト反復子
     */
    public static ListIterator unextendsListIterator(ListIterator listiterator) {
        return new DecoratedListIterator(listiterator);
    }

}
