/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.core.collections;

import java.util.Collection;
import java.util.Comparator;
import java.util.Iterator;

/**
 * 順序付け比較のユーティリティメソッドを提供します。
 */
public class ComparatorUtils {

    /**
     * null 値の比較を追加機能とする、順序付け比較関数を返却します。<br>
     * 返却される比較関数は、compare(Comparable, Comparable, boolean) を呼び出します。
     * 
     * @param nullLess
     *            null を null 以外より小さいと評価する場合は true
     * @return null 値の比較を追加機能とする、順序付け比較関数
     */
    public static Comparator nullComparator(boolean nullLess) {
        final boolean comparableNullLess = nullLess;
        return new Comparator() {
            public int compare(Object o1, Object o2) {
                return ComparatorUtils.compare((Comparable) o1, (Comparable) o2, comparableNullLess);
            }
        };
    }

    /**
     * 順序付けのために null 値の比較機能を追加して、2 つの引数を比較します。 <br>
     * 双方の引数が null の場合は 0 を返却します。<br>
     * 引数の片方のみが null の場合は、引数 nullLess により結果が変わります。<br>
     * 引数 nullLess が false の場合は、null を常に大きいと見なします。<br>
     * 逆に nullLess が true の場合は、null を常に小さいと見なします。
     * 
     * @param comp
     *            評価基の値
     * @param comp2
     *            評価先の値
     * @param nullLess
     *            null を null 以外より小さいと評価する場合は true
     * @return 第１引数と第２引数が同値の場合は 0、第１引数が大きい場合は 1、第１引数が小さい場合は -1
     */
    public static int compare(Comparable comp, Comparable comp2, boolean nullLess) {
        if (comp == null && null == comp2) {
            return 0;
        }
        if (comp == null) {
            return (nullLess) ? -1 : 1;
        }
        if (comp2 == null) {
            return (nullLess) ? 1 : -1;
        }
        return comp.compareTo(comp2);
    }

    /**
     * 順序付けのために null 値の比較機能を追加して、2 つの引数を比較します。 <br>
     * 双方の引数が null の場合は 0 を返却します。<br>
     * 引数の片方のみが null の場合は、引数 nullLess により結果が変わります。<br>
     * 引数 nullLess が false の場合は、null を常に大きいと見なします。<br>
     * 逆に nullLess が true の場合は、null を常に小さいと見なします。
     * 
     * @param o
     *            評価基の値
     * @param o2
     *            評価先の値
     * @param comp
     *            順序付け比較関数
     * @param nullLess
     *            null を null 以外より小さいと評価する場合は true
     * @return 第１引数と第２引数が同値の場合は 0、第１引数が大きい場合は 1、第１引数が小さい場合は -1
     */
    public static int compare(Object o, Object o2, Comparator comp, boolean nullLess) {
        if (o == null && null == o2) {
            return 0;
        }
        if (o == null) {
            return (nullLess) ? -1 : 1;
        }
        if (o2 == null) {
            return (nullLess) ? 1 : -1;
        }
        return comp.compare(o, o2);
    }

    /**
     * 順序付けのために null 値の比較機能を追加して、反復子順に要素を比較します。 <br>
     * 要素は全て Comparable 型の必要が有ります。<br>
     * 引数の片方のみが null の場合は、引数 nullLess により結果が変わります。<br>
     * 引数 nullLess が false の場合は、null を常に大きいと見なします。<br>
     * 逆に nullLess が true の場合は、null を常に小さいと見なします。<br>
     * <br>
     * 以外の場合には、反復子の先頭から比較し、結果が 0 以外の場合にその結果を返却します。<br>
     * 要素数の小さい方の、末尾まで全ての比較が 0 の場合は、要素数で比較します。<br>
     * つまり要素数が同数の場合は 0、第１引数が大きい場合は 1、第１引数が小さい場合は -1、とします。<br>
     * また各要素は compare(Comparable, Comparable, boolean) の基準で比較します。
     * 
     * @param comps
     *            評価基の配列
     * @param comps2
     *            評価先の配列
     * @param nullLess
     *            null を null 以外より小さいと評価する場合は true
     * @return 第１引数と第２引数が同値の場合は 0、第１引数が大きい場合は 1、第１引数が小さい場合は -1
     */
    public static int compare(Collection comps, Collection comps2, boolean nullLess) {
        if (comps == null && null == comps2) {
            return 0;
        }
        if (comps == null) {
            return (nullLess) ? -1 : 1;
        }
        if (comps2 == null) {
            return (nullLess) ? 1 : -1;
        }
        if (comps.size() != comps2.size()) {
            return (comps.size() > comps2.size()) ? 1 : -1;
        }
        // compare all
        final Iterator i = comps.iterator();
        final Iterator i2 = comps2.iterator();
        while (i.hasNext() || i2.hasNext()) {
            if (!i.hasNext()) {
                return -1; // (size < size2)
            }
            if (!i2.hasNext()) {
                return 1; // (size > size2)
            }
            int r = compare((Comparable) i.next(), (Comparable) i2.next(), nullLess);
            if (0 != r) {
                return (r > 0) ? 1 : -1;
            }
        }
        // all zero
        return 0;
    }

    /**
     * 順序付けのために null 値の比較機能を追加して、反復子順に要素を比較します。 <br>
     * 引数の片方のみが null の場合は、引数 nullLess により結果が変わります。<br>
     * 引数 nullLess が false の場合は、null を常に大きいと見なします。<br>
     * 逆に nullLess が true の場合は、null を常に小さいと見なします。<br>
     * <br>
     * 以外の場合には、反復子の先頭から比較し、結果が 0 以外の場合にその結果を返却します。<br>
     * 要素数の小さい方の、末尾まで全ての比較が 0 の場合は、要素数で比較します。<br>
     * つまり要素数が同数の場合は 0、第１引数が大きい場合は 1、第１引数が小さい場合は -1、とします。<br>
     * また各要素は compare(Object, Object, Comparator, boolean) の基準で比較します。
     * 
     * @param coll
     *            評価基の配列
     * @param coll2
     *            評価先の配列
     * @param comp
     *            順序付け比較関数
     * @param nullLess
     *            null を null 以外より小さいと評価する場合は true
     * @return 第１引数と第２引数が同値の場合は 0、第１引数が大きい場合は 1、第１引数が小さい場合は -1
     */
    public static int compare(Collection coll, Collection coll2, Comparator comp, boolean nullLess) {
        if (coll == null && null == coll2) {
            return 0;
        }
        if (coll == null) {
            return (nullLess) ? -1 : 1;
        }
        if (coll2 == null) {
            return (nullLess) ? 1 : -1;
        }
        if (coll.size() != coll2.size()) {
            return (coll.size() > coll2.size()) ? 1 : -1;
        }
        // compare all
        final Iterator i = coll.iterator();
        final Iterator i2 = coll2.iterator();
        while (i.hasNext() || i2.hasNext()) {
            if (!i.hasNext()) {
                return -1; // (size < size2)
            }
            if (!i2.hasNext()) {
                return 1; // (size > size2)
            }
            int r = compare(i.next(), i2.next(), comp, nullLess);
            if (0 != r) {
                return (r > 0) ? 1 : -1;
            }
        }
        // all zero
        return 0;
    }

}
