/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.core.collections;

import java.lang.reflect.Array;
import java.util.Collection;
import java.util.Iterator;

import shohaku.core.collections.decorator.DecoratedCollection;
import shohaku.core.lang.Predicate;

/**
 * Collection インタフェースのユーティリティメソッドを提供します。
 */
public class CollectionsUtils {

    /**
     * 実装型に固有の機能へのアクセスを制限したコレクションを返却します。
     * 
     * @param coll
     *            ラップするコレクション
     * @return 実装を装飾したコレクション
     */
    public static Collection unextendsCollection(Collection coll) {
        return new DecoratedCollection(coll);
    }

    /**
     * コレクションを指定されたクラスの配列に変換します、プリミティブ型の生成を可能とします。
     * 
     * @param coll
     *            生成基のコレクション
     * @param type
     *            配列の基クラス
     * @return コレクションと同じ要素を持つ配列
     */
    public static Object toArray(Collection coll, Class type) {
        Object a = Array.newInstance(type, coll.size());
        int i = 0;
        Iterator iter = coll.iterator();
        while (iter.hasNext()) {
            Array.set(a, i++, iter.next());
        }
        return a;
    }

    /**
     * コレクションに反復子の要素を全て追加します。
     * 
     * @param coll
     *            追加先のコレクション
     * @param elements
     *            追加する反復子
     * @return 引数のコレクション参照
     */
    public static Collection addAll(Collection coll, Iterator elements) {
        while (elements.hasNext()) {
            coll.add(elements.next());
        }
        return coll;
    }

    /**
     * コレクションに反復子の要素を全て追加します。
     * 
     * @param coll
     *            追加先のコレクション
     * @param elements
     *            追加する反復子
     * @param predicate
     *            要素を追加するか評価するクロージャ
     * @return 引数のコレクション参照
     */
    public static Collection addAll(Collection coll, Iterator elements, Predicate predicate) {
        while (elements.hasNext()) {
            Object o = elements.next();
            if (predicate.evaluate(o)) {
                coll.add(o);
            }
        }
        return coll;
    }

    /**
     * クロージャが真と評価する要素をコレクションに追加します。
     * 
     * @param coll
     *            対象のコレクション
     * @param elements
     *            追加する値のコレクション
     * @param predicate
     *            要素を追加するか評価するクロージャ
     * @return コレクションの参照
     */
    public static Collection addAll(Collection coll, Collection elements, Predicate predicate) {
        return addAll(coll, elements.iterator(), predicate);
    }

    /**
     * クロージャが真と評価する要素のみを保持して返却します。
     * 
     * @param coll
     *            対象のコレクション
     * @param predicate
     *            要素を保存するか評価するクロージャ
     * @return コレクションの参照
     */
    public static Collection retainAll(Collection coll, Predicate predicate) {
        Iterator i = coll.iterator();
        while (i.hasNext()) {
            Object o = i.next();
            if (predicate.evaluate(o)) {
                i.remove();
            }
        }
        return coll;
    }

}
