/*
 * shohaku Copyright (C) 2005 tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package shohaku.core.beans.dynamic;

import java.lang.reflect.Method;

import shohaku.core.helpers.HBeans;
import shohaku.core.helpers.HLog;
import shohaku.core.helpers.HValid;
import shohaku.core.lang.IntrospectionBeansException;

/**
 * メソッドに関する情報を格納し、実行時に指定されたクラスを元に実行する機能を提供します。
 */
public final class MethodDesc extends FeatureDesc {

    /** メソッド種別：通常のメソッド。 */
    public static final int METHOD = 0;

    /** メソッド種別：セッタープロパティ。 */
    public static final int SET_PROPERTY = 1;

    /** メソッド種別：ゲッタープロパティ。 */
    public static final int GET_PROPERTY = 2;

    /* メソッド種別。 */
    private int type = METHOD;

    /* メソッド名。 */
    private String methodName;

    /* 引数の情報のリスト。 */
    private ArgumentsDesc arguments;

    /**
     * デフォルトコンストラクタ。
     */
    public MethodDesc() {
        // no op
    }

    /**
     * 引数のプロパティを格納して初期化します。
     * 
     * @param methodName
     *            メソッド名
     * @param params
     *            引数情報のリスト
     * @param type
     *            メソッド種別
     */
    public MethodDesc(String methodName, ArgumentsDesc params, int type) {
        setMethodName(methodName);
        setArguments(params);
        setType(type);
    }

    /**
     * 指定されたクラスからメソッドを呼び出し結果を返却します。
     * 
     * @param clazz
     *            処理対象のクラス
     * @return メソッドの戻り値
     * @throws IntrospectionBeansException
     *             処理の呼出に失敗した場合
     */
    public Object invoke(Class clazz) throws IntrospectionBeansException {
        return invoke(clazz, null);
    }

    /**
     * 指定されたインスタンスからメソッドを呼び出し結果を返却します。
     * 
     * @param obj
     *            処理対象のインスタンス
     * @return メソッドの戻り値
     * @throws IntrospectionBeansException
     *             処理の呼出に失敗した場合
     */
    public Object invoke(Object obj) throws IntrospectionBeansException {
        if (obj == null) {
            throw new NullPointerException("argument is null.");
        }
        return invoke(obj.getClass(), obj);
    }

    /* 指定されたインスタンスとクラスからメソッドまたはコンストラクタを呼び出し結果を返却します。 */
    Object invoke(Class clazz, Object obj) throws IntrospectionBeansException {
        if (clazz == null) {
            throw new NullPointerException("argument is null.");
        }
        return invokeMethod(clazz, obj);
    }

    /* 指定されたインスタンスとクラスからメソッドを呼び出し結果を返却します。 */
    Object invokeMethod(Class clazz, Object obj) throws IntrospectionBeansException {

        if (clazz == null) {
            throw new NullPointerException("argument class is null.");
        }

        Method method = findInvokeMethod(clazz);
        if (method == null) {
            throw new IntrospectionBeansException(HLog.log("no such method. ", clazz, getMethodName(), getArgumentTypes()));
        }

        return HBeans.invokeMethod(obj, method, getArgumentValues());

    }

    /* 指定されたクラスから処理の対象となるメソッドオブジェクトを返却します。 発見できない場合 null を返却します。 */
    Method findInvokeMethod(Class clazz) throws IntrospectionBeansException {
        Method method = null;
        switch (getType()) {
        case METHOD:
            method = HBeans.getAssignmentMethod(clazz, getMethodName(), getArgumentTypes());
            break;
        case SET_PROPERTY:
            method = HBeans.getAssignmentWriteProperty(clazz, getMethodName(), getArgumentTypes());
            break;
        case GET_PROPERTY:
            method = HBeans.getAssignmentWriteProperty(clazz, getMethodName(), getArgumentTypes());
            break;
        default:
            break;
        }
        return method;
    }

    /*
     * Property
     */

    /**
     * メソッド種別を返却します。
     * 
     * @return メソッド種別
     */
    public int getType() {
        return type;
    }

    /**
     * メソッド種別を格納します。
     * 
     * @param type
     *            メソッド種別
     */
    public void setType(int type) {
        if (!HValid.isRange(type, METHOD, GET_PROPERTY)) {
            throw new IllegalArgumentException("illegal mathod type. " + type);
        }
        this.type = type;
    }

    /**
     * メソッド名を返却します。
     * 
     * @return メソッド名
     */
    public String getMethodName() {
        return methodName;
    }

    /**
     * メソッド名を格納します。
     * 
     * @param name
     *            メソッド名
     */
    public void setMethodName(String name) {
        this.methodName = name;
    }

    /**
     * 引数情報のリストを返却します。
     * 
     * @return 引数情報のリスト
     */
    public ArgumentsDesc getArguments() {
        return arguments;
    }

    /**
     * 引数情報のリストを格納します。
     * 
     * @param params
     *            引数情報のリスト
     */
    public void setArguments(ArgumentsDesc params) {
        this.arguments = params;
    }

    /* 引数の値を返却します。 */
    Object[] getArgumentValues() {
        return (arguments == null) ? new Object[0] : arguments.getArgumentValues();
    }

    /* 引数の型情報を返却します。 */
    Class[] getArgumentTypes() {
        return (arguments == null) ? new Class[0] : arguments.getArgumentTypes();
    }

}
