/*
 * shohaku Copyright (C) 2005 tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package shohaku.core.beans.dynamic;

import shohaku.core.helpers.HBeans;
import shohaku.core.lang.IntrospectionBeansException;

/**
 * オブジェクトのフィールド情報を格納する機能を提供します。
 */
public class FieldDesc extends FeatureDesc {

    /* フィールドの名前。 */
    private String name;

    /**
     * デフォルトコンストラクタ。
     */
    public FieldDesc() {
        // no op
    }

    /**
     * 引数のプロパティを格納して初期化します。
     * 
     * @param name
     *            フィールドの名前
     */
    public FieldDesc(String name) {
        setName(name);
    }

    /**
     * 指定されたクラスからフィールドの値を返却します。
     * 
     * @param clazz
     *            処理対象のクラス
     * @return フィールドの値
     * @throws IntrospectionBeansException
     *             処理の呼出に失敗した場合
     */
    public Object get(Class clazz) throws IntrospectionBeansException {
        return getFieldValue(clazz, null);
    }

    /**
     * 指定されたインスタンスからフィールドの値を返却します。
     * 
     * @param obj
     *            処理対象のインスタンス
     * @return フィールドの値
     * @throws IntrospectionBeansException
     *             処理の呼出に失敗した場合
     */
    public Object get(Object obj) throws IntrospectionBeansException {
        if (obj == null) {
            throw new NullPointerException("argument is null.");
        }
        return getFieldValue(obj.getClass(), obj);
    }

    /**
     * 指定されたクラスのフィールドの値を変更し以前の値を返却します。
     * 
     * @param clazz
     *            処理対象のクラス
     * @param newValue
     *            格納するフィールドの値
     * @return 格納前のフィールドの値
     * @throws IntrospectionBeansException
     *             処理の呼出に失敗した場合
     */
    public Object set(Class clazz, Object newValue) throws IntrospectionBeansException {
        return setFieldValue(clazz, null, newValue);
    }

    /**
     * 指定されたインスタンスのフィールドの値を変更し以前の値を返却します。
     * 
     * @param obj
     *            処理対象のインスタンス
     * @param newValue
     *            格納するフィールドの値
     * @return 格納前のフィールドの値
     * @throws IntrospectionBeansException
     *             処理の呼出に失敗した場合
     */
    public Object set(Object obj, Object newValue) throws IntrospectionBeansException {
        if (obj == null) {
            throw new NullPointerException("argument is null.");
        }
        return setFieldValue(obj.getClass(), obj, newValue);
    }

    /* 指定されたクラスとそのインスタンスからフィールドの値を返却します。 */
    Object getFieldValue(Class clazz, Object obj) throws IntrospectionBeansException {
        return HBeans.getFieldValue(clazz, obj, getName());
    }

    /* 指定されたクラスとそのインスタンスのフィールドの値を変更し以前の値を返却します。 */
    Object setFieldValue(Class clazz, Object obj, Object newValue) throws IntrospectionBeansException {
        return HBeans.setFieldValue(clazz, obj, getName(), newValue);

    }

    /*
     * Property
     */

    /**
     * フィールドの名前を返却します。
     * 
     * @return フィールドの名前
     */
    public String getName() {
        return name;
    }

    /**
     * フィールドの名前を格納します。 <br>
     * 
     * @param name
     *            フィールドの名前
     */
    public void setName(String name) {
        this.name = name;
    }

}
