﻿Option Explicit On
Option Strict On
Option Infer Off

Imports System.Net
Imports System.IO.Ports

''' <summary>
''' OMRON PLC, FinsTCP protocol, Binary, TCP socket communication
''' </summary>
''' <remarks></remarks>
Public Class CmodeSerial
    Inherits PlcProtocol

    Dim SerialSync As SerialSync

    Dim PortName As String
    Dim BaudRate As Integer
    Dim Parity As Parity
    Dim DataBits As Integer
    Dim StopBits As StopBits
    Dim NewLine As String
    Dim ReceivedBytesThreshold As Integer
    Dim ReadBufferSize As Integer
    Dim ReadTimeout As Integer
    Dim DtrEnable As Boolean
    Dim RtsEnable As Boolean

    Private Const CommandReadBits1 As String = "RR"   ' Internal relay (bit)
    Private Const CommandReadBits2 As String = "RL"   ' Link relay (bit)
    Private Const CommandReadWords As String = "RD"
    Private Const CommandWriteBits1 As String = "WR"  ' Internal relay (word)
    Private Const CommandWriteBits2 As String = "WL"  ' Link relay (word)
    Private Const CommandWriteWords As String = "WD"

    Private Header As String = "@00"

    ''' <summary>
    ''' Constructor
    ''' </summary>
    ''' <param name="Q_Protocol"></param>
    ''' <remarks></remarks>
    Public Sub New(ByVal Q_Protocol As IEnumerable(Of XElement))
        Me.PortName = Q_Protocol.<PortName>.Value
        Me.BaudRate = CInt(Q_Protocol.<BaudRate>.Value)
        Me.Parity = DirectCast([Enum].Parse(GetType(Parity), Q_Protocol.<Parity>.Value), Parity)

        Me.DataBits = CInt(Q_Protocol.<DataBits>.Value)
        Me.StopBits = DirectCast([Enum].Parse(GetType(StopBits), Q_Protocol.<StopBits>.Value), StopBits)
        Me.NewLine = Q_Protocol.<NewLine>.Value
        Me.ReceivedBytesThreshold = CInt(Q_Protocol.<ReceivedBytesThreshold>.Value)
        Me.ReadBufferSize = CInt(Q_Protocol.<ReadBufferSize>.Value)
        Me.ReadTimeout = CInt(Q_Protocol.<ReadTimeout>.Value)
        Me.DtrEnable = CBool(Q_Protocol.<DtrEnable>.Value)
        Me.RtsEnable = CBool(Q_Protocol.<RtsEnable>.Value)

        SerialSync = New SerialSync(PortName, BaudRate, Parity, DataBits, StopBits, NewLine, ReceivedBytesThreshold, ReadBufferSize, ReadTimeout, DtrEnable, RtsEnable)
    End Sub

    ''' <summary>
    ''' Connect to PLC
    ''' </summary>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Public Overrides Function Connect() As Integer
        Dim result As Integer = SerialSync.Open
        Return result
    End Function

    ''' <summary>
    ''' Disconnect from PLC
    ''' </summary>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Public Overrides Function Disconnect() As Integer
        Dim result As Integer = SerialSync.Close
        Return result
    End Function

    ''' <summary>
    ''' Send word data to PLC
    ''' </summary>
    ''' <param name="DeviceName"></param>
    ''' <param name="StartDevice"></param>
    ''' <param name="DeviceCount"></param>
    ''' <param name="RecvDataRet"></param>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Public Overrides Function ReadWords(ByVal DeviceName As String, ByVal StartDevice As UInteger, ByVal DeviceCount As Integer, ByRef RecvDataRet As UInt16()) As Integer
        Try
            Dim SendData As String = Header
            Dim strValue As String
            Dim ErrCode As String

            ReDim RecvDataRet(DeviceCount - 1)

            If DeviceName.Equals("RR") Or DeviceName.Equals("IR") Then
                SendData += "RR"                                                        ' I/O type: Internal Relay
            ElseIf DeviceName.Equals("LR") Then
                SendData += "RL"                                                        ' I/O type: Link Relay
            ElseIf DeviceName.Equals("DM") Then
                SendData += "RD"                                                        ' I/O type: Data memory
            End If

            SendData += StartDevice.ToString("0000")                                    ' Start Device
            SendData += DeviceCount.ToString("0000")                                    ' Device Count
            SendData += fncCalcFCS(SendData)
            SendData += "*"

            SerialSync.Send(SendData)

            Dim RecvLengthWish As Integer = 7 + DeviceCount * 4 + 3
            Dim RecvData As String = SerialSync.Recv()


            If RecvData.Length >= 7 Then
                If RecvData.Length = RecvLengthWish Then
                    ErrCode = RecvData.Substring(5, 2)
                    If ErrCode.Equals("00") Then
                        strValue = RecvData.Substring(7)
                        For count As Integer = 0 To DeviceCount - 1
                            RecvDataRet(count) = Convert.ToUInt16(strValue.Substring(4 * count, 4), 16)
                        Next
                        Return 0
                    Else
                        err = "ResultCodeError"
                        Return -1
                    End If
                Else
                    err = "RecvLengthError"
                    Return -1
                End If
            Else
                err = "RecvLengthError"
                Return -1
            End If
        Catch ex As Exception
            err = ex.Message
            Return -1
        End Try
    End Function

    Private Function fncCalcFCS(ByVal Msg As String) As String
        Dim TempValue As Integer = 0
        Dim StrValue As String

        For Count As Integer = 1 To Len(Msg)
            TempValue = Asc(Strings.Mid(Msg, Count, 1)) Xor TempValue
        Next

        StrValue = "00" + Hex(TempValue)
        Return Strings.Mid(StrValue, StrValue.Length - 1, 2)
    End Function

    ''' <summary>
    ''' Write multiple word data to PLC
    ''' </summary>
    ''' <param name="DeviceName"></param>
    ''' <param name="StartDevice"></param>
    ''' <param name="DeviceCount"></param>
    ''' <param name="UInt16Data"></param>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Public Overrides Function WriteWords(ByVal DeviceName As String, ByVal StartDevice As UInteger, ByVal DeviceCount As Integer, ByVal UInt16Data() As UInt16) As Integer
        Try
            Dim SendData As String = Header
            Dim strValue As String
            Dim ErrCode As String

            SendData += DeviceName
            If DeviceName.Equals("IR") Then
                SendData += "WR"                                                        ' I/O type: Internal Relay
            ElseIf DeviceName.Equals("LR") Then
                SendData += "WL"                                                        ' I/O type: Link Relay
            ElseIf DeviceName.Equals("DM") Then
                SendData += "WD"                                                        ' I/O type: Data memory
            End If

            SendData += StartDevice.ToString("0000")                                    ' Start Device
            SendData += DeviceCount.ToString("0000")                                    ' Device Count

            For Count As Integer = 0 To DeviceCount - 1
                strValue = "0000" + (UInt16Data(Count).ToString("X"))
                SendData += strValue.Substring(strValue.Length - 4, 4)           ' Data count (4byte)
            Next

            SendData += fncCalcFCS(SendData)
            SendData += "*" + vbCrLf

            SerialSync.Send(SendData)

            Dim RecvLengthWish As Integer = 7 + 3
            Dim RecvData As String = SerialSync.Recv()


            If RecvData.Length = RecvLengthWish Then
                ErrCode = RecvData.Substring(5, 2)
                If ErrCode.Equals("00") Then
                    strValue = RecvData.Substring(7)
                    Return 0
                Else
                    err = "ResultCodeError"
                    Return -1
                End If
            Else
                err = "RecvLengthError"
                Return -1
            End If
        Catch ex As Exception
            err = ex.Message
            Return -1
        End Try
    End Function
End Class
