;;;

;;; The file is a derivative of
;;; egg/anthy.el --- ANTHY Support (high level interface) in Egg
;;;                Input Method Architecture

;; Copyright (C) 2002 The Free Software Initiative of Japan

;; Author: NIIBE Yutaka <gniibe@m17n.org>

;; Maintainer: NIIBE Yutaka <gniibe@m17n.org>

;; Keywords: mule, multilingual, input method

;; This file is part of EGG.

;; EGG is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; EGG is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to the
;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.

;;; Commentary:


;;; Code:

(require 'egg)
(require 'egg-edep)

(defgroup mana nil
  "Mana interface for Tamago 4."
  :group 'egg)

(setplist 'mana-conversion-backend
	  '(egg-start-conversion          mana-convert
	    egg-get-bunsetsu-source       mana-get-bunsetsu-source
	    egg-get-bunsetsu-converted    mana-get-bunsetsu-converted
	    egg-list-candidates           mana-get-candidates
	    egg-decide-candidate          mana-select-candidate
	    egg-change-bunsetsu-length    mana-resize-segment
	    egg-end-conversion            mana-commit
	    ;;
	    egg-get-source-language       mana-get-source-language
	    egg-get-converted-language    mana-get-converted-language
	    egg-word-registration         mana-word-registration))

(defconst mana-backend-alist '((Japanese ((mana-conversion-backend)))))

(egg-set-finalize-backend '(mana-finalize-backend))

;; <ptid> ::= <integer>

(defvar mana-proc nil
  "Process of MANA helper agent.")

(defun mana-cmd-prim (val)
  (let ((buffer (process-buffer mana-proc)))
    (if (and (eq (process-status mana-proc) 'run)
	      (buffer-live-p buffer))
	 (save-excursion
	   (set-buffer buffer)
	   (let ((p (point)))
	     (newline)
	     (prin1 val buffer)
	     (newline)
	     (process-send-region mana-proc p (point))
	     (accept-process-output mana-proc)))
      (egg-error "process %s was killed" mana-proc))))

(defun mana-init-proc () nil)

(defun mana-start-proc ()
  (if (null mana-proc)
      (let ((buf (generate-new-buffer "*MANA*"))
	    (process-connection-type nil)) ; avoid using pty
	(setq mana-proc
	      (start-process "mana" buf "mana"))
	(process-kill-without-query mana-proc)
	(set-process-coding-system mana-proc 'euc-jp-unix 'euc-jp-unix)
	(set-process-sentinel mana-proc 'mana-proc-sentinel)
	(set-marker-insertion-type (process-mark mana-proc) t)
	(save-excursion
	  (set-buffer buf)
	  (erase-buffer)
	  (buffer-disable-undo))
	(mana-init-proc))))

;;; XXX: Don't kill buffer (for now) so that I can debug this program
(defun mana-proc-sentinel (proc reason)
;  (kill-buffer (process-buffer proc))
  (setq mana-proc nil))

(defun mana-cmd (val)
  (mana-start-proc)
  (mana-cmd-prim val))

(defun mana-eval (val)
  (mana-start-proc)
  (let ((buffer (process-buffer mana-proc)))
    (if (and (eq (process-status mana-proc) 'run)
	      (buffer-live-p buffer))
	 (save-excursion
	   (set-buffer buffer)
	   (goto-char (point-max))
	   (let ((p (point)))
	     (prin1 val buffer)
	     (newline)
	     (process-send-region mana-proc p (point))
	     (accept-process-output mana-proc)
	     (goto-char (point-max))
	     (forward-line -1)
	     (beginning-of-line)
	     (read buffer)))
      (egg-error "process %s was killed" mana-proc))))

(defun mana-best-path (yomi state pos len) 
  (mana-eval `(mana-best-path ,yomi , state ,pos ,len)))

(defun mana-list-candidates (yomi state pos mrph-len len)
  (mana-eval `(mana-list-candidates ,yomi ,state ,pos ,mrph-len ,len)))

(defun mana-add-new-word (kaki yomi)
  (mana-cmd `(mana-add-new-word ,kaki ,yomi)))

;;
;; <mana-bunsetsu> ::=
;;  [ <string> <pos> <len> <state> <converted> <candidates> <candidate-pos> ]
(defun mana-make-bunsetsu (string pos len state converted)
  (egg-bunsetsu-create
   'mana-conversion-backend
   (vector string pos len state converted nil 0)))

(defun manabunsetsu-get-string (b)
  (aref (egg-bunsetsu-get-info b) 0))
(defun manabunsetsu-get-pos (b)
  (aref (egg-bunsetsu-get-info b) 1))
(defun manabunsetsu-get-len (b)
  (aref (egg-bunsetsu-get-info b) 2))
(defun manabunsetsu-get-state (b)
  (aref (egg-bunsetsu-get-info b) 3))

(defun manabunsetsu-get-source (b)
  (let ((string (aref (egg-bunsetsu-get-info b) 0))
	(pos (aref (egg-bunsetsu-get-info b) 1))
	(len (aref (egg-bunsetsu-get-info b) 2)))
    (substring string pos (+ pos len))))

(defun manabunsetsu-get-converted (b)
  (aref (egg-bunsetsu-get-info b) 4))
(defun manabunsetsu-get-candidates (b)
  (aref (egg-bunsetsu-get-info b) 5))
(defun manabunsetsu-set-candidates (b z)
  (aset (egg-bunsetsu-get-info b) 5 z))
(defun manabunsetsu-get-candidate-pos (b)
  (aref (egg-bunsetsu-get-info b) 6))
(defun manabunsetsu-set-candidate-pos (b zp)
  (aset (egg-bunsetsu-get-info b) 6 zp))

(defun mana-get-bunsetsu-source (b)
  (manabunsetsu-get-source b))

(defun mana-get-bunsetsu-converted (b)
  (let ((cands (manabunsetsu-get-candidates b)))
    (if cands
	(car (nth (manabunsetsu-get-candidate-pos b) cands))
      (manabunsetsu-get-converted b))))

(defun mana-get-source-language (b) 'Japanese)
(defun mana-get-converted-language (b) 'Japanese)

;;
;; Returns list of bunsetsu
;;

(defun mana-make-bunsetsu-list (string path)
  (mapcar (lambda (node)
	    (let* ((converted (elt node 0))
		   (pos (elt node 1))
		   (len (elt node 2))
		   (state (elt node 3)))
	      (mana-make-bunsetsu string pos len state converted)))
	  path))

(defun mana-convert (backend yomi &optional context)
  "Convert YOMI string to kanji, and enter conversion mode.
Return the list of bunsetsu."
  (set-text-properties 0 (length yomi) nil yomi)
  (mana-make-bunsetsu-list yomi 
			   (mana-best-path yomi 0 0 (length yomi))))

;;
;;
;;

(defun mana-commit (bunsetsu-list abort) nil)

;;
;; Returns ( <pos> <candidates> )
;;
(defun mana-get-candidates (bunsetsu-list prev-bunsetsu next-bunsetsu major)
  (let ((bunsetsu (car bunsetsu-list)))

    (unless (manabunsetsu-get-candidates bunsetsu)
      (let* ((state (manabunsetsu-get-state bunsetsu))
	     (source (manabunsetsu-get-source bunsetsu))
	     (len (manabunsetsu-get-len bunsetsu))
	     (cands (mana-list-candidates source state 0 len len)))
	(cons (manabunsetsu-set-candidate-pos bunsetsu 0)
	      (manabunsetsu-set-candidates bunsetsu cands))))

      (cons (manabunsetsu-get-candidate-pos bunsetsu)
	    (mapcar 'car (manabunsetsu-get-candidates bunsetsu)))))

;; Helper function
(defun mana-get-yomi (bunsetsu-list) 
  (mapconcat 'manabunsetsu-get-source bunsetsu-list ""))

;; Returns list of list of bunsetsu

(defun mana-select-candidate (bunsetsu-list candidate-pos prev-b next-b)
  (let ((bunsetsu (car bunsetsu-list)))
    (manabunsetsu-set-candidate-pos bunsetsu candidate-pos)
    (list (list bunsetsu))))

;; Returns list of list of bunsetsu
(defun mana-resize-segment (bunsetsu-list prev-b next-b len major)
  (let* ((bunsetsu (car bunsetsu-list))
	 (string (manabunsetsu-get-string bunsetsu))
	 (pos (manabunsetsu-get-pos bunsetsu))
	 (state (manabunsetsu-get-state bunsetsu))
	 (candidate (car (mana-list-candidates string state pos len len)))
	 (conv (nth 0 candidate))
	 (next-state (nth 1 candidate))
	 (next-pos (+ pos len))
	 (next-len (- (length string) next-pos))
	 (path (mana-best-path string next-state next-pos next-len)))
    (list (list (mana-make-bunsetsu string pos len state conv))
	  nil 
	  (mana-make-bunsetsu-list string path))))

(defun mana-finalize-backend ()
  (if mana-proc
      (progn
	(delete-process mana-proc)
	(setq mana-proc nil))))

;;Register new word
(defun mana-word-registration (backend str yomi)
  (let ((str (copy-sequence str))
	(yomi (copy-sequence yomi)))
    (set-text-properties 0 (length str) nil str)
    (set-text-properties 0 (length yomi) nil yomi)
    (mana-add-new-word str yomi)
    '("̤" "Ŀͼ")))

;;; setup

(run-hooks 'mana-load-hook)

;;;###autoload
(defun egg-activate-mana (&rest arg)
  "Activate MANA backend of Tamago 4."
  (apply 'egg-mode (append arg mana-backend-alist)))

;;; egg-mana.el ends here.