/* vim: set encoding=utf8:
 *
 * shiki.c
 *
 * This file is main file of Shiki.
 *
 * Copyright(C)2006 WAKATSUKI toshihiro
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * $Id: shiki.c,v 1.65 2006/12/28 05:04:45 aloha Exp $
 */

#include"shiki.h"

/* エディタ全体に関する情報と，現在表示されているタブ情報へのキャッシュを管理する構造体 */
ShikiEditorType Shiki_editor;

static gchar *R5RS_keywords[]  = {"and", "begin", "case", "cond-expand", "cond", "define-accessor", "define-class", "defined?", "define-generic", "define", "define-macro", "define-method", "define-module", "define-private", "define-public", "define-reader-ctor", "define-syntax", "define-syntax-macro", "defmacro", "defmacro*-public", "delay", "do", "else", "fluid-let", "if", "lambda", "let", "let*", "letrec", "letrec-syntax", "let-syntax", "or", "quasiquote", "quote", "set!", "syntax-rules", "unquote", NULL};

static gchar *R5RS_functions[] = {"*", "+", "-", "/", "<", ">", "<=", ">=", "?", "`", "=", "abs", "acos", "angle", "append", "apply", "asin", "assoc", "assq", "assv", "atan", "boolean?", "caaar", "caadr", "caar", "cadar", "caddr", "cadr", "call/cc", "call-with-current-continuation", "call-with-input-file", "call-with-output-file", "call-with-values", "car", "catch", "cdaar", "cdadr", "cdar", "cddar", "cdddr", "cddr", "cdr", "ceiling", "char-alphabetic?", "char-ci>=?", "char-ci>?", "char-ci=?", "char-ci<=?", "char-ci<?", "char-downcase", "char->integer", "char>=?", "char>?", "char=?", "char?", "char-lower-case?", "char<=?", "char<?", "char-numeric?", "char-ready?", "char-upcase", "char-upper-case?", "char-whitespace?", "close-input-port", "close-output-port", "complex?", "cons", "cos", "current-input-port", "current-output-port", "delete-file", "display", "dynamic-wind", "eof-object?", "eq?", "equal?", "eqv?", "eval", "even?", "exact->inexact", "exact?", "exit", "exp", "expt", "file-exists?", "file-or-directory-modify-seconds", "floor", "force", "for-each", "gcd", "gensym", "getenv", "get-output-string", "imag-part", "inexact?", "input-port?", "integer->char", "integer?", "lcm", "length", "list->string", "list->vector", "list", "list?", "list-ref", "list-tail", "load", "log", "magnitude", "make-polar", "make-rectangular", "make-string", "make-vector", "map", "max", "member", "memq", "memv", "min", "modulo", "negative?", "newline", "nil", "not", "null?", "number->string", "number?", "odd?", "open-input-file", "open-input-string", "open-output-file", "open-output-string", "output-port?", "pair?", "peek-char", "port?", "positive?", "procedure?", "quotient", "rational?", "read-char", "read", "read-line", "real?", "real-part", "remainder", "reverse", "reverse!", "round", "set-car!", "set-cdr!", "sin", "sqrt", "string-append", "string-ci>=?", "string-ci>?", "string-ci=?", "string-ci<=?", "string-ci<?", "string-copy", "string-fill!", "string>=?", "string>?",  "string->list", "string->number", "string->symbol", "string", "string=?", "string?", "string-length", "string<=?", "string<?", "string-ref", "string-set!", "substring", "symbol->string", "symbol?", "system", "tan", "truncate", "values", "vector-fill!", "vector->list", "vector", "vector?", "vector-length", "vector-ref", "vector-set!", "with-input-from-file", "with-output-to-file", "write-char", "write", "zero", NULL};

static GHashTable *keywords_hash = NULL;

typedef enum {
  R5RS_KEYWORD_COLOR = 1,
  R5RS_FUNCTION_COLOR,
  GAUCHE_KEYWORD_COLOR,
  GAUCHE_FUNCTION_COLOR
} HIGHILIGHT_COLOR; 

GdkColor COLOR_BLACK;
GdkColor COLOR_GREEN;

/* プロトタイプ */

/* foo_bar_handler() は，イベントハンドラのためのラッパー */
static void kill_buffer_handler();
static void append_default_tabpage_handler();

/* ファイル関係 */
static void save_file();
static void save_file_as();

/* テキストバッファ関係 */
static gchar* get_all_buffer_contents(GtkTextBuffer *buffer);
static gboolean save_text_buffer(const gchar *filename, GtkTextBuffer *buffer);
static void clear_current_buffer_handler();


/* FIXME !!!!!! : なぜかロードすると SEGV で落ちる !!!!!!!!!!!!!!!!!!!!!! */


/* Gauche と S 式操作やインデント・キーワードハイライティング関連 */
static gchar *load_cstring_by_gauche(gchar *s);
static void load_buffer_by_gauche();
static void load_region_by_gauche();
static void load_scheme_file_by_gauche();
static gint get_parent_nest_level_at_cursor(GtkTextBuffer *buffer);
static gboolean is_not_scheme_delimita_p(gunichar ch, gpointer user_data);
static gboolean is_double_quote(gunichar ch, gpointer user_data);
static gboolean is_scheme_delimita_p(gunichar ch, gpointer user_data);
static void scheme_keyword_highlighting_current_buffer();

/* 設定 */
static void select_font();
static void font_selection_ok(GtkWidget *button, GtkWidget *font_dialog);
static void switch_tabpage_handler(GtkNotebook *notebook, GtkNotebookPage *page, guint pagenum) ;
static void tabsborder_on_off(GtkButton *button, GtkNotebook *notebook);
static void rotate_tab_position(GtkButton *button, GtkNotebook *notebook);

/* キーバインド全般 */
static gboolean signal_key_press_handler(GtkWidget *notebook, GdkEventKey *event, gpointer contextid);
static gboolean signal_key_release_handler(GtkWidget *notebook, GdkEventKey *event, gpointer contextid);

/* ヘルプや情報 */
static void open_online_help();
static void about_this_application();

/* エディタの初期化 */
static void Shiki_editor_window_init(int argc, char **argv);

/* バッファの内容を消去 */
static void clear_current_buffer_handler() {
  Shiki_erase_buffer(Shiki_CURRENT_TEXT_BUFFER);
}

static gboolean is_not_scheme_delimita_p(gunichar ch, gpointer user_data) {
  return ch != '(' && ch != ')' &&  !g_unichar_isspace(ch);
}

static gboolean is_double_quote(gunichar ch, gpointer user_data) {
  return ch == '\"';
}

static gboolean is_scheme_delimita_p(gunichar ch, gpointer user_data) {
  return ch == ' ' || ch == '(' || ch == ')' || ch == '\"' || g_unichar_isspace(ch);
}

/* ソースコードの色分け */
static void scheme_keyword_highlighting_current_buffer() {
  GtkTextIter s, e;
  HIGHILIGHT_COLOR c;
  gchar *word;
  gboolean is_comment, is_string;
  gunichar ch;

  gtk_text_buffer_get_start_iter(Shiki_CURRENT_TEXT_BUFFER, &s);

  /* 簡単な Scheme レキシカルアナライザ */
  while(TRUE) {
    is_comment = FALSE;
    is_string = FALSE;
    if((ch = gtk_text_iter_get_char(&s)) != ';' && ch != '\"')
      gtk_text_iter_forward_find_char(&s, is_not_scheme_delimita_p, NULL, NULL);
    e = s;
    if(gtk_text_iter_get_char(&s) == ';') {
      gtk_text_iter_forward_line(&e);
      gtk_text_iter_backward_char(&e);
      is_comment = TRUE;
    } else if(gtk_text_iter_get_char(&s) == '\"') {
      while(TRUE) {
        gtk_text_iter_forward_find_char(&e, is_double_quote, NULL, NULL);
        gtk_text_iter_backward_char(&e);
        if(gtk_text_iter_get_char(&e) != '\\') {
          is_string = TRUE;
          gtk_text_iter_forward_char(&e);
          gtk_text_iter_forward_char(&e);
          break;
        }
        gtk_text_iter_forward_char(&e);
        gtk_text_iter_forward_char(&e);
      }

    } else
      gtk_text_iter_forward_find_char(&e, is_scheme_delimita_p, NULL, NULL);
    
    word = gtk_text_buffer_get_text(Shiki_CURRENT_TEXT_BUFFER, &s, &e, FALSE);

    /* 対応する色をそれぞれトークン部分につける */
    if(is_comment) /* コメント */
      gtk_text_buffer_apply_tag_by_name(Shiki_CURRENT_TEXT_BUFFER, "comment_highlighting", &s, &e);
    else if(is_string) /* 文字列 */
      gtk_text_buffer_apply_tag_by_name(Shiki_CURRENT_TEXT_BUFFER, "string_highlighting", &s, &e);    
    else if(R5RS_KEYWORD_COLOR == (c = GPOINTER_TO_INT(g_hash_table_lookup(keywords_hash, word)))) /* R5RS キーワード */
      gtk_text_buffer_apply_tag_by_name(Shiki_CURRENT_TEXT_BUFFER, "keyword_highlighting", &s, &e);
    else if(R5RS_FUNCTION_COLOR == c) /* R5RS 関数 */
      gtk_text_buffer_apply_tag_by_name(Shiki_CURRENT_TEXT_BUFFER, "function_highlighting", &s, &e);

    /* XXX : get_text() でいちいち文字列がアロケートされるから，非常に効率が悪いと思うけど… GtkTextBuffer から const gchar * が取れれば良いのに… */
    g_free(word); 

    if(gtk_text_iter_is_end(&e)) break;
    s = e;
  }
}

/* タブが切り替わる時のイベントハンドリング */
static void switch_tabpage_handler(GtkNotebook *notebook, GtkNotebookPage *page, guint pagenum) {
  /* タブに対応する情報が格納された構造体を切替える */
  Shiki_CURRENT_TAB_INFO = (ShikiBuffer *)g_list_nth_data(Shiki_EDITOR_BUFFER_LIST, pagenum);

  /* 現在のタブ番号を切替える */
  Shiki_CURRENT_TAB_NUM = pagenum;

  /* タブのラベルをウィンドウのタイトルに */
  if(!Shiki_CURRENT_TAB_INFO) return;
  gtk_window_set_title (GTK_WINDOW(Shiki_EDITOR_WINDOW), Shiki_CURRENT_FILENAME);

  Shiki_update_modeline(Shiki_CURRENT_TEXT_BUFFER);  
}

/* GtkTextCharPredicate */
static gboolean is_kakko_or_kokka(gunichar ch, gpointer p) {
  return ch == '(' || ch == ')';
}
static gboolean is_kakko(gunichar ch, gpointer p) {return ch == '(';}
static gboolean is_kokka(gunichar ch, gpointer p) {return ch == ')';}

static gboolean search_sexp_kokka(GtkTextIter *end) {
  gint nest_level = 0;

  /* 対応する ')' を探す */
  while(1) {
    if(!gtk_text_iter_forward_find_char(end, is_kakko_or_kokka, NULL, NULL))
      return FALSE;

    if(gtk_text_iter_get_char(end) == '(')
      nest_level++;
    else {
      if(!nest_level)
        break;
      else
        nest_level--;
    }
  }
  return TRUE;
}

/* カーソル以降の '(' に対応する ')' までの文字列 (S 式) を切り出す */
static gboolean search_sexp(GtkTextIter *start, GtkTextIter *end) {

  /* カーソルの位置を取得 */
  gtk_text_buffer_get_iter_at_mark(Shiki_CURRENT_TEXT_BUFFER, start, gtk_text_buffer_get_insert(Shiki_CURRENT_TEXT_BUFFER));

  if(gtk_text_iter_get_char(start) != '(')
    gtk_text_iter_forward_find_char(start, is_kakko, NULL, NULL);

  *end = *start;

  /* カーソル位置の前にある S 式を切り出す */
  if(!search_sexp_kokka(end)) return FALSE;
  gtk_text_iter_forward_char(end);
  return TRUE;
}

/* ')' に対応する '(' までの文字列 (S 式) を切り出す */
static gboolean search_last_sexp_kakko(GtkTextIter *start) {
  gint nest_level = 0;
  /* ネストレベルを計算しながら ')' を探す */
  while(1) {
    if(!gtk_text_iter_backward_find_char(start, is_kakko_or_kokka, NULL, NULL))
      return FALSE;

    if(gtk_text_iter_get_char(start) == ')')
      nest_level++;
    else {
      if(!nest_level)
        break;
      else
        nest_level--;
    }
  }
  return TRUE;
}

/* カーソル以前の ')' に対応する '(' までの文字列 (S 式) を切り出す */
static gboolean search_last_sexp(GtkTextIter *start, GtkTextIter *end) {

  /* カーソルの位置を取得 */
  gtk_text_buffer_get_iter_at_mark(Shiki_CURRENT_TEXT_BUFFER, end, gtk_text_buffer_get_insert(Shiki_CURRENT_TEXT_BUFFER));

  gtk_text_iter_backward_char(end);

  if(gtk_text_iter_get_char(end) != ')')
    gtk_text_iter_backward_find_char(end, is_kokka, NULL, NULL);
  *start = *end;
  gtk_text_iter_forward_char(end);

  /* カーソル位置の前にある S 式を切り出す */
  if(!search_last_sexp_kakko(start)) return FALSE;

  return TRUE;
}

/* キーが押された */
static gboolean signal_key_press_handler (GtkWidget *notebook, GdkEventKey *event, gpointer contextid) {
  GtkTextIter start, end;

  /* 括弧の対応の強調を無効に */
  gtk_text_buffer_get_start_iter(Shiki_CURRENT_TEXT_BUFFER, &start);
  gtk_text_buffer_get_end_iter(Shiki_CURRENT_TEXT_BUFFER, &end);
  gtk_text_buffer_remove_tag_by_name(Shiki_CURRENT_TEXT_BUFFER, "parent_emphasis_background", &start, &end);

  if(event->state & GDK_CONTROL_MASK && event->state & GDK_MOD1_MASK) {
    switch(event->keyval) {
      case GDK_at : /* C-M-SPC */
        { GtkTextIter start, end;
          if(!search_sexp(&start, &end)) return FALSE;
          gtk_text_buffer_select_range(Shiki_CURRENT_TEXT_BUFFER, &start, &end);
        }
        break;
      case GDK_space : /* C-M-SPC */
        { GtkTextIter start, end;
          if(!search_last_sexp(&start, &end)) return FALSE;
          gtk_text_buffer_select_range(Shiki_CURRENT_TEXT_BUFFER, &start, &end);
        }
        break;
    }
  } else if(event->state & GDK_CONTROL_MASK) {
    switch(event->keyval) {
      case GDK_f :  /* Ctrl + f : forward */
        Shiki_forward_char();
        break;
      case GDK_b : /* Ctrl + b : backward */
        Shiki_backward_char();
        break;
      case GDK_n : /* Ctrl + n : next line */
        Shiki_forward_line(1);
        break;
      case GDK_p :  /* Ctrl + p : previous line */
        Shiki_forward_line(-1);
        break;
      case GDK_h :
        { GtkTextIter p;
          gtk_text_buffer_get_iter_at_mark(Shiki_CURRENT_TEXT_BUFFER,&p, gtk_text_buffer_get_insert(Shiki_CURRENT_TEXT_BUFFER));
          gtk_text_buffer_backspace(Shiki_CURRENT_TEXT_BUFFER, &p, FALSE, TRUE);
        }
        break;

      case GDK_e :  /* Ctrl + e : eval-expression */
        Shiki_eval_expression();
        break;

      case GDK_j :  /* Ctrl + j : eval-last-sexp */
        Shiki_eval_last_sexp();
        break;

      case GDK_backslash : /* Ctrl + \ : Undo */
        Shiki_undo();
        break;

      case GDK_underscore : /* Ctrl + _ : Redo */
        Shiki_redo();
        break;

      case GDK_t : /* Ctrl + t : タブを開く */
        append_default_tabpage_handler();
        break;

      case GDK_k : /* Ctrl + k : タブを閉じる */
        kill_buffer_handler();
        break;

      case GDK_w : /* Ctrl + w : カット */
        gtk_text_buffer_cut_clipboard(Shiki_CURRENT_TEXT_BUFFER, Shiki_EDITOR_CLIPBOARD, TRUE);
        break;

      case GDK_y : /* Ctrl + y : ヤンク */
        {GtkTextIter p;
          gtk_text_buffer_get_iter_at_mark(Shiki_CURRENT_TEXT_BUFFER,&p, gtk_text_buffer_get_insert(Shiki_CURRENT_TEXT_BUFFER));
          gtk_text_buffer_paste_clipboard(Shiki_CURRENT_TEXT_BUFFER, Shiki_EDITOR_CLIPBOARD, &p, TRUE);
        }
        break;        
    }
  }
  return FALSE;
}

static void append_default_tabpage_handler() {
  Shiki_new_buffer_create(g_strdup("*scratch*")); 
}

/* ノートブックからタブとページ (バッファ) を削除 */
static void kill_buffer_handler() {
  Shiki_kill_buffer(Shiki_CURRENT_TEXT_BUFFER);
}

/* バッファをまるごとロード */
static void load_buffer_by_gauche() {
  GtkTextIter p;
  gtk_text_buffer_get_end_iter(Shiki_CURRENT_TEXT_BUFFER, &p);
  gtk_text_buffer_insert(Shiki_CURRENT_TEXT_BUFFER, &p, "\n\n", -1);
  gtk_text_buffer_insert(Shiki_CURRENT_TEXT_BUFFER, &p, load_cstring_by_gauche(get_all_buffer_contents(Shiki_CURRENT_TEXT_BUFFER)), -1); 
}



/* ファイルをロード */
static void load_scheme_file_by_gauche() {
  const gchar *filename = Shiki_file_name_dialog("File Selection");

  if(!filename) return;
  Shiki_load_file(filename);
}

static gint compBuffer(gconstpointer a, gconstpointer b) {
  return ((ShikiBuffer *)a)->text_buffer == b ? 0 : b - a;
}

/* バッファに関連付けられたファイルをリロード．環境はクリアされる */
static void reload_buffer() {
#if 0 
  Shiki_CURRENT_BUFFER_ENV = Scm_MakeModule(NULL, FALSE);

  /* xyzzy lisp 関数を登録 */
  Scm_Init_xyzzylisp(SCM_MODULE(Shiki_CURRENT_BUFFER_ENV));
  Scm_EvalCString("(set! *mode-line-format* (lambda () (format #f \"--~A- ~A (Gauche Interaction) [GtkDefault (utf8)]     L~S:~S             \" (if (buffer-modified-p) \"--\" \"**\") (buffer-name (selected-buffer)) (current-line-number) (current-column))))", Shiki_CURRENT_BUFFER_ENV);
#endif

  Shiki_load_file(Shiki_CURRENT_FILENAME);
}


/* gauche を起動して文字列をロード */
static gchar *load_cstring_by_gauche(gchar *s) {
  gchar *msg;

  g_print("%s\n", s);
  ScmObj result, error;
  /* 入力文字列ポートを開く */
  ScmObj is = Scm_MakeInputStringPort(SCM_STRING(SCM_MAKE_STR_COPYING(s)), TRUE);
  /* 出力文字列ポート開く */
  ScmObj os = Scm_MakeOutputStringPort(TRUE);

  Scm_Define(SCM_MODULE(Shiki_CURRENT_BUFFER_ENV), SCM_SYMBOL(SCM_INTERN("*input*")), is);
  Scm_Define(SCM_MODULE(Shiki_CURRENT_BUFFER_ENV), SCM_SYMBOL(SCM_INTERN("*error*")), SCM_FALSE);
  /* Scheme のレベルでエラーハンドリングをしつつ，ポートから順番に S 式を読み込み，評価していく．コンパイルエラーなどは *error* に捕捉される */
  result = Scm_EvalCString("(guard (e (else (set! *error* e) #f)) (eval (load-from-port *input*) (current-module)))", SCM_OBJ(Shiki_CURRENT_BUFFER_ENV));

  error = Scm_GlobalVariableRef(SCM_MODULE(Shiki_CURRENT_BUFFER_ENV), SCM_SYMBOL(SCM_INTERN("*error*")), 0);

  /* 文字列を評価した結果をポートに書き込む */
  if (!SCM_FALSEP(error))
    Scm_Write(error, os, SCM_WRITE_DISPLAY);
  else
    Scm_Write(result, os, SCM_WRITE_DISPLAY);

  msg = Scm_GetString(SCM_STRING(Scm_GetOutputString(SCM_PORT(os))));
  /* ポート閉じる */
  Scm_ClosePort(SCM_PORT(is));
  Scm_ClosePort(SCM_PORT(os));

  return msg;
}

static void font_selection_ok(GtkWidget *button, GtkWidget *font_dialog) {
  gchar *font_name = gtk_font_selection_dialog_get_font_name (GTK_FONT_SELECTION_DIALOG (font_dialog));
  if(font_name) {
    GtkRcStyle *style = gtk_rc_style_new ();
    pango_font_description_free(style->font_desc);
    style->font_desc = pango_font_description_from_string(font_name);
    gtk_widget_modify_style (GTK_WIDGET(Shiki_CURRENT_TEXT_VIEW), style);
    gtk_rc_style_unref (style);
    g_free (font_name);
  }
}

/* フォントを選択させるイベントハンドラ */
static void select_font(){
  GtkWidget *font_dialog = gtk_font_selection_dialog_new("Font Selection Dialog");
  g_signal_connect (GTK_FONT_SELECTION_DIALOG (font_dialog)->ok_button, "clicked", G_CALLBACK(font_selection_ok), font_dialog);
  gtk_dialog_run(GTK_DIALOG(font_dialog));
  gtk_widget_destroy(font_dialog);
}

/* このアプリケーションについて */
static void about_this_application() {
  GtkAboutDialog *about = GTK_ABOUT_DIALOG(gtk_about_dialog_new());
  const gchar *authors[] = {
    "若槻俊宏 (あろは) <alohakun@gmail.com>\n",
    "Contribute : tkng さん",
    "(http://d.hatena.ne.jp/tkng/20061113)", NULL
  };
  gtk_about_dialog_set_authors(about, authors);
  gtk_about_dialog_set_copyright(about, "Copyright(C)2006  WAKATSUKI Toshihiro");
  gtk_about_dialog_set_name(about, "式 (SHIKI)");
  gtk_about_dialog_set_website_label(about, "本当に30日でエディタが出来上がるのかを試してみるBlog");
  gtk_about_dialog_set_website(about, "http://alohakun.blog7.fc2.com/blog-category-29.html");
  gtk_dialog_run(GTK_DIALOG(about));
  gtk_widget_destroy(GTK_WIDGET(about));
}

#if 0
/* 開発途中に使用したダミーイベントハンドラー */
static void dummy_handler() {
  GtkWidget *dummy = gtk_message_dialog_new(GTK_WINDOW(Shiki_EDITOR_WINDOW),
      GTK_DIALOG_DESTROY_WITH_PARENT, GTK_MESSAGE_WARNING, GTK_BUTTONS_CLOSE,
      "Sorry... This Button is Dummy.");
  gtk_dialog_run(GTK_DIALOG(dummy));
  gtk_widget_destroy(dummy);
}
#endif

/* テキストバッファから全ての文字列を取り出す */
static gchar* get_all_buffer_contents(GtkTextBuffer *buffer) {
  GtkTextIter start, end;
  gtk_text_buffer_get_start_iter(buffer, &start);
  gtk_text_buffer_get_end_iter(buffer, &end);
  return gtk_text_buffer_get_text(buffer, &start, &end, FALSE);
}

/* buffer の内容をファイル filename に保存 */
static gboolean save_text_buffer(const gchar *filename, GtkTextBuffer *buffer) {
  gchar *contents, *text;
  gsize br, bw;
  GError *err = NULL;

  if(!filename) return FALSE;
  contents = get_all_buffer_contents(buffer);
  text = g_locale_from_utf8(contents, -1, &br, &bw, &err);
  /* 文字列をファイルに保存 */
  g_file_set_contents(filename, text, -1, NULL);
  gtk_text_buffer_set_modified(buffer, FALSE);
  Shiki_update_modeline(Shiki_CURRENT_TEXT_BUFFER);
  g_free(contents); g_free(text);
  return TRUE;
}

/* 現在表示されているページの内容をファイルに保存 */
static void save_file() {

  /* ヘルプへの変更は保存しない */
  if(strcmp("*help*", Shiki_CURRENT_TAB_TITLE) == 0) return;

  /* 変更が無ければ何もしない */
  if(!gtk_text_buffer_get_modified(Shiki_CURRENT_TEXT_BUFFER)) return;

  /* まだファイル名が設定されていなかったら，ダイアログを開いて入力させる */
  if(strcmp("*scratch*", Shiki_CURRENT_FILENAME) == 0) {
    const gchar *filename = Shiki_file_name_dialog("Save File As ...");
    if(!filename) return;
    if(!save_text_buffer(filename, Shiki_CURRENT_TEXT_BUFFER)) return;
    gtk_notebook_set_tab_label_text(Shiki_EDITOR_NOTEBOOK, GTK_WIDGET(Shiki_CURRENT_TAB), filename);
    gtk_window_set_title (GTK_WINDOW(Shiki_EDITOR_WINDOW), filename);
  } else
    save_text_buffer(Shiki_CURRENT_TAB_TITLE, Shiki_CURRENT_TEXT_BUFFER);
}

/* 現在表示されているページの内容をファイルに別名保存 */
static void save_file_as() {
  const gchar *filename = Shiki_file_name_dialog("Save File As ...");

  if(!filename) return;
  if(!save_text_buffer(filename, Shiki_CURRENT_TEXT_BUFFER)) return;

  gtk_notebook_set_tab_label_text(Shiki_EDITOR_NOTEBOOK, GTK_WIDGET(Shiki_CURRENT_TAB), filename);
  gtk_window_set_title (GTK_WINDOW (Shiki_EDITOR_WINDOW), filename);
}

/* 緑ボタンのハンドリング．バッファの選択範囲の S 式をロード (逐次評価) */
static void load_region_by_gauche() {

  GtkTextIter start, end, p;
  gchar *code;
  gtk_text_buffer_get_end_iter(Shiki_CURRENT_TEXT_BUFFER, &p);
  gtk_text_buffer_insert(Shiki_CURRENT_TEXT_BUFFER, &p, "\n\n", -1);

  /* マウスで選択されている範囲の文字列を取得 */
  if(gtk_text_buffer_get_selection_bounds(Shiki_CURRENT_TEXT_BUFFER, &start, &end)) {
    code = gtk_text_buffer_get_text(Shiki_CURRENT_TEXT_BUFFER, &start, &end, FALSE);
    gtk_text_buffer_insert(Shiki_CURRENT_TEXT_BUFFER, &p, load_cstring_by_gauche(code), -1);
    g_free(code);
  }
}

/* カーソル位置のネストレベルを返す */
static gint get_parent_nest_level_at_cursor(GtkTextBuffer *buffer) {
  gint nest_level = 0;
  GtkTextIter start, end;
  gtk_text_buffer_get_start_iter(buffer, &start);
  if(gtk_text_iter_get_char(&start) == '(') nest_level++;

  /* カーソルの位置 (= end) を取得 */
  gtk_text_buffer_get_iter_at_mark(buffer,&end, gtk_text_buffer_get_insert(buffer));

  while(1) {
    /* end まで '(' か ')' を探して見つからなかったら終了 */
    if(!gtk_text_iter_forward_find_char(&start, is_kakko_or_kokka, NULL, &end))
      return nest_level;

    if(gtk_text_iter_get_char(&start) == '(')
      nest_level++;
    else 
      nest_level--;
  }
}

/* ページのタブと境界線を on/off */
static void tabsborder_on_off(GtkButton *button, GtkNotebook *notebook) {
  gint tval = FALSE;
  gint bval = FALSE;
  if(notebook->show_tabs == FALSE)
    tval = TRUE; 
  if(notebook->show_border == FALSE)
    bval = TRUE;

  gtk_notebook_set_show_tabs(notebook, tval);
  gtk_notebook_set_show_border(notebook, bval);
}

/* タブの位置を調整 */
static void rotate_tab_position(GtkButton *button, GtkNotebook *notebook ) {
  gtk_notebook_set_tab_pos(notebook, (notebook->tab_pos + 1) % 4);
}

/* キーが離された */
static gboolean signal_key_release_handler (GtkWidget *notebook, GdkEventKey *event, gpointer contextid) {
  static gint metakey_pressed = 0;
  static gint controlx_pressed = 0;

  if(event->keyval == GDK_parenright && event->state & GDK_SHIFT_MASK) {
    GtkTextIter start, end;

    /* カーソルの位置を取得 */
    gtk_text_buffer_get_iter_at_mark(Shiki_CURRENT_TEXT_BUFFER, &end, gtk_text_buffer_get_insert(Shiki_CURRENT_TEXT_BUFFER));

    start = end;
    gtk_text_iter_backward_char(&start);

    /* カーソル位置の前にある S 式を切り出す */
    if(!search_last_sexp_kakko(&start)) return FALSE;

    gtk_text_buffer_apply_tag_by_name(Shiki_CURRENT_TEXT_BUFFER, "parent_emphasis_background", &start, &end);
  }

  /* 改行されるたびに，自動的に括弧のネストの深さに応じた数のスペース (インデント) が行頭に入る */
  if(event->keyval == GDK_Return) {
    gint indentWidth = get_parent_nest_level_at_cursor(Shiki_CURRENT_TEXT_BUFFER) * SCM_INT_VALUE(Scm_GlobalVariableRef(SCM_MODULE(Shiki_CURRENT_BUFFER_ENV), SCM_SYMBOL(SCM_INTERN("*indent-width*")), 0)), i;
    static gchar *indent = " ";
    for(i = 0; i < indentWidth; i++)
      gtk_text_buffer_insert_at_cursor(Shiki_CURRENT_TEXT_BUFFER, indent, -1);
  }

  /* C-x */
  if(event->keyval == GDK_x && event->state & GDK_CONTROL_MASK) {
    controlx_pressed++;
    gtk_statusbar_push(GTK_STATUSBAR(Shiki_EDITOR_STATUSBAR), GPOINTER_TO_INT(contextid), "C-x -");
  } else if(event->state & GDK_CONTROL_MASK) {

    if(controlx_pressed > 0) {
      switch(event->keyval) {
        case GDK_c :/* C-x C-c : 終了 */
          gtk_statusbar_push(GTK_STATUSBAR(Shiki_EDITOR_STATUSBAR), GPOINTER_TO_INT(contextid), "C-c");
          {/* "delete-event" を発生させる．ウィンドウの × ボタンが押されたのと同じ */
            GdkEvent ev;

            ev.any.type = GDK_DELETE;
            ev.any.window = Shiki_EDITOR_WINDOW->window;
            ev.any.send_event = FALSE;
            gdk_event_put (&ev);
          }
          break;

        case GDK_f : /* C-x C-f : ファイル開く */
          gtk_statusbar_push(GTK_STATUSBAR(Shiki_EDITOR_STATUSBAR), GPOINTER_TO_INT(contextid), "C-f");
          Shiki_open_file_dialog();
          break;

        case GDK_s : /* C-x C-s : ファイル保存 */
          gtk_statusbar_push(GTK_STATUSBAR(Shiki_EDITOR_STATUSBAR), GPOINTER_TO_INT(contextid), "C-s");
          save_file();    
          break;

        case GDK_w : /* C-x C-w : ファイル別名保存 */
          gtk_statusbar_push(GTK_STATUSBAR(Shiki_EDITOR_STATUSBAR), GPOINTER_TO_INT(contextid), "C-w");
          save_file_as();    
          break;
      }
      controlx_pressed = 0;
    }

    switch(event->keyval) {
      case GDK_g :/* C-g : キャンセル */
        metakey_pressed = 0;
        controlx_pressed = 0;

        gtk_statusbar_push(GTK_STATUSBAR(Shiki_EDITOR_STATUSBAR), GPOINTER_TO_INT(contextid), "Quit");
        break;
    }

  }
  return FALSE;
}
static void open_online_help() {
  GtkTextIter p;
  Shiki_new_buffer_create(g_strdup("*help*"));
  gtk_text_buffer_set_text(Shiki_CURRENT_TEXT_BUFFER,
      "コマンドラインからの起動方法\n"
      "$ ./shiki [file1 file2 ....]\n\n"
      "[フォルダのアイコン]             ファイルを開く (C-x C-f)\n"
      "[フロッピーディスクのアイコン]   バッファを保存 (C-x C-s)\n"
      "[フロッピーとえんぴつのアイコン] バッファを別名保存 (C-x C-w)\n"
      "[ギアのアイコン]                 選択領域を gauche にロード\n"
      "[新規アイコン]                   新規バッファを開く (C-t)\n"
      "[左向き矢印アイコン]             やり直し (Undo) (C-\\)\n"
      "[右向き矢印アイコン]             再試行   (Redo) (C-_)\n"
      "[ルーペ]                         検索\n"
      "[ルーペとペン]                   置換\n"
      "[ゴミ箱アイコン]                 バッファをクリア\n"
      "\n"
      "(注 : 通常，バッファの内容が失われる恐れがあるときには警告しますが，クリアだけは警告をしません．もし間違って消してしまった場合には慌てずに Undo してください)\n"
      "\n"
      "[× アイコン]                     バッファを閉じる (C-k)\n"
      "[A アイコン]                     フォントの選択\n"
      "[顔みたいなアイコン]             Scheme ファイルのロード\n"
      "[緑っぽい，変換アイコン]         バッファの内容を全てロード\n"
      "[Abc]                            シンタックスハイライティング\n"
      "[ヘルプ (?) アイコン]            このヘルプ画面を表示します\n"
      "[A]                              フォントの選択\n"
      "[マーキング (ちょん) のアイコン] タブの on/off\n"
      "[工具 (スパナ) のアイコン]       タブの位置の調整\n"
      "[info アイコン]                  このアプリケーションについての情報\n"
      "\n"
      "C-f : → に移動 (forward)\n"
      "C-b : ← に移動 (backward)\n"
      "C-n : ↓ に移動 (next line)\n"
      "C-p : ↑ に移動 (previous line)\n"
      "\n"
      "C-h : バックスペース\n"
      "C-w : カット\n"
      "C-y : ヤンク (ペースト)\n"
      "C-\\ : Undo\n"
      "C-_ : Redo\n"
      "\n"
      "C-e : カーソル後ろの S 式を評価 (eval-expression)\n"
      "C-j : カーソル手前の S 式を評価 (eval-last-sexp)\n"
      "(emacs/xyzzy の *scratch* バッファと同じ)\n"
      "\n"
      "C-M-@ : カーソル以降の S 式を選択 (mark-sexp)\n"
      "C-M-SPC : カーソル手前の S 式を選択 (mark-last-sexp)\n"
      "C-x C-c : 終了．ウィンドウの × ボタンを押したのと同じ\n"
      , -1);
  gtk_text_buffer_set_modified(Shiki_CURRENT_TEXT_BUFFER, FALSE);
  /* カーソル位置を先頭に */
  gtk_text_buffer_get_start_iter(Shiki_CURRENT_TEXT_BUFFER, &p);
  gtk_text_buffer_place_cursor(Shiki_CURRENT_TEXT_BUFFER, &p);    
  gtk_text_view_set_editable(Shiki_CURRENT_TEXT_VIEW, FALSE);
}

/* エディタの編集画面の初期化 */
static void Shiki_editor_window_init(int argc, char **argv) {
  GtkWidget *vbox, *toolbar, *modeline_bg = gtk_event_box_new();
  GtkToolItem *icon;
  GtkIconSize iconsize;
  GtkTooltips *toolbar_tips = gtk_tooltips_new();
  /* 「開く」「保存」「別名保存」「選択領域を実行」アイコン */
  GtkToolItem *oicon, *sicon, *saicon, *eicon;

  gint contextid, i;

  /* ハイライティングキーワードのハッシュテーブルを初期化 */
  keywords_hash = g_hash_table_new(g_str_hash, g_str_equal);
  i = 0;
  while(R5RS_keywords[i] != NULL)
    g_hash_table_insert(keywords_hash, R5RS_keywords[i++], GINT_TO_POINTER(R5RS_KEYWORD_COLOR));
  i = 0;
  while(R5RS_functions[i] != NULL)
    g_hash_table_insert(keywords_hash, R5RS_functions[i++], GINT_TO_POINTER(R5RS_FUNCTION_COLOR));

  /* 窓を作る */
  Shiki_EDITOR_WINDOW = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  g_signal_connect(G_OBJECT(Shiki_EDITOR_WINDOW), "destroy", G_CALLBACK(gtk_main_quit), NULL);

  /* システムのデフォルトクリップボードを取得 */
  Shiki_EDITOR_CLIPBOARD = gtk_clipboard_get(GDK_SELECTION_PRIMARY);

  /* パッキングボックスを作る */
  vbox = gtk_vbox_new(FALSE, 0);
  /* ツールバー作る */
  toolbar = gtk_toolbar_new();
  gtk_box_pack_start(GTK_BOX(vbox), toolbar, FALSE, FALSE, 0);

  Shiki_EDITOR_NOTEBOOK = GTK_NOTEBOOK(gtk_notebook_new());
  g_signal_connect(G_OBJECT(Shiki_EDITOR_NOTEBOOK), "switch-page", GTK_SIGNAL_FUNC(switch_tabpage_handler), NULL);

  /* ツールバーに付けるアイコンの設定 */
  gtk_toolbar_set_style(GTK_TOOLBAR (toolbar), GTK_TOOLBAR_ICONS);
  iconsize = gtk_toolbar_get_icon_size (GTK_TOOLBAR (toolbar));

  /* アイコン作る */

  /* ファイル開く */
  oicon = gtk_tool_button_new(gtk_image_new_from_stock (GTK_STOCK_OPEN, iconsize), "");
  /* 「開く」ボタンにファイルを読み込むアクションを関連付ける */
  g_signal_connect(G_OBJECT(oicon), "clicked", G_CALLBACK(Shiki_open_file_dialog), NULL);
  gtk_container_add(GTK_CONTAINER (toolbar), GTK_WIDGET(oicon));
  gtk_tool_item_set_tooltip(oicon, toolbar_tips, "ファイルを開きます",
      "新しいバッファを開いて，そこにファイルの内容を展開します．");

  /* バッファ保存 */
  sicon = gtk_tool_button_new(gtk_image_new_from_stock (GTK_STOCK_SAVE, iconsize), "");
  /* 「保存」ボタンにファイルを書き出すアクションを関連付ける */
  g_signal_connect(G_OBJECT(sicon), "clicked", G_CALLBACK(save_file), NULL);
  gtk_container_add (GTK_CONTAINER (toolbar), GTK_WIDGET(sicon));      
  gtk_tool_item_set_tooltip(sicon, toolbar_tips, "バッファを保存します",
      "バッファにファイル名が設定されていない場合には，ダイアログを開いてファイル名を取得します");

  /* バッファを別名保存 */
  saicon = gtk_tool_button_new(gtk_image_new_from_stock (GTK_STOCK_SAVE_AS, iconsize), "");
  /* 「別名保存」ボタンに別名のファイル開いて内容を書き出すアクションを関連付ける */
  g_signal_connect(G_OBJECT(saicon), "clicked", G_CALLBACK(save_file_as), NULL);
  gtk_container_add (GTK_CONTAINER (toolbar), GTK_WIDGET(saicon));
  gtk_tool_item_set_tooltip(saicon, toolbar_tips, "バッファを別名保存します",
      "");

  /* バッファ実行 */
  eicon = gtk_tool_button_new(gtk_image_new_from_stock (GTK_STOCK_EXECUTE, iconsize), "");
  /* 「選択範囲をロード」ボタンに libgauche を関連付ける */
  g_signal_connect(G_OBJECT(eicon), "clicked", G_CALLBACK(load_region_by_gauche), NULL);
  gtk_container_add (GTK_CONTAINER (toolbar), GTK_WIDGET(eicon));
  gtk_tool_item_set_tooltip(eicon, toolbar_tips, "選択範囲の S 式をロードします (load-region-lisp)",
      "Scheme (gauche) で評価できる S 式を評価します．");

  gtk_container_add(GTK_CONTAINER(Shiki_EDITOR_WINDOW), vbox);
  gtk_container_add(GTK_CONTAINER(vbox), GTK_WIDGET(Shiki_EDITOR_NOTEBOOK));

  icon = gtk_tool_button_new(gtk_image_new_from_stock (GTK_STOCK_NEW, iconsize), "");
  g_signal_connect(G_OBJECT(icon), "clicked", G_CALLBACK(append_default_tabpage_handler), NULL);
  gtk_container_add(GTK_CONTAINER (toolbar), GTK_WIDGET(icon));
  gtk_tool_item_set_tooltip(icon, toolbar_tips, "新規バッファを開きます", "");

  icon = gtk_tool_button_new(gtk_image_new_from_stock (GTK_STOCK_UNDO, iconsize), "");
  g_signal_connect(G_OBJECT(icon), "clicked", G_CALLBACK(Shiki_undo), NULL);
  gtk_container_add(GTK_CONTAINER (toolbar), GTK_WIDGET(icon));
  gtk_tool_item_set_tooltip(icon, toolbar_tips, "Undo","");

  icon = gtk_tool_button_new(gtk_image_new_from_stock (GTK_STOCK_REDO, iconsize), "");
  g_signal_connect(G_OBJECT(icon), "clicked", G_CALLBACK(Shiki_redo), NULL);
  gtk_container_add(GTK_CONTAINER (toolbar), GTK_WIDGET(icon));
  gtk_tool_item_set_tooltip(icon, toolbar_tips, "Redo", "");

  icon = gtk_tool_button_new(gtk_image_new_from_stock (GTK_STOCK_FIND, iconsize), "");
  g_signal_connect(G_OBJECT(icon), "clicked", G_CALLBACK(Shiki_search_buffer), NULL);
  gtk_container_add(GTK_CONTAINER (toolbar), GTK_WIDGET(icon));
  gtk_tool_item_set_tooltip(icon, toolbar_tips, "検索", "");


  icon = gtk_tool_button_new(gtk_image_new_from_stock (GTK_STOCK_FIND_AND_REPLACE, iconsize), "");
  g_signal_connect(G_OBJECT(icon), "clicked", G_CALLBACK(Shiki_replace_buffer), NULL);
  gtk_container_add(GTK_CONTAINER (toolbar), GTK_WIDGET(icon));
  gtk_tool_item_set_tooltip(icon, toolbar_tips, "置換", "");

  icon = gtk_tool_button_new(gtk_image_new_from_stock (GTK_STOCK_REFRESH, iconsize), "");
  g_signal_connect(G_OBJECT(icon), "clicked", G_CALLBACK(reload_buffer), NULL);
  gtk_container_add(GTK_CONTAINER (toolbar), GTK_WIDGET(icon));
  gtk_tool_item_set_tooltip(icon, toolbar_tips, "リロード", "環境はクリアされます");

  icon = gtk_tool_button_new(gtk_image_new_from_stock (GTK_STOCK_DELETE, iconsize), "");
  g_signal_connect(G_OBJECT(icon), "clicked", G_CALLBACK(clear_current_buffer_handler), NULL);
  gtk_container_add(GTK_CONTAINER (toolbar), GTK_WIDGET(icon));
  gtk_tool_item_set_tooltip(icon, toolbar_tips, "バッファの内容を全消去",
      "まだ内容が保存されていない場合でも警告しません");

  icon = gtk_tool_button_new(gtk_image_new_from_stock (GTK_STOCK_CLOSE, iconsize), "");
  g_signal_connect(G_OBJECT(icon), "clicked", G_CALLBACK(kill_buffer_handler), NULL);
  gtk_container_add(GTK_CONTAINER (toolbar), GTK_WIDGET(icon));
  gtk_tool_item_set_tooltip(icon, toolbar_tips, "バッファを閉じます",
      "まだ内容が保存されていない場合は警告します");

  icon = gtk_tool_button_new(gtk_image_new_from_stock (GTK_STOCK_CONNECT, iconsize), "");
  g_signal_connect(G_OBJECT(icon), "clicked", G_CALLBACK(load_scheme_file_by_gauche), NULL);
  gtk_container_add(GTK_CONTAINER (toolbar), GTK_WIDGET(icon));
  gtk_tool_item_set_tooltip(icon, toolbar_tips, "Scheme ファイルをロード", "");

  icon = gtk_tool_button_new(gtk_image_new_from_stock (GTK_STOCK_CONVERT, iconsize), "");
  g_signal_connect(G_OBJECT(icon), "clicked", G_CALLBACK(load_buffer_by_gauche), G_OBJECT(Shiki_EDITOR_NOTEBOOK));
  gtk_container_add(GTK_CONTAINER (toolbar), GTK_WIDGET(icon));
  gtk_tool_item_set_tooltip(icon, toolbar_tips, "バッファをロード", "");

  icon = gtk_tool_button_new(gtk_image_new_from_stock (GTK_STOCK_SPELL_CHECK, iconsize), "");
  g_signal_connect(G_OBJECT(icon), "clicked", G_CALLBACK(scheme_keyword_highlighting_current_buffer), NULL);
  gtk_container_add(GTK_CONTAINER (toolbar), GTK_WIDGET(icon));
  gtk_tool_item_set_tooltip(icon, toolbar_tips, "シンタックスハイライティング", "");


  icon = gtk_tool_button_new(gtk_image_new_from_stock (GTK_STOCK_DIALOG_QUESTION, iconsize), "");
  g_signal_connect(G_OBJECT(icon), "clicked", G_CALLBACK(open_online_help), NULL);
  gtk_container_add(GTK_CONTAINER (toolbar), GTK_WIDGET(icon));
  gtk_tool_item_set_tooltip(icon, toolbar_tips, "ヘルプ", "");

  icon = gtk_tool_button_new(gtk_image_new_from_stock (GTK_STOCK_BOLD, iconsize), "");
  g_signal_connect(G_OBJECT(icon), "clicked", G_CALLBACK(select_font), NULL);
  gtk_container_add(GTK_CONTAINER (toolbar), GTK_WIDGET(icon));
  gtk_tool_item_set_tooltip(icon, toolbar_tips, "フォントの変更", "");

  icon = gtk_tool_button_new(gtk_image_new_from_stock (GTK_STOCK_APPLY, iconsize), "");
  g_signal_connect(G_OBJECT(icon), "clicked", G_CALLBACK(tabsborder_on_off), G_OBJECT(Shiki_EDITOR_NOTEBOOK));
  gtk_container_add(GTK_CONTAINER (toolbar), GTK_WIDGET(icon));
  gtk_tool_item_set_tooltip(icon, toolbar_tips, "タブの on/off", "");

  icon = gtk_tool_button_new(gtk_image_new_from_stock (GTK_STOCK_PREFERENCES, iconsize), "");
  g_signal_connect(G_OBJECT(icon), "clicked", G_CALLBACK(rotate_tab_position), G_OBJECT(Shiki_EDITOR_NOTEBOOK));
  gtk_container_add(GTK_CONTAINER (toolbar), GTK_WIDGET(icon));
  gtk_tool_item_set_tooltip(icon, toolbar_tips, "タブ位置の設定", "");

  icon = gtk_tool_button_new(gtk_image_new_from_stock (GTK_STOCK_DIALOG_INFO, iconsize), "");
  g_signal_connect(G_OBJECT(icon), "clicked", G_CALLBACK(about_this_application), NULL);
  gtk_container_add(GTK_CONTAINER (toolbar), GTK_WIDGET(icon));
  gtk_tool_item_set_tooltip(icon, toolbar_tips, "このアプリケーションについて", "");

  /* モードライン作る */
  Shiki_EDITOR_MODELINE_LABEL = gtk_label_new(NULL);
  gtk_container_add(GTK_CONTAINER (modeline_bg), Shiki_EDITOR_MODELINE_LABEL);

  gdk_color_parse("black", &COLOR_BLACK);
  gdk_color_parse("green", &COLOR_GREEN);

  gtk_widget_modify_fg(Shiki_EDITOR_MODELINE_LABEL, GTK_STATE_NORMAL, &COLOR_GREEN);
  gtk_widget_modify_bg(modeline_bg, GTK_STATE_NORMAL, &COLOR_BLACK);

  gtk_box_pack_start(GTK_BOX(vbox), modeline_bg, TRUE, TRUE, 0);

  /* C-x C-s などの状態を表示するステータスバーをウィンドウボトムに追加 */
  Shiki_EDITOR_STATUSBAR = gtk_statusbar_new();
  gtk_box_pack_start(GTK_BOX(vbox), Shiki_EDITOR_STATUSBAR, TRUE, TRUE, 0);
  contextid = gtk_statusbar_get_context_id(GTK_STATUSBAR(Shiki_EDITOR_STATUSBAR), "");

  /* キーバインドのハンドリングを登録 */
  g_signal_connect(G_OBJECT(Shiki_EDITOR_NOTEBOOK), "key-press-event", G_CALLBACK (signal_key_press_handler), GINT_TO_POINTER(contextid));
  g_signal_connect(G_OBJECT(Shiki_EDITOR_NOTEBOOK), "key-release-event", G_CALLBACK (signal_key_release_handler), GINT_TO_POINTER(contextid));

  /* 引数に指定されたファイルを開く */
  if(argc >= 2) {
    int i;
    for(i = 1; i < argc; i++)
      Shiki_create_file_buffer(argv[i]);
  } else /* 指定されてなければ，デフォルトのページを追加 */
    open_online_help(Shiki_EDITOR_NOTEBOOK);

  gtk_widget_grab_focus(GTK_WIDGET(Shiki_EDITOR_NOTEBOOK));
  gtk_widget_show_all(Shiki_EDITOR_WINDOW);
}

ScmClass *ShikiBufferClass = NULL;

static void buffer_print(ScmObj obj, ScmPort *out, ScmWriteContext *ctx) {
  GtkTextBuffer *b = SHIKI_BUFFER_UNBOX(obj);
  GList *l  = g_list_find_custom(Shiki_EDITOR_BUFFER_LIST, b, compBuffer);
  if(l)
    Scm_Printf(out, "#<buffer: %s>", ((ShikiBuffer *)(l->data))->name);
  else
    Scm_Printf(out, "#<deleted buffer: %p>", b);
}

static void buffer_cleanup(ScmObj obj)
{
  g_object_unref(SHIKI_BUFFER_UNBOX(obj));
}

int main(int argc, char *argv[]) {
  /* 初期化してメインループへ */
  Shiki_EDITOR_DEFAULT_LOCALE = g_locale_to_utf8(gtk_set_locale(), -1, NULL, NULL, NULL);
  gtk_init(&argc, &argv);
  GC_INIT(); Scm_Init(GAUCHE_SIGNATURE);
  Scm_Load("gauche-init.scm", 0);
  ShikiBufferClass = Scm_MakeForeignPointerClass(SCM_CURRENT_MODULE(),
      "<buffer>", buffer_print, buffer_cleanup,
      SCM_FOREIGN_POINTER_KEEP_IDENTITY
      |
      SCM_FOREIGN_POINTER_MAP_NULL);
  Shiki_editor_window_init(argc, argv);
  gtk_main();
  Scm_Exit(0);
  return 0;
}
