/* vim: set encoding=utf8:
 *
 * buffer.c
 *
 * This file is part of Shiki.
 *
 * Copyright(C)2006 WAKATSUKI toshihiro
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * $Id: buffer.c,v 1.10 2006/11/29 05:08:46 aloha Exp $
 */
#include"shiki.h"

ScmClass *ShikiBufferClass;
extern void Scm_Init_xyzzylisp(ScmModule *module);

/* GtkTextBuffer から，対応する ShikiBuffer 構造体を逆引き */
static gint compBuffer(gconstpointer a, gconstpointer b) {
  return ((ShikiBuffer *)a)->text_buffer == b ? 0 : b - a;
}

static GList *get_ShikiBufferListElement_By_GtkTextBuffer(GtkTextBuffer *b) {
  return g_list_find_custom(Shiki_EDITOR_BUFFER_LIST, b, compBuffer);
}

static void buffer_print(ScmObj obj, ScmPort *out, ScmWriteContext *ctx) {
  GtkTextBuffer *b = SHIKI_BUFFER_UNBOX(obj);
  GList *l  = g_list_find_custom(Shiki_EDITOR_BUFFER_LIST, b, compBuffer);
    if(l)
      Scm_Printf(out, "#<buffer: %s>", ((ShikiBuffer *)(l->data))->name);
    else
      Scm_Printf(out, "#<deleted buffer: %p>", b);
}

static void buffer_cleanup(ScmObj obj)
{
  g_object_unref(SHIKI_BUFFER_UNBOX(obj));
}

/* バッファがまだ保存されていないのに本当に終了するのかを尋ねる */
static gboolean delete_event_handler(GtkWidget *widget, GdkEvent *event, GtkTextBuffer *buffer){
  /* delete-event が発行された際，FALSE を返したらバッファは破棄される */
  return Shiki_need_buffer_save_p(buffer) && !Shiki_yes_or_no_p("バッファは変更されています．変更内容を破棄しますか ?");
}

/* バッファにテキストが挿入された */
static void insert_text_handler(GtkTextBuffer *buffer, GtkTextIter *iter, gchar *str, gint len) {
  /* Undo のための情報を記録 */
  ShikiUndoInfo *undoInfo = g_malloc(sizeof(ShikiUndoInfo));
  g_return_if_fail(undoInfo != NULL);
  undoInfo->action = SHIKI_UNDO_INSERT;
  undoInfo->str    = g_strdup(str);
  undoInfo->strlen = len;
  undoInfo->start  = gtk_text_iter_get_offset(iter);
  undoInfo->end    = undoInfo->start + undoInfo->strlen;
  Shiki_CURRENT_UNDO_INFO_LIST = g_list_prepend(Shiki_CURRENT_UNDO_INFO_LIST, undoInfo);
}

/* バッファの状態に合わせてモードラインを変更 */
static void update_modeline_label() {
  static gchar label[1024];
  GtkTextIter p;
  gtk_text_buffer_get_iter_at_mark(Shiki_CURRENT_TEXT_BUFFER, &p, gtk_text_buffer_get_insert(Shiki_CURRENT_TEXT_BUFFER));

  g_snprintf(label, 1024, "--%s- %-10s (Gauche Interaction) [%s]     L%d:%d    ",
      gtk_text_buffer_get_modified(Shiki_CURRENT_TEXT_BUFFER) ? "**" : "--",
      Shiki_CURRENT_BASENAME,
      Shiki_CURRENT_CES,
      gtk_text_iter_get_line(&p) + 1,
      gtk_text_iter_get_line_offset (&p) + 1);
  gtk_label_set_text(GTK_LABEL(Shiki_EDITOR_MODELINE_LABEL), label);
}

/* バッファからテキストが消去された */
static void delete_range_handler(GtkTextBuffer *buffer, GtkTextIter *start, GtkTextIter *end) {
  /* Undo のための情報を記録 */
  ShikiUndoInfo *undoInfo = g_malloc(sizeof(ShikiUndoInfo));
  g_return_if_fail(undoInfo != NULL);
  undoInfo->action = SHIKI_UNDO_DELETE;
  undoInfo->str    = gtk_text_buffer_get_text(buffer, start, end, FALSE);
  undoInfo->start  = gtk_text_iter_get_offset(start);
  undoInfo->end    = gtk_text_iter_get_offset(end);
  undoInfo->strlen = end - start;
  Shiki_CURRENT_UNDO_INFO_LIST = g_list_prepend(Shiki_CURRENT_UNDO_INFO_LIST, undoInfo);
}

/* ノートブックにタブとページ (バッファ) を追加 */
GtkTextBuffer *Shiki_new_buffer_create(gchar *filename) {
  /*-------------------- 新しいタブを作る ----------------------------------*/
  /* ShikiBuffer には，タブに関連する情報が全て保持されている */
  ShikiBuffer *tabinfo  = g_malloc(sizeof(ShikiBuffer));
  tabinfo->locale        = "Gtk Default (utf8)";
  tabinfo->undoInfoList  = NULL;
  tabinfo->filename      = filename;
  tabinfo->name      = g_path_get_basename(filename);
  tabinfo->tabpage_label = g_strndup(tabinfo->name, 10);
  tabinfo->env           = Scm_MakeModule(NULL, FALSE);
  
  ShikiBufferClass = Scm_MakeForeignPointerClass(SCM_MODULE(tabinfo->env),
                         "<buffer>", buffer_print, buffer_cleanup,
                         SCM_FOREIGN_POINTER_KEEP_IDENTITY
                         |
                         SCM_FOREIGN_POINTER_MAP_NULL);

  /* xyzzy lisp 関数を登録 */
  Scm_Init_xyzzylisp(SCM_MODULE(tabinfo->env));

  /* スクロールウィンドウ (タブの中身の大外) を作る */
  tabinfo->tabpage = GTK_SCROLLED_WINDOW(gtk_scrolled_window_new(NULL, NULL));
  gtk_scrolled_window_set_policy (tabinfo->tabpage, GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

  /* 枠の中に格納するテキストビューワと，テキストバッファを作る */
  tabinfo->text_view = GTK_TEXT_VIEW(gtk_text_view_new());
  gtk_text_view_set_wrap_mode(tabinfo->text_view, GTK_WRAP_WORD);
  tabinfo->text_buffer = gtk_text_view_get_buffer(tabinfo->text_view);
  gtk_widget_set_size_request(GTK_WIDGET(tabinfo->text_view), 680, 700);

  gtk_container_add(GTK_CONTAINER(tabinfo->tabpage), GTK_WIDGET(tabinfo->text_view));
  g_signal_connect(tabinfo->text_buffer, "mark_set", G_CALLBACK(update_modeline_label), tabinfo->text_view);
  g_signal_connect(tabinfo->text_buffer, "insert-text", G_CALLBACK(insert_text_handler), NULL);
  g_signal_connect(tabinfo->text_buffer, "delete-range", G_CALLBACK(delete_range_handler), NULL);

  /* タブを削除する際，デリートハンドラを削除しておかないと警告が出るから */
  tabinfo->delete_handler_id = g_signal_connect(Shiki_EDITOR_WINDOW, "delete_event", G_CALLBACK(delete_event_handler), tabinfo->text_buffer);

  /* 様々な初期化処理 */

  /* 括弧の強調表示のためのタグを作る */
  gtk_text_buffer_create_tag(tabinfo->text_buffer, "parent_emphasis_background", "background", "green", NULL);

  /* キーワードハイライティング用 */
  gtk_text_buffer_create_tag(tabinfo->text_buffer, "keyword_highlighting", "foreground", "blue", NULL);
  /* 関数 */
  gtk_text_buffer_create_tag(tabinfo->text_buffer, "function_highlighting", "foreground", "red", NULL);
  /* コメント */
  gtk_text_buffer_create_tag (tabinfo->text_buffer, "comment_highlighting", "foreground", "purple", NULL);
  /* 文字列 */
  gtk_text_buffer_create_tag (tabinfo->text_buffer, "string_highlighting", "foreground", "orange", NULL);
  /* タブをノートブックに登録する */
  gtk_notebook_append_page(Shiki_EDITOR_NOTEBOOK, GTK_WIDGET(tabinfo->tabpage), gtk_label_new(tabinfo->tabpage_label));
  /* 対応するタブ情報を大域テーブルに保存しておく */
  Shiki_EDITOR_BUFFER_LIST = g_list_append(Shiki_EDITOR_BUFFER_LIST, tabinfo);

  gtk_widget_show_all(GTK_WIDGET(Shiki_EDITOR_NOTEBOOK));  
  /* 開いたページに移動する */
  gtk_notebook_set_current_page(Shiki_EDITOR_NOTEBOOK, g_list_length(Shiki_EDITOR_BUFFER_LIST) - 1);
  //Shiki_CURRENT_TAB_INFO = tabinfo;
  return tabinfo->text_buffer;
}

void Shiki_create_file_buffer(const gchar *filename) {
  gchar *text, *contents;
  GtkTextIter p;
  gsize br, bw, len;
  GError *err = NULL;

  /* 新しいバッファを開く */
  Shiki_new_buffer_create(g_strdup(filename));
  gtk_window_set_title (GTK_WINDOW (Shiki_EDITOR_WINDOW), filename);

  /* open-input-conversion-port が失敗したならば，とりあえずシステム
   * ロケールに変換してみる．駄目だったら諦めて駄目元で UTF8
   */

  if(g_file_get_contents(filename, &contents, &len, &err)) {
    if(!(text = g_locale_to_utf8(contents, -1, &br, &bw, &err)))
      gtk_text_buffer_set_text(Shiki_CURRENT_TEXT_BUFFER, text, -1);
    else
      gtk_text_buffer_set_text(Shiki_CURRENT_TEXT_BUFFER, contents, -1);
    g_free(contents);
  }
  
  /* バッファ未変更に */
  gtk_text_buffer_set_modified(Shiki_CURRENT_TEXT_BUFFER, FALSE);
  /* カーソル位置を先頭に */
  gtk_text_buffer_get_start_iter(Shiki_CURRENT_TEXT_BUFFER, &p);
  gtk_text_buffer_place_cursor(Shiki_CURRENT_TEXT_BUFFER, &p);
  update_modeline_label();
  gtk_widget_show_all(GTK_WIDGET(Shiki_EDITOR_NOTEBOOK));
}

void Shiki_open_file_dialog() {
  const gchar *filename = Shiki_file_name_dialog("ファイルを開く");

  if(!filename) return;
  Shiki_create_file_buffer(filename);
}

void Shiki_delete_buffer(GtkTextBuffer *buffer) {
  /* バッファリストから，リストの要素，バッファ，バッファの番号を逆引き */
  /* 効率が悪いが，Scheme の世界になるべく Gtk のオブジェクトを持ち込まないため */
  GList *bufListElem = get_ShikiBufferListElement_By_GtkTextBuffer(buffer);
  ShikiBuffer *tabInfo  = bufListElem->data;
  gint bufNum = g_list_position(Shiki_EDITOR_BUFFER_LIST, bufListElem);

  /* タブが 1 つしか残っていなかったら消させない */
  if(g_list_length(Shiki_EDITOR_BUFFER_LIST) == 1)
    return;
  /* デリートハンドラをエディタトップレベルのウィジットから取り除く */
  g_signal_handler_disconnect(Shiki_EDITOR_WINDOW, tabInfo->delete_handler_id);
  Shiki_EDITOR_BUFFER_LIST = g_list_delete_link(Shiki_EDITOR_BUFFER_LIST, bufListElem);  
  gtk_widget_destroy(GTK_WIDGET(tabInfo->tabpage));
  g_free(tabInfo->tabpage_label);
  g_free(tabInfo->name);
  g_free(tabInfo->filename);
  g_free(tabInfo);
  gtk_notebook_remove_page(Shiki_EDITOR_NOTEBOOK, bufNum);
  /* 強制再描画 */
  gtk_widget_queue_draw(GTK_WIDGET(Shiki_EDITOR_NOTEBOOK));
}

GtkTextBuffer *Shiki_find_buffer(const gchar *name) {
  GList *l;
  for(l = Shiki_EDITOR_BUFFER_LIST; l != NULL; l = l->next)
    if(strcmp(((ShikiBuffer *)l->data)->name, name) == 0)
      return ((ShikiBuffer *)l->data)->text_buffer;
  return NULL;
}

  gchar *Shiki_buffer_substring(gint start, gint end) {
    if(start >= end)
      return NULL;
    else {
      GtkTextIter s, e;
      gtk_text_buffer_get_iter_at_offset(Shiki_CURRENT_TEXT_BUFFER, &s, start); 
      gtk_text_buffer_get_iter_at_offset(Shiki_CURRENT_TEXT_BUFFER, &e, end);

      return gtk_text_buffer_get_text(Shiki_CURRENT_TEXT_BUFFER, &s, &e, FALSE);
    }
  }

  void Shiki_delete_region(gint start, gint end) {
    if(start >= end)
      return;
    else {
      GtkTextIter s, e;
      gtk_text_buffer_get_iter_at_offset(Shiki_CURRENT_TEXT_BUFFER, &s, start); 
      gtk_text_buffer_get_iter_at_offset(Shiki_CURRENT_TEXT_BUFFER, &e, end);

      return gtk_text_buffer_delete(Shiki_CURRENT_TEXT_BUFFER, &s, &e);
    }
  }

gint Shiki_point() {
  GtkTextIter p;
  gtk_text_buffer_get_iter_at_mark(Shiki_CURRENT_TEXT_BUFFER,&p, gtk_text_buffer_get_insert(Shiki_CURRENT_TEXT_BUFFER));
  return gtk_text_iter_get_offset(&p);
}

gint Shiki_point_max() {
  GtkTextIter p;
  gtk_text_buffer_get_end_iter(Shiki_CURRENT_TEXT_BUFFER, &p);
  return gtk_text_iter_get_offset(&p);
}

gint Shiki_point_min() {
  return 0;
}

void Shiki_goto_char(gint offset) {
  GtkTextIter p;
  gtk_text_buffer_get_iter_at_offset(Shiki_CURRENT_TEXT_BUFFER, &p, offset);
  gtk_text_buffer_place_cursor(Shiki_CURRENT_TEXT_BUFFER, &p);
}

void Shiki_forward_char() {
  GtkTextIter p;
  gtk_text_buffer_get_iter_at_mark(Shiki_CURRENT_TEXT_BUFFER,&p, gtk_text_buffer_get_insert(Shiki_CURRENT_TEXT_BUFFER));
  gtk_text_iter_forward_char(&p);
  gtk_text_buffer_place_cursor(Shiki_CURRENT_TEXT_BUFFER, &p);
}

void Shiki_backward_char() {
  GtkTextIter p;
  gtk_text_buffer_get_iter_at_mark(Shiki_CURRENT_TEXT_BUFFER,&p, gtk_text_buffer_get_insert(Shiki_CURRENT_TEXT_BUFFER));
  gtk_text_iter_backward_char(&p);
  gtk_text_buffer_place_cursor(Shiki_CURRENT_TEXT_BUFFER, &p);
}

void Shiki_goto_line(gint line) {
  GtkTextIter p;
  gtk_text_buffer_get_iter_at_line(Shiki_CURRENT_TEXT_BUFFER, &p, line);
  gtk_text_buffer_place_cursor(Shiki_CURRENT_TEXT_BUFFER, &p);  
}

void Shiki_goto_bol() {
  GtkTextIter p;
  gtk_text_buffer_get_iter_at_mark(Shiki_CURRENT_TEXT_BUFFER,&p, gtk_text_buffer_get_insert(Shiki_CURRENT_TEXT_BUFFER));
  gtk_text_buffer_get_iter_at_line_offset(Shiki_CURRENT_TEXT_BUFFER, &p, gtk_text_iter_get_line(&p), 0);
  gtk_text_buffer_place_cursor(Shiki_CURRENT_TEXT_BUFFER, &p);
}

void Shiki_goto_eol() {
  GtkTextIter p;
  gtk_text_buffer_get_iter_at_mark(Shiki_CURRENT_TEXT_BUFFER,&p, gtk_text_buffer_get_insert(Shiki_CURRENT_TEXT_BUFFER));
  gtk_text_iter_forward_to_line_end(&p);
  gtk_text_iter_backward_char(&p);
  gtk_text_buffer_place_cursor(Shiki_CURRENT_TEXT_BUFFER, &p);
}

void Shiki_forward_line(gint count) {
  GtkTextIter p;
  gint i;
  gtk_text_buffer_get_iter_at_mark(Shiki_CURRENT_TEXT_BUFFER,&p, gtk_text_buffer_get_insert(Shiki_CURRENT_TEXT_BUFFER));
  
  if(count >= 0) {
    for(i = count; i != 0; i--)
      gtk_text_view_forward_display_line(Shiki_CURRENT_TEXT_VIEW, &p);
  } else {
    for(i = count; i != 0; i++)
      gtk_text_view_backward_display_line(Shiki_CURRENT_TEXT_VIEW, &p);
  }
  gtk_text_buffer_place_cursor(Shiki_CURRENT_TEXT_BUFFER, &p);
}

const char *Shiki_buffer_name(GtkTextBuffer *buffer) {
  GList *l = get_ShikiBufferListElement_By_GtkTextBuffer(buffer);
  if(l)
    return ((ShikiBuffer *)(l->data))->name;
  else
    return NULL;
}

gboolean Shiki_deleted_buffer_p(GtkTextBuffer *buffer) {
  GList *l = get_ShikiBufferListElement_By_GtkTextBuffer(buffer);
  if(l)
    return FALSE;
  else
    return TRUE;
}

GtkTextBuffer *Shiki_get_next_buffer(GtkTextBuffer *buffer) {
  GList *l = get_ShikiBufferListElement_By_GtkTextBuffer(buffer);
  if(l && l->next)
    return ((ShikiBuffer *)(l->next->data))->text_buffer;
  else
    return NULL;
}

GtkTextBuffer *Shiki_get_previous_buffer(GtkTextBuffer *buffer) {
  GList *l = get_ShikiBufferListElement_By_GtkTextBuffer(buffer);
  if(l && l->prev)
    return ((ShikiBuffer *)(l->prev->data))->text_buffer;
  else
    return NULL;
}

ScmObj Shiki_buffer_list() {
  GList *l;
  GtkTextBuffer *b;
  ScmObj bl = SCM_NIL;

  for(l = Shiki_EDITOR_BUFFER_LIST; l != NULL; l = l->next) {
    b= ((ShikiBuffer *)(l->data))->text_buffer;
    bl = Scm_Cons(SHIKI_BUFFER_BOX(g_object_ref(b)), bl);
  }
  return bl;
}

void Shiki_erase_buffer(GtkTextBuffer *buffer) {
  GtkTextIter start, end;
  gtk_text_buffer_get_start_iter(buffer, &start);
  gtk_text_buffer_get_end_iter(buffer, &end);
  gtk_text_buffer_delete(buffer, &start, &end);
}

const gchar *Shiki_file_name_dialog(const gchar *msg) {

  GtkWidget *dialog = gtk_file_selection_new(msg);
  gint resp = gtk_dialog_run(GTK_DIALOG(dialog));
  const gchar *filename = NULL;

  if(resp == GTK_RESPONSE_OK)
    filename = gtk_file_selection_get_filename(GTK_FILE_SELECTION(dialog));

  gtk_widget_destroy(dialog);
  return filename;
}

gboolean Shiki_yes_or_no_p(const gchar *msg) {
  GtkWidget *dialog = gtk_message_dialog_new(GTK_WINDOW(Shiki_EDITOR_WINDOW),
               GTK_DIALOG_DESTROY_WITH_PARENT, GTK_MESSAGE_QUESTION,
               GTK_BUTTONS_YES_NO, msg);
  gint resp;
  gtk_dialog_set_default_response(GTK_DIALOG(dialog), GTK_RESPONSE_YES);
  resp = gtk_dialog_run(GTK_DIALOG(dialog));
  gtk_widget_destroy(dialog);
  if(GTK_RESPONSE_YES == resp)
    return TRUE;
  return FALSE;
}

gboolean Shiki_no_or_yes_p(const gchar *msg) {
  GtkWidget *dialog = gtk_message_dialog_new(GTK_WINDOW(Shiki_EDITOR_WINDOW),
               GTK_DIALOG_DESTROY_WITH_PARENT, GTK_MESSAGE_QUESTION,
               GTK_BUTTONS_YES_NO, msg);
  gint resp;
  gtk_dialog_set_default_response(GTK_DIALOG(dialog), GTK_RESPONSE_NO);  
  resp = gtk_dialog_run(GTK_DIALOG(dialog));
  gtk_widget_destroy(dialog);
  if(GTK_RESPONSE_YES == resp)
    return TRUE;
  return FALSE;
}

gboolean Shiki_need_buffer_save_p(GtkTextBuffer *buffer) {
  return gtk_text_buffer_get_modified(buffer);
}

/* 確認あり */
void Shiki_kill_buffer(GtkTextBuffer *buffer) {
  if(!Shiki_need_buffer_save_p(buffer) || Shiki_yes_or_no_p("バッファは変更されています．変更内容を破棄しますか ?"))
    Shiki_delete_buffer(buffer);
}
