/*
RSSPanelController.m

Author: Makoto Kinoshita

Copyright 2004-2006 The Shiira Project. All rights reserved.

Redistribution and use in source and binary forms, with or without modification, are permitted 
provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice, this list of conditions 
  and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright notice, this list of 
  conditions and the following disclaimer in the documentation and/or other materials provided 
  with the distribution.

THIS SOFTWARE IS PROVIDED BY THE SHIIRA PROJECT ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, 
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE SHIIRA PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
POSSIBILITY OF SUCH DAMAGE.
*/

#import "SRAppController.h"

#import "RSSFeedList.h"
#import "RSSPersistentStack.h"

#import "RSSPanelController.h"

// Frame auto save name
NSString*   RSSPanelFrameAutoSaveName = @"RSSPanelFrameAutoSaveName";

@implementation RSSPanelController

//--------------------------------------------------------------//
#pragma mark -- Initialize --
//--------------------------------------------------------------//

+ (id)sharedInstance
{
    static RSSPanelController*  _sharedInstance = nil;
    if (!_sharedInstance) {
        _sharedInstance = [[RSSPanelController alloc] init];
    }
    
    return _sharedInstance;
}

- (id)init
{
    self = [super initWithWindowNibName:@"RSSPanel"];
    if (!self) {
        return nil;
    }
    
    // Register observer
    NSNotificationCenter*   center;
    center = [NSNotificationCenter defaultCenter];
    [center addObserver:self selector:@selector(rssArticlesRead:) 
            name:RSSArticlesRead object:nil];
    [center addObserver:self selector:@selector(rssPersistentStackRefreshed:) 
            name:RSSPersistentStackRefreshed object:[RSSPersistentStack sharedInstance]];
    
    return self;
}

- (void)awakeFromNib
{
    // Configure window
    NSWindow*   window;
    window = [self window];
    [window setFrameAutosaveName:RSSPanelFrameAutoSaveName];
    
    // Configure RSS table
    [_tableView setTarget:self];
    [_tableView setDoubleAction:@selector(openRSSAction:)];
    
    // Set image text cell
    NSTableColumn*          column;
    NSCell*                 oldCell;
    HMImageTextFieldCell*   cell;
    column = [_tableView tableColumnWithIdentifier:@"rss"];
    oldCell = [column dataCell];
    cell = [[HMImageTextFieldCell alloc] init];
    [cell setFont:[oldCell font]];
    [column setDataCell:cell];
    [cell release];
    
    // Configure menu button
    [_menuButton setDelegate:self];
    
    // Register notification
    NSNotificationCenter*   center;
    center = [NSNotificationCenter defaultCenter];
    //[center addObserver:self selector:@selector(webHistoryUpdated:) 
    //        name:SRWebHistoryUpdated object:nil];
}

- (void)dealloc
{
    // Remove observer
    [[NSNotificationCenter defaultCenter] removeObserver:self];
    
    [super dealloc];
}

//--------------------------------------------------------------//
#pragma mark -- RSS --
//--------------------------------------------------------------//

- (NSArray*)selectedRSSItems
{
    // Get selected objects
    NSArray*    objects;
    objects = [_rssArrayController selectedObjects];
    if (!objects || [objects count] == 0) {
        return nil;
    }
    
    return objects;
}

- (NSMenu*)_contextMenuForView:(id)view 
        event:(NSEvent*)event
{
    NSUserDefaults* defaults;
    defaults = [NSUserDefaults standardUserDefaults];
    
#if 0
    // Get modifier key flag
    unsigned int    modifierFlags;
    unsigned int    cmdFlag, optionFlag, shiftFlag;
    modifierFlags = [event modifierFlags];
    cmdFlag = modifierFlags & NSCommandKeyMask;
    optionFlag = modifierFlags & NSAlternateKeyMask;
    shiftFlag = modifierFlags & NSShiftKeyMask;
    
    // Create array for tags
    NSMutableArray* tags;
    tags = [NSMutableArray array];
    
    // Select history item under the cursor
    NSPoint point;
    int     rowUnderPoint;
    point = [view convertPoint:[event locationInWindow] 
            fromView:nil];
    rowUnderPoint = [view rowAtPoint:point];
    if (![[view selectedRowIndexes] containsIndex:rowUnderPoint]) {
        [view selectRowIndexes:[NSIndexSet indexSetWithIndex:rowUnderPoint] 
                byExtendingSelection:NO];
    }
    
    // Get selected rows
    NSIndexSet* indexSet;
    indexSet = [view selectedRowIndexes];
    
    WebHistoryItem* historyItem = nil;
    NSArray*        historyItems = nil;
    
    // No history is selected
    if ([indexSet count] == 0) {
        // Create tag array
        [tags addObject:[NSNumber numberWithInt:SRShowHistoryShelf]];
    }
    else {
        // Just one history item is selected
        if ([indexSet count] == 1) {
            // Get history item
            historyItem = [[self selectedHistoryItems] objectAtIndex:0];
        }
        else {
            // Get history items
            historyItems = [self selectedHistoryItems];
        }
        
        // Case of one history item
        if (historyItem && [historyItem isKindOfClass:[WebHistoryItem class]]) {
            // Create tag array
            [tags addObject:[NSNumber numberWithInt:SROpenHistoryTag]];
            [tags addObject:[NSNumber numberWithInt:SROpenHistoryInNewWindowTag]];
            [tags addObject:[NSNumber numberWithInt:SROpenHistoryInNewTabTag]];
            [tags addObject:[NSNumber numberWithInt:SROpenHistoryInNewBackgroundWindowTag]];
            [tags addObject:[NSNumber numberWithInt:SROpenHistoryInNewBackgroundTabTag]];
            [tags addObject:[NSNumber numberWithInt:SROpenHistoryInNewWindowTag]];
            
            //[tags addObject:[NSNumber numberWithInt:SRShowHistoryInShelf]];
            [tags addObject:[NSNumber numberWithInt:SRShowHistoryShelf]];
        }
        // Case of one folder
        if (historyItem && [historyItem isKindOfClass:[NSCalendarDate class]]) {
            // Create tag array
            //[tags addObject:[NSNumber numberWithInt:SRShowHistoryInShelf]];
            [tags addObject:[NSNumber numberWithInt:SRShowHistoryShelf]];
        }
        // Case of multiple history items
        if (historyItems) {
            // Check existence of history item
            BOOL            isContainedHistoryItem = NO;
            NSEnumerator*   enumerator;
            WebHistoryItem* tmp;
            enumerator = [historyItems objectEnumerator];
            while (tmp = [enumerator nextObject]) {
                if ([tmp isKindOfClass:[WebHistoryItem class]]) {
                    isContainedHistoryItem = YES;
                    break;
                }
            }
            
            // Create tag array
            if (isContainedHistoryItem) {
                [tags addObject:[NSNumber numberWithInt:SROpenHistoryInTabsTag]];
            }
            [tags addObject:[NSNumber numberWithInt:SRShowHistoryShelf]];
        }
    }
    
    if ([tags count] > 0) {
        // Copy menu
        NSMenu* menu;
        menu = HMCopyMenuWithTags([SRHistoryContextMenu contextMenu], tags, self);
        
        // Set represented object
        if (historyItem) {
            [[menu itemArray] makeObjectsPerformSelector:@selector(setRepresentedObject:) 
                    withObject:historyItem];
        }
        if (historyItems) {
            [[menu itemArray] makeObjectsPerformSelector:@selector(setRepresentedObject:) 
                    withObject:historyItems];
        }
        
        // Set alt menu
        NSMenuItem* altMenuItem;
        if ([defaults boolForKey:SRTabSelectNewTabs]) {
            altMenuItem = [menu itemWithTag:SROpenHistoryInNewBackgroundWindowTag];
            [altMenuItem setKeyEquivalentModifierMask:NSShiftKeyMask];
            [altMenuItem setAlternate:YES];
            
            altMenuItem = [menu itemWithTag:SROpenHistoryInNewBackgroundTabTag];
            [altMenuItem setKeyEquivalentModifierMask:NSShiftKeyMask];
            [altMenuItem setAlternate:YES];
        }
        else {
            altMenuItem = [menu itemWithTag:SROpenHistoryInNewWindowTag];
            [altMenuItem setKeyEquivalentModifierMask:NSShiftKeyMask];
            [altMenuItem setAlternate:YES];
            
            altMenuItem = [menu itemWithTag:SROpenHistoryInNewTabTag];
            [altMenuItem setKeyEquivalentModifierMask:NSShiftKeyMask];
            [altMenuItem setAlternate:YES];
        }
        
        return menu;
    }
#endif
    
    return nil;
}

//--------------------------------------------------------------//
#pragma mark -- Persistent stack --
//--------------------------------------------------------------//

- (NSManagedObjectContext*)managedObjectContext
{
    return [[RSSPersistentStack sharedInstance] managedObjectContext];
}

//--------------------------------------------------------------//
#pragma mark -- Action --
//--------------------------------------------------------------//

#if 0
- (void)showWindow:(id)sender
{
    // Check window
    if ([[self window] isVisible]) {
        return;
    }
    
    // Reset managed object context
    [[self managedObjectContext] reset];
    [_rssArrayController rearrangeObjects];
    [_tableView reloadData];
    
    // Show window
    [super showWindow:sender];
}
#endif

- (void)openRSSAction:(id)sender
{
    // Get selected RSS
    NSArray*    rssItems;
    rssItems = [self selectedRSSItems];
    if (!rssItems || [rssItems count] == 0) {
        return;
    }
    
    // Get item
    NSDictionary*   rssItem;
    rssItem = [rssItems objectAtIndex:0];
    
    NSString*   urlString = nil;
    
    // For feed
    if ([rssItem valueForKey:@"feedURL"]) {
        // Get feed URL
        urlString = [rssItem valueForKey:@"feedURL"];
    }
    
    if (!urlString) {
        return;
    }
    
    // Change scheme
    if ([urlString hasPrefix:@"http://"]) {
        urlString = [NSString stringWithFormat:@"rss://%@", [urlString substringFromIndex:7]];
    }
    
    // Open URL
    WebView*    webView;
    webView = [[[SRAppController sharedInstance] openURLString:urlString] webView];
    
    // Make web view window key window
    NSWindow*   window;
    window = [webView window];
    if (window) {
        [window makeKeyWindow];
    }
}

- (void)showShelfAction:(id)sender
{
    [[SRAppController sharedInstance] openShelf:@"jp.hmdt.shiira.rssshelf"];
}

//--------------------------------------------------------------//
#pragma mark -- NSTableView data source --
//--------------------------------------------------------------//

- (int)numberOfRowsInTableView:(NSTableView*)tableView
{
    return 0;
}

- (id)tableView:(NSTableView*)tableView 
        objectValueForTableColumn:(NSTableColumn*)tableColumn row:(int)rowIndex
{
    return nil;
}

//--------------------------------------------------------------//
#pragma mark -- NSTableView delegate --
//--------------------------------------------------------------//

- (int)_unreadArticlesNumberWithFeedURL:(NSString*)urlString
{
    // Get managed object context
    NSManagedObjectContext* context;
    context = [self managedObjectContext];
    
    // Get feed
    NSFetchRequest* request;
    request = [[[NSFetchRequest alloc] init] autorelease];
    [request setEntity:
            [NSEntityDescription entityForName:@"RSSFeed" inManagedObjectContext:context]];
    [request setPredicate:
            [NSPredicate predicateWithFormat:@"feedURL == %@", urlString]];
    
    NSArray*    items;
    items = [context executeFetchRequest:request error:NULL];
    if ([items count] != 1) {
        return 0;
    }
    
    // Get number of unread articles
    return [[[items objectAtIndex:0] valueForKey:@"numberOfUnreadArticles"] intValue];
}

- (void)tableView:(NSTableView*)tableView 
        willDisplayCell:(id)cell 
        forTableColumn:(NSTableColumn*)column 
        row:(int)index
{
    // RSS table
    if (tableView == _tableView) {
        // Get column identifier
        id  identifier;
        identifier = [column identifier];
        
        // For RSS column
        if ([identifier isEqualToString:@"rss"]) {
            // Get object
            NSArray*    objects;
            objects = [_rssArrayController arrangedObjects];
            if ([objects count] > index) {
                id  object;
                object = [objects objectAtIndex:index];
                
                // Get feed URL
                NSImage*    image = nil;
                NSString*   urlString;
                urlString = [object valueForKey:@"feedURL"];
                if (urlString) {
                    WebIconDatabase*    database;
                    database = [WebIconDatabase sharedIconDatabase];
                    
                    if ([database iconURLForURL:urlString]) {
                        image = [database iconForURL:urlString withSize:NSMakeSize(16, 16)];
                    }
                }
                if (!image) {
                    image = [NSImage imageNamed:@"rssIcon"];
                }
                
                // Set image
                [cell setImage:image];
                
                // Get unread articles number
                int number;
                number = [self _unreadArticlesNumberWithFeedURL:urlString];
                [cell setTextFieldType:HMNumberedTextFieldType];
                [cell setNumber:number];
            }
            
            return;
        }
    }
}

//--------------------------------------------------------------//
#pragma mark -- NSTableViewEx delegate --
//--------------------------------------------------------------//

- (NSMenu*)tableView:(NSTableView*)tableView menuForEvent:(NSEvent*)event
{
    return [self _contextMenuForView:tableView event:event];
}

//--------------------------------------------------------------//
#pragma mark -- HMMenuButton delegate --
//--------------------------------------------------------------//

- (NSMenu*)menuButton:(HMMenuButton*)menuButton menuForEvent:(NSEvent*)event
{
    return [self _contextMenuForView:_tableView event:event];
}

//--------------------------------------------------------------//
#pragma mark -- RSSController notification --
//--------------------------------------------------------------//

- (void)rssArticlesRead:(NSNotification*)notification
{
    // Reload table
    [_tableView reloadData];
}

//--------------------------------------------------------------//
#pragma mark -- RSSPersistentStack notification --
//--------------------------------------------------------------//

- (void)rssPersistentStackRefreshed:(NSNotification*)notification
{
    // Prepare content
    [_rssArrayController prepareContent];
}

@end
