/*
HMImageTextFieldCell.m

Author: Makoto Kinoshita, Kazuki Miura

Copyright 2004-2006 The Shiira Project. All rights reserved.

Redistribution and use in source and binary forms, with or without modification, are permitted 
provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice, this list of conditions 
  and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright notice, this list of 
  conditions and the following disclaimer in the documentation and/or other materials provided 
  with the distribution.

THIS SOFTWARE IS PROVIDED BY THE SHIIRA PROJECT ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, 
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE SHIIRA PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
POSSIBILITY OF SUCH DAMAGE.
*/

#import "HMAppKitEx.h"
#import "HMImageTextFieldCell.h"

static int  HMMarginImage = 4;
static int  HMMarginBetweenImageAndText = 4;
static int  HMMarginNumber = 4;
static int  HMNumberHeight = 14;

static NSImage* _numberBackBlkLeftImage = nil;
static NSImage* _numberBackBlkMiddleImage = nil;
static NSImage* _numberBackBlkRightImage = nil;

static NSDictionary*    _numberAttr = nil;

@interface NSTextFieldCell (Private)
- (NSRect)_focusRingFrameForFrame:(NSRect)frame cellFrame:(NSRect)cellFrame;
- (NSDictionary*)_textAttributes;
@end

@implementation HMImageTextFieldCell

//--------------------------------------------------------------//
#pragma mark -- Initialize --
//--------------------------------------------------------------//

+ (void)load
{
    NSAutoreleasePool*  pool;
    pool = [[NSAutoreleasePool alloc] init];
    
    // Get resources
    if (!_numberBackBlkLeftImage) {
        NSBundle*   bundle;
        bundle = [NSBundle bundleForClass:[self class]];
    
        _numberBackBlkLeftImage = [[NSImage alloc] initWithContentsOfFile:
                [bundle pathForImageResource:@"blkNumBackL"]];
        _numberBackBlkMiddleImage = [[NSImage alloc] initWithContentsOfFile:
                [bundle pathForImageResource:@"blkNumBackM"]];
        _numberBackBlkRightImage = [[NSImage alloc] initWithContentsOfFile:
                [bundle pathForImageResource:@"blkNumBackR"]];
        
        _numberAttr = [[NSDictionary dictionaryWithObjectsAndKeys:
                [NSFont boldSystemFontOfSize:[NSFont smallSystemFontSize]], NSFontAttributeName, 
                [NSColor colorWithCalibratedWhite:0.2f alpha:1.0f], NSForegroundColorAttributeName, 
                nil] retain];
    }
    
    [pool release];
}

- (id)init
{
    self = [super init];
    if (!self) {
        return nil;
    }
    
    // Initialize instance variables
    _padding = HMMarginBetweenImageAndText;
    _focusRingIncludesImage = NO;
    _useBoldForHighlight = NO;
    
    // Configure itself
    [self setWraps:NO];
    
    return self;
}

- (void)dealloc
{
    [_image release], _image = nil;
    
    [super dealloc];
}

- (id)copyWithZone:(NSZone*)zone
{
    HMImageTextFieldCell*   cell;
    cell = (HMImageTextFieldCell*)[super copyWithZone:zone];
    cell->_image = [_image retain];
    
    return cell;
}

//--------------------------------------------------------------//
#pragma mark -- Image management --
//--------------------------------------------------------------//

- (void)setImage:(NSImage*)image {
    if (_image != image) {
        [_image release];
        _image = [image retain];
    }
}

- (NSImage *)image {
    return _image;
}

- (int)paddingBetweenImageAndText
{
    return _padding;
}

- (void)setPaddingBetweenImageAndText:(int)padding
{
    _padding = padding;
}

- (BOOL)focusRingIncludesImage
{
    return _focusRingIncludesImage;
}

- (void)setFocusRingIncludesImage:(BOOL)flag
{
    _focusRingIncludesImage = flag;
}

- (BOOL)focusOnlyTitle
{
	return _focusOnlyTitle;
}

- (void)setFocusOnlyTitle:(BOOL)flag
{
	_focusOnlyTitle = flag;
}

//--------------------------------------------------------------//
#pragma mark -- Text field type --
//--------------------------------------------------------------//

- (int)textFieldType
{
    return _textFieldType;
}

- (void)setTextFieldType:(int)type
{
    _textFieldType = type;
}

- (int)textFieldNumberType
{
    return _textFieldNumberType;
}

- (void)setTextFieldNumberType:(int)type
{
    _textFieldNumberType = type;
}

//--------------------------------------------------------------//
#pragma mark -- Numbered field --
//--------------------------------------------------------------//

- (int)number
{
    return _number;
}

- (void)setNumber:(int)number
{
    _number = number;
}

//--------------------------------------------------------------//
#pragma mark -- Separator --
//--------------------------------------------------------------//

- (BOOL)isSeparator
{
    return _isSeparator;
}

- (void)setSeparator:(BOOL)isSepartor
{
    _isSeparator = isSepartor;
}

//--------------------------------------------------------------//
#pragma mark -- Highlight --
//--------------------------------------------------------------//

- (BOOL)useBoldForHighlight
{
    return _useBoldForHighlight;
}

- (void)setUseBoldForHighlight:(BOOL)flag
{
    _useBoldForHighlight = flag;
}

//--------------------------------------------------------------//
#pragma mark -- Size --
//--------------------------------------------------------------//

- (float)imageWidthWithSpace
{
	if ([self image]) {
		return HMMarginImage + [[self image] size].width + [self paddingBetweenImageAndText];
	}
	
	return 0.0;
}

- (NSSize)cellSize
{
	// Super
	NSSize cellSize;
	cellSize = [super cellSize];
	
	// Add image width
	cellSize.width += [self imageWidthWithSpace];
	
	return cellSize;
}

- (NSSize)cellSizeForBounds:(NSRect)theRect
{
	// Super
	NSSize cellSize;
	cellSize = [super cellSizeForBounds:theRect];
	
	// Add image width
	cellSize.width += [self imageWidthWithSpace];
	
	// Limite width
	cellSize.width = fminf(NSWidth(theRect), cellSize.width);
	
	return cellSize;
}

// * Prior to title
- (NSRect)imageRectForBounds:(NSRect)theRect
{
	// Get image size
	NSSize imgSize;
	imgSize = [[self image] size];
	
	// Get image rect
	NSRect imgRect;
	imgRect.origin.x = NSMinX(theRect);
	if (!HMIsEmptySize(imgSize)) {
		imgRect.origin.x += HMMarginImage;
	}
	imgRect.size = imgSize;
	imgRect.origin.y = floor(NSMidY(theRect) - (imgRect.size.height / 2.0));
	
	return imgRect;
}

- (NSRect)numberRectForBounds:(NSRect)rect
{
    // Check number
    if (_textFieldType != HMNumberedTextFieldType || _number <= 0) {
        return NSZeroRect;
    }
    
    // Get number rect
    NSRect              numberRect;
    NSAttributedString* attrString;
    attrString = [[NSAttributedString alloc] 
            initWithString:[NSString stringWithFormat:@"%d", _number] attributes:_numberAttr];
    numberRect.size.width = [attrString size].width + HMMarginNumber * 2;
    [attrString release];
    
    numberRect.size.height = rect.size.height;
    numberRect.origin.x = rect.origin.x + rect.size.width - numberRect.size.width - 2;
    numberRect.origin.y = rect.origin.y + (rect.size.height - HMNumberHeight) / 2.0f;
    
    return numberRect;
}

- (NSRect)titleRectForBounds:(NSRect)theRect
{
	// Get title rect
	NSRect  titleRect;
	NSRect  imgRect;
	NSDivideRect(theRect, &imgRect, &titleRect, [self imageWidthWithSpace], NSMinXEdge);
	
    NSRect  numberRect;
    numberRect = [self numberRectForBounds:theRect];
    titleRect.size.width -= numberRect.size.width + HMMarginNumber * 2 + 2;
    
	return titleRect;
}

//--------------------------------------------------------------//
#pragma mark -- Dragging and dropping --
//--------------------------------------------------------------//

- (id)dragDelegate
{
	return _dragDelegate;
}

- (void)setDragDelegate:(id)delegate
{
	_dragDelegate = delegate;
}

- (BOOL)shouldImageTrackMouse:(NSEvent*)event 
        inRect:(NSRect)cellFrame 
        ofView:(NSView*)controlView 
{
    // Check mouse is in image or not
    NSRect  textFrame, imageFrame;
    NSPoint point;
    NSDivideRect(cellFrame, &imageFrame, &textFrame, _padding + [_image size].width, NSMinXEdge);
    point = [controlView convertPoint:[event locationInWindow] fromView:nil];
    
    return NSPointInRect(point, imageFrame);
}

- (BOOL)imageTrackMouse:(NSEvent*)event 
        inRect:(NSRect)cellFrame 
        ofView:(NSView*)controlView 
{
    // Check mouse is in image or not
    NSRect  textFrame, imageFrame;
    NSPoint point;
    NSDivideRect(cellFrame, &imageFrame, &textFrame, _padding + [_image size].width, NSMinXEdge);
    point = [controlView convertPoint:[event locationInWindow] fromView:nil];
    if (!NSPointInRect(point, imageFrame)) {
        return NO;
    }
    
    //
    // Start dragging
    //
    
    id  dragDelegate = nil;
    if ([controlView respondsToSelector:@selector(dragDelegate)]) {
        dragDelegate = [controlView performSelector:@selector(dragDelegate)];
    }
    
    // Create drag image
    NSImage*    dragImage = nil;
    if (dragDelegate && [dragDelegate 
            respondsToSelector:@selector(imageTextCell:dragImageForEvent:inRect:ofView:dragImageOffset:)])
    {
        // Invoke delegate
        NSPoint dragImageOffset;
        dragImage = [dragDelegate imageTextCell:self 
                dragImageForEvent:event 
                inRect:cellFrame 
                ofView:controlView 
                dragImageOffset:&dragImageOffset];
    }
    else {
    }
    
    // Write data to pasteboard
    NSPasteboard*   pboard;
    pboard = [NSPasteboard pasteboardWithName:NSDragPboard];
    if (dragDelegate && [dragDelegate 
            respondsToSelector:@selector(imageTextCell:writeToPasteboard:)])
    {
        // Invoke delegate
        if (![dragDelegate imageTextCell:self writeToPasteboard:pboard]) {
            return NO;
        }
    }
    else {
    }
    
    // Start dragging
    NSPoint startAt;
    startAt = imageFrame.origin;
    if ([controlView isFlipped]) {
        startAt.y = cellFrame.size.height - startAt.y;
    }
    
    [controlView dragImage:dragImage 
            at:startAt 
            offset:NSZeroSize 
            event:event 
            pasteboard:pboard 
            source:self 
            slideBack:YES];
    
    return YES;
}

- (void)resetCursorRect:(NSRect)cellFrame 
        inView:(NSView*)controlView
{
    NSRect  textFrame, imageFrame;
    NSDivideRect(cellFrame, &imageFrame, &textFrame, _padding + [_image size].width, NSMinXEdge);
    
    [super resetCursorRect:textFrame inView:controlView];
}

//--------------------------------------------------------------//
#pragma mark -- Drawing --
//--------------------------------------------------------------//

- (NSRect)_focusRingFrameForFrame:(NSRect)frame
		cellFrame:(NSRect)cellFrame
{
	NSRect ringRect;
	ringRect = [super _focusRingFrameForFrame:frame cellFrame:cellFrame];
	if (![self focusOnlyTitle]) {
		ringRect.size.width += ringRect.origin.x;
		ringRect.origin.x = 0;
	}
	return ringRect;
}

- (void)_drawSeparatorWithFrame:(NSRect)cellFrame 
		inView:(NSView*)controlView
{
	// Draw separator
	NSRect	rect;
	
#if 1
	[[NSColor colorWithCalibratedWhite:0.36f alpha:1.0f] set];
	rect.origin.x = cellFrame.origin.x;
	rect.origin.y = cellFrame.origin.y + ceil(cellFrame.size.height / 2) - 1;
	rect.size.width = cellFrame.size.width;
	rect.size.height = 1;
	NSRectFill(rect);
#else
	[[NSColor colorWithCalibratedWhite:0.6f alpha:0.8f] set];
	rect.origin.x = cellFrame.origin.x;
	rect.origin.y = cellFrame.origin.y + ceil(cellFrame.size.height / 2) - 1;
	rect.size.width = cellFrame.size.width;
	rect.size.height = 1;
	NSRectFill(rect);
	
	[[NSColor colorWithCalibratedWhite:0.9f alpha:0.8f] set];
	rect.origin.y += 1;
	NSRectFill(rect);
#endif
}

- (void)drawWithFrame:(NSRect)cellFrame
		inView:(NSView*)controlView
{
	// For separator
	if (_isSeparator) {
		[self _drawSeparatorWithFrame:cellFrame inView:controlView];
		return;
	}
	
	[super drawWithFrame:cellFrame inView:controlView];
}

- (void)drawInteriorWithFrame:(NSRect)cellFrame 
		inView:(NSView*)controlView
{
	// Draw image
	if ([self image]) {
		[self drawImageWithFrame:cellFrame inView:controlView];
	}
	
    // Draw number
    if (_textFieldType == HMNumberedTextFieldType) {
        [self drawNumberWithFrame:cellFrame inView:controlView];
    }
    
	// Draw title
	[self drawTitleWithFrame:cellFrame inView:controlView];
}

- (void)drawImageWithFrame:(NSRect)cellFrame
		inView:(NSView*)controlView
{
	// Draw image
	NSRect	imageFrame;
	NSImage *image;
	imageFrame = [self imageRectForBounds:cellFrame];
	image = [self image];
	if (image && !NSIsEmptyRect(imageFrame)) {
		
		// Draw
		[image drawInRect:imageFrame
				fromRect:HMMakeRect(NSZeroPoint, [image size])
				operation:NSCompositeSourceOver
				fraction:1.0
				contextRect:cellFrame
				isContextFlipped:[controlView isFlipped]];
	}
}

- (void)drawNumberWithFrame:(NSRect)cellFrame
		inView:(NSView*)controlView
{
    // Get number rect
    NSRect  numberRect;
    numberRect = [self numberRectForBounds:cellFrame];
    if (numberRect.size.width == 0) {
        return;
    }
    
    NSRect  srcRect, destRect;
    float   leftWidth = 0, rightWidth = 0;
    
    // Draw background
    NSImage*    backLeftImage = nil;
    NSImage*    backMiddleImage = nil;
    NSImage*    backRightImage = nil;
    switch (_textFieldNumberType) {
    case HMBlackTextFieldNumber: {
        backLeftImage = _numberBackBlkLeftImage;
        backMiddleImage = _numberBackBlkMiddleImage;
        backRightImage = _numberBackBlkRightImage;
        
        break;
    }
    }
    
    if (backLeftImage && backMiddleImage && backRightImage) {
        // Check flip
        BOOL    isFlipped;
        isFlipped = [controlView isFlipped];
        if (isFlipped != [backLeftImage isFlipped]) {
            [backLeftImage setFlipped:isFlipped];
        }
        if (isFlipped != [backMiddleImage isFlipped]) {
            [backMiddleImage setFlipped:isFlipped];
        }
        if (isFlipped != [backRightImage isFlipped]) {
            [backRightImage setFlipped:isFlipped];
        }
        
        // Draw background left
        srcRect.origin = NSZeroPoint;
        srcRect.size = [backLeftImage size];
        leftWidth = srcRect.size.width;
        destRect.origin = numberRect.origin;
        destRect.size = srcRect.size;
        [backLeftImage drawInRect:destRect fromRect:srcRect operation:NSCompositeSourceOver fraction:1.0f];
        
        // Draw background right
        srcRect.origin = NSZeroPoint;
        srcRect.size = [backRightImage size];
        rightWidth = srcRect.size.width;
        destRect.origin.x = numberRect.origin.x + numberRect.size.width - srcRect.size.width;
        destRect.origin.y = numberRect.origin.y;
        destRect.size = srcRect.size;
        [backRightImage drawInRect:destRect fromRect:srcRect operation:NSCompositeSourceOver fraction:1.0f];
        
        // Draw background middle
        srcRect.origin = NSZeroPoint;
        srcRect.size = [backMiddleImage size];
        destRect.origin.x = numberRect.origin.x + leftWidth;
        destRect.origin.y = numberRect.origin.y;
        destRect.size.width = numberRect.size.width - leftWidth - rightWidth;
        destRect.size.height = srcRect.size.height;
        [backMiddleImage drawInRect:destRect fromRect:srcRect operation:NSCompositeSourceOver fraction:1.0f];
    }
    
    // Create attributed string
    NSAttributedString* attrString;
    attrString = [[NSAttributedString alloc] 
            initWithString:[NSString stringWithFormat:@"%d", _number] attributes:_numberAttr];
    
    // Draw number
    NSRect  attrStrRect;
    attrStrRect = numberRect;
    attrStrRect.origin.x += HMMarginNumber;
    [attrString drawInRect:attrStrRect];
    [attrString release];
}

- (void)drawTitleWithFrame:(NSRect)cellFrame
		inView:(NSView*)controlView
{
	// Get title rect
	NSRect titleRect;
	titleRect = [self titleRectForBounds:cellFrame];
	if (![controlView isFlipped]) {
		titleRect = HMFlipRect(titleRect, cellFrame);
	}
	
	// Get title
	NSAttributedString *title;
	title = [self attributedStringValue];
	
	// Refine title rect
	titleRect.origin.x += 2.0;
	titleRect.origin.y = HMCenterSize([title size], titleRect).origin.y;
	
	// Draw title
	[title drawInRect:titleRect];
}

//--------------------------------------------------------------//
#pragma mark -- Text --
//--------------------------------------------------------------//

- (BOOL)isEditing
{
	return _isEditing;
}

- (NSDictionary*)_textAttributes
{
	// Copy attributes
	NSMutableDictionary*	attrs;
	attrs = [NSMutableDictionary dictionaryWithDictionary:[super _textAttributes]];
	
	// Add paragraph style
	static NSMutableParagraphStyle* _paragraph = nil;
	if (!_paragraph) {
		_paragraph = [[NSMutableParagraphStyle alloc] init];
		[_paragraph setLineBreakMode:NSLineBreakByTruncatingTail];
	}
	[attrs setObject:_paragraph forKey:NSParagraphStyleAttributeName];
	
	// For highlight
	NSView*		controlView;
	NSWindow*	window;
	controlView = [self controlView];
	window = [controlView window];
	if ([self isHighlighted] && 
		[window isKeyWindow] && 
		[window firstResponder] == controlView &&
		![self isEditing])
	{
		// White
		[attrs setObject:[NSColor whiteColor] forKey:NSForegroundColorAttributeName];
		
		if (_useBoldForHighlight) {
			// Make font bold
			NSFont* font;
			font = [attrs objectForKey:NSFontAttributeName];
			font = [[NSFontManager sharedFontManager] 
					convertFont:font toHaveTrait:NSBoldFontMask];
			[attrs setObject:font forKey:NSFontAttributeName];
			
			// Add shadow
			static NSShadow*	_shadow = nil;
			if (!_shadow) {
				_shadow = [[NSShadow alloc] init];
				[_shadow setShadowOffset:NSMakeSize(0, -1)];
				[_shadow setShadowColor:[NSColor blackColor]];
			}
			[attrs setObject:_shadow forKey:NSShadowAttributeName];
		}
	}
	
	return attrs;
}

- (NSText *)setUpFieldEditorAttributes:(NSText *)textObj
{
	_isEditing = YES;
	return textObj;
}

- (void)editWithFrame:(NSRect)rect 
		inView:(NSView*)controlView 
		editor:(NSText*)textObj 
		delegate:(id)delegate 
		event:(NSEvent*)event
{
	// Get title frame
	NSRect titleFrame;
	titleFrame = [self titleRectForBounds:rect];
	if (![controlView isFlipped]) {
		titleFrame = HMFlipRect(titleFrame, rect);
	}
	
	// Edit with title frame
	[super editWithFrame:titleFrame inView:controlView editor:textObj delegate:delegate event:event];
}

- (void)selectWithFrame:(NSRect)rect 
		inView:(NSView*)controlView 
		editor:(NSText*)textObj 
		delegate:(id)delegate 
		start:(int)selStart 
		length:(int)selLength
{
	// Get title frame
	NSRect titleFrame;
	titleFrame = [self titleRectForBounds:rect];
	if (![controlView isFlipped]) {
		titleFrame = HMFlipRect(titleFrame, rect);
	}
	
	// Select with title frame
	[super selectWithFrame:titleFrame inView:controlView editor:textObj delegate:delegate start:selStart length:selLength];
}

- (void)endEditing
{
	_isEditing = NO;
}

@end
