/*
RSSRepresentation.m

Author: Makoto Kinoshita

Copyright 2004-2006 The Shiira Project. All rights reserved.

Redistribution and use in source and binary forms, with or without modification, are permitted 
provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice, this list of conditions 
  and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright notice, this list of 
  conditions and the following disclaimer in the documentation and/or other materials provided 
  with the distribution.

THIS SOFTWARE IS PROVIDED BY THE SHIIRA PROJECT ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, 
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE SHIIRA PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
POSSIBILITY OF SUCH DAMAGE.
*/

#import "RSSRepresentation.h"
#import "RSSView.h"

@implementation RSSRepresentation

//--------------------------------------------------------------//
#pragma mark -- RSS MIMEs --
//--------------------------------------------------------------//

+ (NSArray*)RSSMimes
{
    // Create RSS MIME array
    static NSArray* _RSSMimes = nil;
    if (!_RSSMimes) {
        _RSSMimes = [NSArray arrayWithObjects:
                @"application/rss+xml", 
                @"application/rdf+xml", 
                @"application/atom+xml", 
                nil];
    }
    
    return _RSSMimes;
}

//--------------------------------------------------------------//
#pragma mark -- Initialize --
//--------------------------------------------------------------//

+ (void)load
{
    NSAutoreleasePool*  pool;
    pool = [[NSAutoreleasePool alloc] init];
    
    // Get RSS MIME array
    NSArray*    RSSMimes;
    RSSMimes = [self RSSMimes];
    
    // Register representation and view
    NSEnumerator*   enumerator;
    NSString*       MIMEType;
    enumerator = [RSSMimes objectEnumerator];
    while (MIMEType = [enumerator nextObject]) {
        [WebView registerViewClass:NSClassFromString(@"RSSView") 
                representationClass:NSClassFromString(@"RSSRepresentation") 
                forMIMEType:MIMEType];
    }
    
    [pool release];
}

- (void)dealloc
{
    [_dataSource release], _dataSource = nil;
    [_rssController release], _rssController = nil;
    
    [super dealloc];
}

//--------------------------------------------------------------//
#pragma mark -- Feed info --
//--------------------------------------------------------------//

- (NSString*)link
{
    if (_rssController) {
        return [_rssController link];
    }
    return nil;
}

//--------------------------------------------------------------//
#pragma mark -- WebDocumentRepresentation protocol --
//--------------------------------------------------------------//

- (void)setDataSource:(WebDataSource*)dataSource
{
    [_dataSource release];
    _dataSource = [dataSource retain];
}

- (void)receivedData:(NSData*)data withDataSource:(WebDataSource*)dataSource
{
}

- (void)receivedError:(NSError*)error withDataSource:(WebDataSource*)dataSource
{
}

- (void)finishedLoadingWithDataSource:(WebDataSource*)dataSource
{
    // Get RSS view
    RSSView*    rssView;
    rssView = (RSSView*)[[[dataSource webFrame] frameView] documentView];
    [rssView layout];
    
    // Create RSS controller
    if (!_rssController) {
        _rssController = [[RSSController alloc] init];
    }
    
    // Set view
    NSView* view;
    view = [_rssController view];
    [view setFrame:[rssView bounds]];
    [rssView addSubview:view];
    
    // Set data source
    [_rssController finishedLoadingWithDataSource:dataSource];
    
    // Notify title
    WebFrame*   webFrame;
    WebView*    webView;
    id          delegate;
    webFrame = [dataSource webFrame];
    webView = [webFrame webView];
    delegate = [webView frameLoadDelegate];
    if ([delegate respondsToSelector:@selector(webView:didReceiveTitle:forFrame:)]) {
        [delegate webView:webView didReceiveTitle:[self title] forFrame:webFrame];
    }
    
    // Notify finish load frame
    if ([delegate respondsToSelector:@selector(webView:didFinishLoadForFrame:)]) {
        [delegate webView:webView didFinishLoadForFrame:webFrame];
    }
}

- (BOOL)canProvideDocumentSource
{
    return YES;
}

- (NSString*)documentSource
{
    NSData*             data;
    NSString*           textEncodingName;
    NSStringEncoding    encoding = NSASCIIStringEncoding;
    data = [_dataSource data];
    textEncodingName = [_dataSource textEncodingName];
    if (textEncodingName) {
        CFStringEncoding    cfEncoding;
        cfEncoding = CFStringConvertIANACharSetNameToEncoding((CFStringRef)textEncodingName);
        encoding = CFStringConvertEncodingToNSStringEncoding(cfEncoding);
    }
    
    NSString*   source;
    source = [[NSString alloc] initWithData:data encoding:encoding];
    [source autorelease];
    
    return source;
}

- (NSString*)title
{
    if (_rssController) {
        return [_rssController title];
    }
    if (_dataSource) {
        return [[[_dataSource request] URL] absoluteString];
    }
    return @"RSS";
}

@end
