/*
HMFoundationEx.m

Author: Makoto Kinoshita

Copyright 2004-2006 The Shiira Project. All rights reserved.

Redistribution and use in source and binary forms, with or without modification, are permitted 
provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice, this list of conditions 
  and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright notice, this list of 
  conditions and the following disclaimer in the documentation and/or other materials provided 
  with the distribution.

THIS SOFTWARE IS PROVIDED BY THE SHIIRA PROJECT ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, 
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE SHIIRA PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
POSSIBILITY OF SUCH DAMAGE.
*/

#import "HMFoundationEx.h"

BOOL HMIsEmptySize(NSSize size)
{
	if (size.width <= 0.0 || size.height <= 0.0) {
		return YES;
	}
	
	return NO;
}

NSSize HMScaleSizeInSize(NSSize size, NSSize limit)
{
	// Get ratio
	float ratio;
	float xRatio, yRatio;
	xRatio = size.width ? (limit.width / size.width) : MAXFLOAT;
	yRatio = size.height ? (limit.height / size.height) : MAXFLOAT;
	ratio = fabsf(xRatio) < fabsf(yRatio) ? xRatio : yRatio;
	return NSMakeSize(size.width * ratio, size.height * ratio);
}

#pragma mark -

@implementation NSAffineTransform (Extension)

+ (NSAffineTransform*)verticalFlipTransformForRect:(NSRect)rect
{
	NSAffineTransform *transform;
	transform = [NSAffineTransform transform];
	[transform translateXBy:0 yBy:NSMaxY(rect)];
	[transform scaleXBy:1.0 yBy:-1.0];
	
	return transform;
}

- (NSRect)transformRect:(NSRect)aRect
{
	NSRect theRect;
	theRect.origin = [self transformPoint:aRect.origin];
	theRect.size = [self transformSize:aRect.size];
	
	return theRect;
}
@end

#pragma mark -

// NSArray
@implementation NSArray (Subarray)

- (NSArray*)subarrayFromIndex:(unsigned)index
{
    return [self subarrayWithRange:NSMakeRange(index, [self count] - index)];
}

- (NSArray*)subarrayToIndex:(unsigned)index
{
    return [self subarrayWithRange:NSMakeRange(0, index)];
}

- (NSArray*)subarrayWithIndexsArray:(NSArray*)indexes
{
    // Check arguments
    if (!indexes) {
        return nil;
    }
    
    // Create subarray
    NSMutableArray* subarray;
    NSEnumerator*   enumerator;
    NSNumber*       index;
    subarray = [NSMutableArray arrayWithCapacity:[indexes count]];
    enumerator = [indexes objectEnumerator];
    while (index = [enumerator nextObject]) {
        unsigned int    indexValue;
        indexValue = [index unsignedIntValue];
        if (indexValue > [self count]) {
            continue;
        }
        
        [subarray addObject:[self objectAtIndex:indexValue]];
    }
    
    return subarray;
}

- (NSArray*)subarrayWithIndexSet:(NSIndexSet*)set
{
    // Check arguments
    if (!set) {
        return nil;
    }
    
    // Create subarray
    NSMutableArray* subarray;
    unsigned        maxIndex;
    unsigned int    index;
    subarray = [NSMutableArray arrayWithCapacity:[set count]];
    maxIndex = [self count] - 1;
    index = [set firstIndex];
    while (index != NSNotFound) {
        if (index > maxIndex) {
            break;
        }
        
        [subarray addObject:[self objectAtIndex:index]];
        
        index = [set indexGreaterThanIndex:index];
    }
    
    return subarray;
}

@end

@implementation NSArray (Querying)

- (id)objectOfTag:(int)tag
{
    NSEnumerator*   enumerator;
    id              object;
    enumerator = [self objectEnumerator];
    while (object = [enumerator nextObject]) {
        if ([object respondsToSelector:@selector(tag)]) {
		if ((int)[object performSelector:@selector(tag)] == tag) {
                break;
            }
        }
    }
    
    return object;
}

@end

@implementation NSArray (StringElements)

- (NSString*)componentsJoinedByString:(NSString*)separator
		prefix:(NSString*)prefix
		suffix:(NSString*)suffix
{
	// Filter
	if (![self count]) {
		return @"";
	}
	
	// Get string
	NSMutableString *string;
	string = [NSMutableString string];
	if (prefix) {
		[string appendString:prefix];
	}
	[string appendString:[self componentsJoinedByString:separator]];
	if (suffix) {
		[string appendString:suffix];
	}
	
	// Return string
	return [NSString stringWithString:string];
}

@end

#pragma mark -

@implementation NSCharacterSet (NewLine)

+ (NSCharacterSet*)newLineCharacterSet
{
    static NSCharacterSet*  _newlineCharacterSet = nil;
    if (!_newlineCharacterSet) {
        unichar     newlineChars[] = {0x000A, 0x000D, 0x0085};
        NSString*   newlineString;
        newlineString = [NSString stringWithCharacters:newlineChars 
                length:sizeof(newlineChars) / sizeof(unichar)];
        _newlineCharacterSet = [[NSCharacterSet characterSetWithCharactersInString:newlineString] retain];
    }
    
    return _newlineCharacterSet;
}

@end

#pragma mark -

@implementation NSDate (DateFormat)

static  NSMutableArray* _formats = nil;

+ (void)load
{
    NSAutoreleasePool*  pool;
    pool = [[NSAutoreleasePool alloc] init];
    
    if (!_formats) {
        _formats = [[NSMutableArray array] retain];
        
        // Add default formats
        NSBundle*   bundle;
        NSString*   path;
        bundle = [NSBundle bundleWithIdentifier:@"jp.hmdt.framework.hmdtfoundation"];
        path = [bundle pathForResource:@"DateFormats" ofType:@"plist"];
        if (path) {
            [self addDateFromatsWithContentsOfFile:path];
        }
    }
    
    [pool release];
}

+ (NSArray*)dateFormats
{
    return [NSArray arrayWithArray:_formats];
}

+ (void)addDateFromatsWithContentsOfFile:(NSString*)filePath
{
    // Load file
    NSArray*    formats;
    formats = [NSArray arrayWithContentsOfFile:filePath];
    if (!formats) {
        // Warning
        NSLog(@"Can't load %@", filePath);
        return;
    }
    
    // Add formats
    [NSDate addDateFormats:formats];
}

+ (void)addDateFormats:(NSArray*)formats
{
    NSEnumerator*   enumerator;
    NSString*       format;
    enumerator = [formats objectEnumerator];
    while (format = [enumerator nextObject]) {
        if (![_formats containsObject:format]) {
            NSString*   copiedFormat;
            copiedFormat = [format copy];
            [_formats addObject:copiedFormat];
            [copiedFormat release];
        }
    }
}

+ (void)removeDateFormats:(NSArray*)formats
{
    [_formats removeObjectsInArray:formats];
}

+ (id)dateWithFormattedString:(NSString*)string
{
    if (!string) {
        return nil;
    }
    
    // Remove ':' between HH and MM of time zone
    int length;
    length = [string length];
    if (length > 6) {
        unichar colon, sign;
        colon = [string characterAtIndex:length - 3];
        sign = [string characterAtIndex:length - 6];
        if (colon == ':' && (sign == '-' || sign == '+')) {
            string = [NSString stringWithFormat:@"%@%@", 
                    [string substringToIndex:length - 3], 
                    [string substringFromIndex:length - 2]];
        }
    }
    
    // Swap 'Thur' to 'Thu'
    NSRange range;
    range = [string rangeOfString:@"Thur"];
    if (range.location != NSNotFound) {
        string = [NSString stringWithFormat:@"%@Thu%@", 
                [string substringToIndex:range.location], 
                [string substringFromIndex:range.location + range.length]];
    }
    
    // Apply format
    NSEnumerator*   enumerator;
    NSString*       format;
    enumerator = [_formats objectEnumerator];
    while (format = [enumerator nextObject]) {
        NSCalendarDate* date;
        date = [NSCalendarDate dateWithString:string calendarFormat:format];
        if (date) {
            return date;
        }
    }
    
    // No proper format
    return nil;
}

+ (NSString*)formatForFormattedString:(NSString*)string
{
    // Apply format
    NSEnumerator*   enumerator;
    NSString*       format;
    enumerator = [_formats objectEnumerator];
    while (format = [enumerator nextObject]) {
        NSCalendarDate* date;
        date = [NSCalendarDate dateWithString:string calendarFormat:format];
        if (date) {
            return format;
        }
    }
    
    // No proper format
    return nil;
}

@end

#pragma mark -

@implementation NSFileManager (UniqueFilePath)

static int  _maxRepeatTime = 999;

- (NSString*)makeUniqueFilePath:(NSString*)filePath
{
    if (![self fileExistsAtPath:filePath]) {
        return filePath;
    }
    
    // Get file name and extension separately
    NSRange     range;
    NSString*   fileName;
    NSString*   extension;
    range = [filePath rangeOfString:@"."];  // Find first '.'
    if (range.location != NSNotFound) {
        fileName = [filePath substringToIndex:range.location];
        extension = [filePath substringFromIndex:range.location + 1];
    }
    else {
        fileName = filePath;
        extension = nil;
    }
    
    // Make new file path
    int i;
    for (i = 1; i < _maxRepeatTime; i++) {
        NSString*   newFilePath;
        if (extension) {
            newFilePath = [NSString stringWithFormat:@"%@-%d.%@", fileName, i, extension];
        }
        else {
            newFilePath = [NSString stringWithFormat:@"%@-%d", fileName, i];
        }
        
        if (![self fileExistsAtPath:newFilePath]) {
            return newFilePath;
        }
    }
    
    return nil;
}

@end

#pragma mark -

@implementation NSMutableString (Modifying)

- (void)prependString:(NSString*)string
{
    [self insertString:string atIndex:0];
}

@end

#pragma mark -

// NSObject
@implementation NSObject (TryKeyValueCoding)

- (id)tryValueForKey:(NSString*)key
{
	id obj = nil;
	@try {
		obj = [self valueForKey:key];
	}
	@catch (NSException *ex) {
	}
	
	return obj;
}

- (id)tryValueForKeyPath:(NSString*)keyPath
{
	id obj = nil;
	@try {
		obj = [self valueForKeyPath:keyPath];
	}
	@catch (NSException *ex) {
	}
	
	return obj;
}

@end

#pragma mark -

@implementation NSString (CharacterReference)

- (NSString*)stringByReplacingCharacterReferences
{
    NSMutableString*    scanedString;
    scanedString = [NSMutableString string];
    
    NSScanner*  scanner;
    scanner = [NSScanner scannerWithString:self];
    
    // Scan '&#'
    while (![scanner isAtEnd]) {
        NSString*   string;
        if ([scanner scanUpToString:@"&#" intoString:&string]) {
            [scanedString appendString:string];
        }
        
        if ([scanner scanString:@"&#" intoString:NULL]) {
            NSString*   digit;
            if  ([scanner scanUpToString:@";" intoString:&digit] && 
                 [scanner scanString:@";" intoString:NULL])
            {
                [scanedString appendFormat:@"%C", (unichar)[digit intValue]];
            }
        }
    }
    
    return scanedString;
}

@end

@implementation NSString (CharacterSet)

- (NSString*)stringBySamplingCharactersInSet:(NSCharacterSet*)set
{
    NSMutableString*    string;
    NSRange             range;
    string = [NSMutableString string];
    range = NSMakeRange(0, [self length]);
    
    while (YES) {
        range = [self rangeOfCharacterFromSet:set options:0 range:range];
        if (range.location == NSNotFound) {
            break;
        }
        
        [string appendString:[self substringWithRange:range]];
        range.location = range.location + range.length;
        range.length = [self length] - range.location;
    }
    
    return string;
}

@end

#pragma mark -

@implementation NSXMLNode (Extension)

- (NSXMLNode*)singleNodeForXPath:(NSString*)XPath
{
    NSArray*    nodes;
    nodes = [self nodesForXPath:XPath error:NULL];
    if (!nodes || [nodes count] != 1) {
        return nil;
    }
    return [nodes objectAtIndex:0];
}

- (NSString*)stringValueForXPath:(NSString*)XPath
{
    NSXMLNode*  node;
    node = [self singleNodeForXPath:XPath];
    if (!node) {
        return nil;
    }
    
    //return [node stringValue];
    return [[node stringValue] stringByReplacingCharacterReferences];
}

@end
