/*
HMWindow.m

Author: Makoto Kinoshita

Copyright 2004-2006 The Shiira Project. All rights reserved.

Redistribution and use in source and binary forms, with or without modification, are permitted 
provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice, this list of conditions 
  and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright notice, this list of 
  conditions and the following disclaimer in the documentation and/or other materials provided 
  with the distribution.

THIS SOFTWARE IS PROVIDED BY THE SHIIRA PROJECT ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, 
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE SHIIRA PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
POSSIBILITY OF SUCH DAMAGE.
*/

#import "HMWindow.h"

// Default keys
NSString*   HMWindowTexture = @"HMWindowTexture";
NSString*   HMWindowUseUnifiedToolbar = @"HMWindowUseUnifiedToolbar";
NSString*   HMWindowIgnoreButtons = @"HMWindowIgnoreButtons";

@interface NSWindow (bottomCorner)
- (void)setBottomCornerRounded:(BOOL)flag;
@end

@implementation HMWindow

//--------------------------------------------------------------//
#pragma mark -- Initialize --
//--------------------------------------------------------------//

- (id)initWithContentRect:(NSRect)contentRect 
        styleMask:(unsigned int)styleMask 
        backing:(NSBackingStoreType)backingType 
        defer:(BOOL)flag
{
    NSUserDefaults* defaults;
    defaults = [NSUserDefaults standardUserDefaults];
    
    // Initialize instance variables
    _mouseMoveListeners = [[NSMutableSet set] retain];
    
    // Get texture type
    int texture;
    texture = [defaults integerForKey:HMWindowTexture];
    
    // Set style mask
    if (texture == HMWindowTextureMetal && !(styleMask & NSTexturedBackgroundWindowMask)) {
        styleMask |= NSTexturedBackgroundWindowMask;
    }
    
    // Use unified title and toolbar window
    BOOL    unified;
    unified = [defaults boolForKey:HMWindowUseUnifiedToolbar];
    if (unified) {
        styleMask |= NSUnifiedTitleAndToolbarWindowMask;
    }
    
    // Invoke super
    self = [super initWithContentRect:contentRect 
            styleMask:styleMask 
            backing:backingType 
            defer:flag];
    if (!self) {
        return nil;
    }
    
    // Set bottom corner rounded
    if (texture == HMWindowTextureMetal) {
        [self setBottomCornerRounded:NO];
    }
    
    // Set accepts mouse moved events
    [self setAcceptsMouseMovedEvents:YES];
    
    return self;
}

- (void)dealloc
{
    [_mouseMoveListeners release], _mouseMoveListeners = nil;
    
    [super dealloc];
}

//--------------------------------------------------------------//
#pragma mark -- Window buttons --
//--------------------------------------------------------------//

+ (NSButton*)standardWindowButton:(NSWindowButton)button 
        forStyleMask:(unsigned int)styleMask
{
    NSUserDefaults* defaults;
    defaults = [NSUserDefaults standardUserDefaults];
    
    // Ignore window buttons
    NSArray*    ignoreButtons;
    ignoreButtons = [defaults objectForKey:HMWindowIgnoreButtons];
    if (ignoreButtons) {
        if ([ignoreButtons containsObject:[NSNumber numberWithInt:button]]) {
            return nil;
        }
    }
    
    return [super standardWindowButton:button forStyleMask:styleMask];
}

//--------------------------------------------------------------//
#pragma mark -- Actions --
//--------------------------------------------------------------//

- (void)closeWindowAction:(id)sender
{
    // Perform close
    [self performClose:sender];
}

//--------------------------------------------------------------//
#pragma mark -- Mouse move listener --
//--------------------------------------------------------------//

- (void)addMouseMoveListener:(id)listener
{
    [_mouseMoveListeners addObject:listener];
}

- (void)removeMouseMoveListener:(id)listener
{
    [_mouseMoveListeners removeObject:listener];
}

//--------------------------------------------------------------//
#pragma mark -- Menu item validation --
//--------------------------------------------------------------//

- (BOOL)passMenuValidationToDelegate
{
    return _passMenuValidationToDelegate;
}

- (void)setPassMenuValidationToDelegate:(BOOL)flag
{
    _passMenuValidationToDelegate = flag;
}

- (BOOL)validateMenuItem:(id<NSMenuItem>)menuItem
{
    // Pass this validation to delegate
    if (_passMenuValidationToDelegate) {
        return [[self delegate] validateMenuItem:menuItem];
    }
    return YES;
}

//--------------------------------------------------------------//
#pragma mark -- NSResponder override --
//--------------------------------------------------------------//

- (void)mouseMoved:(NSEvent*)event
{
    // Notify to listener
    NSEnumerator*   enumerator;
    id              listener;
    enumerator = [_mouseMoveListeners objectEnumerator];
    while (listener = [enumerator nextObject]) {
        if ([listener respondsToSelector:@selector(hmWindow:mouseMoved:)]) {
            [listener hmWindow:self mouseMoved:event];
        }
    }
    
    [super mouseMoved:event];
}

//--------------------------------------------------------------//
#pragma mark -- NSWindow override --
//--------------------------------------------------------------//

- (BOOL)canBecomeKeyWindow
{
    // Always can become key window
    return YES;
}

@end

#pragma mark -

@implementation NSWindow (HMDTEx)

//--------------------------------------------------------------//
#pragma mark -- Texture --
//--------------------------------------------------------------//

- (BOOL)isMetal
{
    return [self styleMask] & NSTexturedBackgroundWindowMask;
}

@end
