/*
HMAppKitEx.m

Author: Makoto Kinoshita

Copyright 2004-2006 The Shiira Project. All rights reserved.

Redistribution and use in source and binary forms, with or without modification, are permitted 
provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice, this list of conditions 
  and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright notice, this list of 
  conditions and the following disclaimer in the documentation and/or other materials provided 
  with the distribution.

THIS SOFTWARE IS PROVIDED BY THE SHIIRA PROJECT ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, 
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE SHIIRA PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
POSSIBILITY OF SUCH DAMAGE.
*/

#import "HMAppKitEx.h"

@implementation NSAlert (Dispalying)

+ (void)runModalWithMessage:(NSString*)message 
        informativeText:(NSString*)informativeText
{
    // Create alert
    NSAlert*    alert;
    alert = [[NSAlert alloc] init];
    if (message) {
        [alert setMessageText:message];
    }
    if (informativeText) {
        [alert setInformativeText:informativeText];
    }
    
    // Run modal
    [alert runModal];
    
    [alert release];
}

@end

#pragma mark -

@implementation NSAttributedString (Truncate)

- (NSAttributedString*)truncatedAttributedStringForWidth:(int)inWidth
{
	static NSString*	_horizontalEllipsis = nil;
	if (!_horizontalEllipsis) {
		unichar uni[1];
		uni[0] = 0x2026;
		_horizontalEllipsis = [[NSString stringWithCharacters:uni length:1] retain];
	}
	
	NSAttributedString *result = self;
	if ([self size].width > inWidth) {
		NSMutableAttributedString *newString;
		newString = [[NSMutableAttributedString alloc] initWithAttributedString:self];
		[newString autorelease];
		
		int curLength = [self length] - 1; // start by chopping off at least one
		
		while ([newString size].width > inWidth) {
			if (curLength == 0) {
				return nil;
			}
			NSRange range;
			range = NSMakeRange(curLength - 1, 2); // replace 2 characters with '…'
			[newString replaceCharactersInRange:range withString:_horizontalEllipsis];
			curLength--;
		}
		result = newString;
	}
	return result;
}
@end

#pragma mark -

@implementation NSBezierPath (ellipse)

+ (NSBezierPath*)ellipseInRect:(NSRect)rect withRadius:(float)r
{
    float   x, y, w, h;
    x = rect.origin.x;
    y = rect.origin.y;
    w = rect.size.width;
    h = rect.size.height;
    
    // Create ellipse bezier path
    NSBezierPath*   path;
    path = [NSBezierPath bezierPath];
    
    [path moveToPoint:NSMakePoint(x, y + r)];
    [path lineToPoint:NSMakePoint(x, y + h - r)];
    [path curveToPoint:NSMakePoint(x + r, y + h) 
            controlPoint1:NSMakePoint(x, y + h - r/2) 
            controlPoint2:NSMakePoint(x + r/2, y + h)];
    [path lineToPoint:NSMakePoint(x + w - r, y + h)];
    [path curveToPoint:NSMakePoint(x + w, y + h - r) 
            controlPoint1:NSMakePoint(x + w - r/2, y + h) 
            controlPoint2:NSMakePoint(x + w, y + h - r/2)];
    [path lineToPoint:NSMakePoint(x + w, y + r)];
    [path curveToPoint:NSMakePoint(x + w - r, y) 
            controlPoint1:NSMakePoint(x + w, y + r/2) 
            controlPoint2:NSMakePoint(x + w - r/2, y)];
    [path lineToPoint:NSMakePoint(x + r, y)];
    [path curveToPoint:NSMakePoint(x, y + r) 
            controlPoint1:NSMakePoint(x + r/2, y) 
            controlPoint2:NSMakePoint(x, y + r/2)];
    [path closePath];
    
    return path;
}

@end

#pragma mark -

@implementation NSBrowser (HMDTCellRepresentaionPath)

- (NSString*)representedObjectPath
{
    return [self representedObjectPathToColumn:[self lastColumn]];
}

- (BOOL)setRepresentedObjectPath:(NSString*)path
{
    // Filter
    if (![path length]) {
        return NO;
    }
    
    // Set to root
    if ([path isEqualToString:@"/"]) {
        return [self setPath:@"/"];
    }
    
    // Calc start column, Get components
    int startColumn;
    NSArray *components;
    
    // Absolute
    if ([path hasPrefix:@"/"]) {
        startColumn = 0;
        components = [[path componentsSeparatedByString:[self pathSeparator]] subarrayFromIndex:1];
    }
    // Relative
    else {
        startColumn = [self lastColumn];
        components = [path componentsSeparatedByString:[self pathSeparator]];
    }
    
    // Select components
    int i;
    for (i = 0; i < [components count]; i++) {
        // Get path component
        NSString *component;
        component = [components objectAtIndex:i];
        
        // Get current column
        int column;
        column = startColumn + i;
        
        // Check if already selected
        NSBrowserCell *cell;
        cell = [self selectedCellInColumn:column];
        if (cell && [[cell representedObject] isEqualToString:component]) {
            continue;
        }
        
        // Select cell
        NSMatrix *matrix;
        int j;
        matrix = [self matrixInColumn:column];
        for (j = 0; j < [matrix numberOfRows]; j++) {
            // Get cell
            cell = [matrix cellAtRow:j column:0];
            if (!cell) {
                continue;
            }
            
            // Check represented object
            if ([[cell representedObject] isEqualToString:component]) {
                // Select cell adn reload column
                [self selectRow:j inColumn:column];
                [self reloadColumn:column];
                break;
            }
        }
        
        // Not found in this matrix
        if (j == [matrix numberOfRows]) {
            return NO;
        }
    }
    
    return YES;
}

- (NSString*)representedObjectPathToColumn:(int)column
{
    // Get path separator
    NSString*   pathSeparator;
    pathSeparator = [self pathSeparator];
    
    // Create path
    NSMutableString*    path;
    path = [NSMutableString string];
    
    int i;
    for (i = 0; i <= column; i++) {
        // Check column
        if (i > [self lastColumn]) {
            break;
        }
        
        // Get represented object
        NSCell* cell;
        id      representedObject;
        cell = [self selectedCellInColumn:i];
        if (!cell) {
            break;
        }
        representedObject = [cell representedObject];
        
        if (!representedObject || ![representedObject isKindOfClass:[NSString class]]) {
            break;
        }
        
        // Add path
        [path appendFormat:@"%@%@", pathSeparator, representedObject];
    }
    
    if ([path length] == 0) {
        [path appendString:pathSeparator];
    }
    return path;
}

@end

#pragma mark -

@implementation NSDocumentController (MIMEType)

- (NSString*)typeFromMIMEType:(NSString*)MIMEType
{
    // Get info dictionary
    NSDictionary*   infoDict;
    infoDict = [[NSBundle mainBundle] infoDictionary];
    if (!infoDict) {
        return nil;
    }
    
    // Get CFBundleDocumentTypes
    NSArray*    types;
    types = [infoDict objectForKey:@"CFBundleDocumentTypes"];
    if (!types) {
        return nil;
    }
    
    // Enumerate document types
    NSEnumerator*   enumerator;
    NSDictionary*   type;
    enumerator = [types objectEnumerator];
    while (type = [enumerator nextObject]) {
        // Get MIME types
        NSArray*    mimeTypes;
        mimeTypes = [type objectForKey:@"CFBundleTypeMIMETypes"];
        if (!mimeTypes) {
            continue;
        }
        
        // Check MIME type
        NSEnumerator*   mimeEnumerator;
        NSString*       mime;
        mimeEnumerator = [mimeTypes objectEnumerator];
        while (mime = [mimeEnumerator nextObject]) {
            if ([mime isEqualToString:MIMEType]) {
                // Return type name
                return [type objectForKey:@"CFBundleTypeName"];
            }
        }
    }
    
    return nil;
}

@end

#pragma mark -

@implementation NSImage (Assemble)

+ (NSImage*)imageWithSize:(NSSize)size
		leftImage:(NSImage*)leftImage
		middleImage:(NSImage*)middleImage
		rightImage:(NSImage*)rightImage
		middleRect:(NSRect*)outMiddleRect
{
	// Filter
	if (!leftImage || !middleImage || !rightImage || ![leftImage size].height || ![rightImage size].height || !size.width || !size.height) {
		return nil;
	}
	
	// Get copy of images
	NSImage *left, *middle, *right;
	left = [[leftImage copy] autorelease];
	middle = [[middleImage copy] autorelease];
	right = [[rightImage copy] autorelease];
	
	float floatHeight;
	int intHeight;
	floatHeight = size.height;
	intHeight = rintf(floatHeight);
	
	// Resize images
	NSSize s;
	int middleWidth;
	s = [left size];
	[left setSize:NSMakeSize(rintf(floatHeight / s.height * s.width), intHeight)];
	s = [right size];
	[right setSize:NSMakeSize(rintf(size.height / s.height * s.width), intHeight)];
	middleWidth = rintf(size.width - ([left size].width + [right size].width));
	if (middleWidth < 1) {
		return nil;
	}
	[middle setSize:NSMakeSize(middleWidth, intHeight)];
	
	// Calculate middle rect
	NSRect middleRect;
	middleRect = NSMakeRect([left size].width, 0, middleWidth, intHeight);
	
	// Make new image
	NSImage *image;
	image = [[NSImage alloc] initWithSize:size];
	[image lockFocus];
	[left drawAtPoint:NSZeroPoint
			fromRect:HMMakeRect(NSZeroPoint, [left size])
			operation:NSCompositeSourceOver
			fraction:1.0];
	[middle drawAtPoint:middleRect.origin
			fromRect:HMMakeRect(NSZeroPoint, [middle size])
			operation:NSCompositeSourceOver
			fraction:1.0];
	[right drawAtPoint:NSMakePoint([left size].width + [middle size].width, 0)
			fromRect:HMMakeRect(NSZeroPoint, [right size])
			operation:NSCompositeSourceOver
			fraction:1.0];
	[image unlockFocus];
	
	// Set outMiddleRect
	if (outMiddleRect) {
		*outMiddleRect = middleRect;
	}
	
	return [image autorelease];
}
@end

@implementation NSImage (Drawing)
- (void)drawInRect:(NSRect)dstRect
		fromRect:(NSRect)srcRect
		operation:(NSCompositingOperation)op
		fraction:(float)delta
		contextRect:(NSRect)ctxRect
		isContextFlipped:(BOOL)flag
{
	if (flag) {
		NSAffineTransform *transform;
		transform = [NSAffineTransform transform];
		[transform scaleXBy:1.0 yBy:-1.0];
		[transform concat];
		[transform invert];
		[self drawInRect:[transform transformRect:dstRect] fromRect:srcRect operation:op fraction:delta];
		[transform concat];
	}
	else {
		[self drawInRect:HMFlipRect(dstRect, ctxRect) fromRect:srcRect operation:op fraction:delta];
	}
}
@end

#pragma mark -

@implementation NSMatrix (FindingCells)

- (id)cellAtRowAndColumn:(NSArray*)rowAndColumn
{
    // Check argument
    if ([rowAndColumn count] < 2) {
        return nil;
    }
    
    // Get row and column
    int row, col;
    row = [[rowAndColumn objectAtIndex:0] intValue];
    col = [[rowAndColumn objectAtIndex:1] intValue];
    
    // Get cell
    return [self cellAtRow:row column:col];
}

- (NSArray*)cellsAtRowsAndColumns:(NSArray*)rowsAndColumns
{
    NSMutableArray* cells;
    cells = [NSMutableArray array];
    
    // Get cells
    NSEnumerator*   enumerator;
    NSArray*        rowAndColumn;
    enumerator = [rowsAndColumns objectEnumerator];
    while (rowAndColumn = [enumerator nextObject]) {
        id  cell;
        cell = [self cellAtRowAndColumn:rowAndColumn];
        if (cell) {
            [cells addObject:cell];
        }
    }
    
    return cells;
}

@end

#pragma mark -

@implementation NSNextStepFrame (appearance)

- (float)contentAlpha
{
    return 1.0f;
}

@end

#pragma mark -

@implementation NSOutlineView (Selecting)

- (void)selectRowsForItems:(NSArray*)items byExtendingSelection:(BOOL)extend
{
	// Filter
	if (!items || ![items count]) {
		return;
	}
	
	// Get rows
	NSMutableIndexSet *rows;
	NSEnumerator *enumerator;
	id item;
	enumerator = [items objectEnumerator];
	rows = [NSMutableIndexSet indexSet];
	while (item = [enumerator nextObject]) {
		int row;
		row = [self rowForItem:item];
		if (row >= 0) {
			[rows addIndex:row];
		}
	}
	
	// Select rows
	if ([rows count]) {
		[self selectRowIndexes:rows byExtendingSelection:extend];
	}
}
@end

@implementation NSOutlineView (ExpandingAndCollapsing)

- (void)expandAllItems
{
    // Expand items
    int i;
    for (i = 0; i < [self numberOfRows]; i++) {
        id  item;
        item = [self itemAtRow:i];
        
        [self expandItem:item expandChildren:YES];
    }
}

- (void)collapseAllItems
{
    // Collapse items
    int i;
    for (i = 0; i < [self numberOfRows]; i++) {
        id  item;
        item = [self itemAtRow:i];
        
        [self collapseItem:item collapseChildren:YES];
    }
}

@end

@implementation NSOutlineView (ContextMenu)

- (NSMenu*)menuForEvent:(NSEvent*)event
{
    if ([[self delegate] respondsToSelector:@selector(outlineView:menuForEvent:)]) {
        return [[self delegate] outlineView:self menuForEvent:event];
    }
    
    return nil;
}

- (void)draggedImage:(NSImage*)image 
        endedAt:(NSPoint)point 
        operation:(NSDragOperation)operation
{
    if ([[self delegate] respondsToSelector:@selector(draggedImage:endedAt:operation:)]) {
        [[self delegate] draggedImage:image endedAt:point operation:operation];
    }
}

@end

#pragma mark -

@implementation NSSplitView (HMDTAutosave)

- (void)saveLayoutWithName:(NSString*)name
{
    NSMutableArray* viewRects;
    NSEnumerator*   enumerator;
    NSView*         view;
    NSRect          frame;
    
    viewRects = [NSMutableArray array];
    enumerator = [[self subviews] objectEnumerator];
    while (view = [enumerator nextObject]) {
        // For collapsed view
        if ([self isSubviewCollapsed:view]) {
            frame = NSZeroRect;
        }
        // Normal view
        else {
            frame = [view frame];
        }
        
        [viewRects addObject:NSStringFromRect(frame)];
    }
    
    // Store in default database
    [[NSUserDefaults standardUserDefaults] setObject:viewRects forKey:name];
}

- (void)loadLayoutWithName:(NSString*)name
{
    // Get view rects from default database
    NSMutableArray* viewRects;
    viewRects = [[NSUserDefaults standardUserDefaults] objectForKey:name];
    
    // Count views
    NSArray*    views;
    int         i, count;
    views = [self subviews];
    count = MIN([viewRects count], [views count]);
    
    for (i = 0; i < count; i++) {
        // Get subview
        NSView* view;
        view = [views objectAtIndex:i];
        
        // Set subview frame saize
        NSRect  frame;
        frame = NSRectFromString([viewRects objectAtIndex:i]);
        
        // For zero rect
        if (NSIsEmptyRect(frame)) {
            frame = [view frame];
            if ([self isVertical]) {
                frame.size.width = 0;
            }
            else {
                frame.size.height = 0;
            }
        }
        
        [view setFrame:frame];
    }
}

@end

#pragma mark -

@implementation NSTableView (HMDTColumn)

- (unsigned int)indexOfTableColumn:(NSTableColumn*)tableColumn
{
	// Filter
	if (!tableColumn) {
		return NSNotFound;
	}
	
	return [[self tableColumns] indexOfObject:tableColumn];
}

- (unsigned int)indexOfTableColumnWithIdentifier:(id)identifier
{
	NSTableColumn *tableColumn;
	tableColumn = [self tableColumnWithIdentifier:identifier];
	
	return [self indexOfTableColumn:tableColumn];
}

- (void)removeAllTableColumns
{
    NSArray*        columns;
    NSEnumerator*   enumerator;
    NSTableColumn*  column;
    columns = [self tableColumns];
    enumerator = [columns objectEnumerator];
    while (column = [enumerator nextObject]) {
        [self removeTableColumn:column];
    }
}

- (NSArray*)tableColumnInfo
{
    NSMutableArray* identifierArray;
    NSMutableArray* widthArray;
    identifierArray = [NSMutableArray array];
    widthArray = [NSMutableArray array];
    
    // Store column info
    NSArray*        tableColumns;
    NSEnumerator*   enumerator;
    NSTableColumn*  tableColumn;
    tableColumns = [self tableColumns];
    enumerator = [tableColumns objectEnumerator];
    while (tableColumn = [enumerator nextObject]) {
        [identifierArray addObject:[tableColumn identifier]];
        [widthArray addObject:[NSNumber numberWithFloat:[tableColumn width]]];
    }
    
    return [NSArray arrayWithObjects:
            identifierArray, widthArray, nil];
}

- (void)setTableColumnInfo:(NSArray*)info
{
    // Check argument
    if (!info || [info count] != 2) {
        return;
    }
    
    // Get identifier and width
    NSArray*    identifierArray;
    NSArray*    widthArray;
    identifierArray = [info objectAtIndex:0];
    widthArray = [info objectAtIndex:1];
    if (![identifierArray isKindOfClass:[NSArray class]] || 
        ![widthArray isKindOfClass:[NSArray class]] || 
        [identifierArray count] != [widthArray count])
    {
        return;
    }
    
    // Set column info
    int i;
    for (i = 0; i < [identifierArray count]; i++) {
        // Get table column
        NSString*       identifier;
        NSTableColumn*  tableColumn;
        identifier = [identifierArray objectAtIndex:i];
        tableColumn = [self tableColumnWithIdentifier:identifier];
        if (!tableColumn) {
            continue;
        }
        
        // Check order
        int index;
        index = [self columnWithIdentifier:identifier];
        if (index != i) {
            if (i >= [self numberOfColumns]) {
                continue;
            }
            
            // Move column
            [self moveColumn:index toColumn:i];
        }
        
        // Check width
        float   width;
        width = [[widthArray objectAtIndex:i] floatValue];
        if ([tableColumn width] != width) {
            [tableColumn setWidth:width];
        }
    }
}

- (NSArray*)tableColumnsWidth
{
    NSMutableArray* identifierArray;
    NSMutableArray* widthArray;
    identifierArray = [NSMutableArray array];
    widthArray = [NSMutableArray array];
    
    // Store columns width
    NSArray*        tableColumns;
    NSEnumerator*   enumerator;
    NSTableColumn*  tableColumn;
    tableColumns = [self tableColumns];
    enumerator = [tableColumns objectEnumerator];
    while (tableColumn = [enumerator nextObject]) {
        [identifierArray addObject:[tableColumn identifier]];
        [widthArray addObject:[NSNumber numberWithFloat:[tableColumn width]]];
    }
    
    return [NSArray arrayWithObjects:
            identifierArray, widthArray, nil];
}

- (void)setTableColumnsWidth:(NSArray*)columnsWidth
{
    // Check argument
    if (!columnsWidth || [columnsWidth count] != 2) {
        return;
    }
    
    // Get identifier and width
    NSArray*    identifierArray;
    NSArray*    widthArray;
    identifierArray = [columnsWidth objectAtIndex:0];
    widthArray = [columnsWidth objectAtIndex:1];
    if (![identifierArray isKindOfClass:[NSArray class]] || 
        ![widthArray isKindOfClass:[NSArray class]] || 
        [identifierArray count] != [widthArray count])
    {
        return;
    }
    
    // Set columns width
    int i;
    for (i = 0; i < [identifierArray count]; i++) {
        // Get table column
        NSString*       identifier;
        NSTableColumn*  tableColumn;
        identifier = [identifierArray objectAtIndex:i];
        tableColumn = [self tableColumnWithIdentifier:identifier];
        if (!tableColumn) {
            continue;
        }
        
        // Set width
        float   width;
        width = [[widthArray objectAtIndex:i] floatValue];
        if ([tableColumn width] != width) {
            [tableColumn setWidth:width];
        }
    }
}

- (NSArray*)tableColumnsOrder
{
    NSMutableArray* identifierArray;
    identifierArray = [NSMutableArray array];
    
    // Store columns order
    NSArray*        tableColumns;
    NSEnumerator*   enumerator;
    NSTableColumn*  tableColumn;
    tableColumns = [self tableColumns];
    enumerator = [tableColumns objectEnumerator];
    while (tableColumn = [enumerator nextObject]) {
        [identifierArray addObject:[tableColumn identifier]];
    }
    
    return identifierArray;
}

- (void)setTableColumnsOrder:(NSArray*)columnsOrder
{
    // Check arguments
    if (!columnsOrder) {
        return;
    }
    
    // Set columns order
    int i;
    for (i = 0; i < [columnsOrder count]; i++) {
        // Get table column
        NSString*       identifier;
        NSTableColumn*  tableColumn;
        identifier = [columnsOrder objectAtIndex:i];
        tableColumn = [self tableColumnWithIdentifier:identifier];
        if (!tableColumn) {
            continue;
        }
        
        // Check order
        int index;
        index = [self columnWithIdentifier:identifier];
        if (index != i) {
            // Move column to last
            [self moveColumn:index toColumn:[self numberOfColumns] - 1];
        }
    }
}

@end

@implementation NSTableView (ContextMenu)

- (NSMenu*)menuForEvent:(NSEvent*)event
{
    if ([[self delegate] respondsToSelector:@selector(tableView:menuForEvent:)]) {
        return [[self delegate] tableView:self menuForEvent:event];
    }
    
    return nil;
}

@end

@implementation NSTableView (Dragging)

- (void)draggedImage:(NSImage*)image 
        endedAt:(NSPoint)point 
        operation:(NSDragOperation)operation
{
    if ([[self delegate] respondsToSelector:@selector(draggedImage:endedAt:operation:)]) {
        [[self delegate] draggedImage:image endedAt:point operation:operation];
    }
}

@end

#pragma mark -

@implementation NSTabView (HMDTTabIdentifier)

- (int)indexOfSelectedTab
{
    return [self indexOfTabViewItem:[self selectedTabViewItem]];
}

- (NSArray*)tabIdentifiers
{
    // Get tab view items
    NSArray*    tabViewItems;
    tabViewItems = [self tabViewItems];
    if (!tabViewItems) {
        return nil;
    }
    
    // Correct tab identifiers
    NSMutableArray* tabIdentifiers;
    NSEnumerator*   enumerator;
    NSTabViewItem*  tabViewItem;
    tabIdentifiers = [NSMutableArray array];
    enumerator = [tabViewItems objectEnumerator];
    while (tabViewItem = [enumerator nextObject]) {
        [tabIdentifiers addObject:[tabViewItem identifier]];
    }
    
    return tabIdentifiers;
}

- (NSTabViewItem*)tabViewItemWithIdentifier:(id)identifier
{
    int index;
    index = [self indexOfTabViewItemWithIdentifier:identifier];
    if (index == NSNotFound) {
        return nil;
    }
    
    return [self tabViewItemAtIndex:index];
}

@end

#pragma mark -

@implementation NSToolbar (ToolbarItem)

- (NSToolbarItem*)toolbarItemWithIdentifier:(id)identifier
{
    NSArray*        items;
    NSEnumerator*   enumerator;
    NSToolbarItem*  item;
    items = [self items];
    enumerator = [items objectEnumerator];
    while (item = [enumerator nextObject]) {
        if ([[item itemIdentifier] isEqual:identifier]) {
            return item;
        }
    }
    
    return nil;
}

@end
