/*
HMBrowserCell.m

Author: Makoto Kinoshita

Copyright 2004-2006 The Shiira Project. All rights reserved.

Redistribution and use in source and binary forms, with or without modification, are permitted 
provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice, this list of conditions 
  and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright notice, this list of 
  conditions and the following disclaimer in the documentation and/or other materials provided 
  with the distribution.

THIS SOFTWARE IS PROVIDED BY THE SHIIRA PROJECT ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, 
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE SHIIRA PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
POSSIBILITY OF SUCH DAMAGE.
*/

#import "HMBrowserCell.h"

static int  HMMarginImage = 4;
static int  HMMarginBetweenImageAndText = 4;

@implementation HMBrowserCell

//--------------------------------------------------------------//
#pragma mark -- Initialize --
//--------------------------------------------------------------//

- (id)init
{
    self = [super init];
    if (!self) {
        return nil;
    }
    
    // Initialize instance variables
    _height = 19.0f;
    
    return self;
}

//--------------------------------------------------------------//
#pragma mark -- Cell size --
//--------------------------------------------------------------//

- (NSSize)cellSize
{
    NSSize  size;
    size = [super cellSize];
    size.height = _height;
    
    return size;
}

//--------------------------------------------------------------//
#pragma mark -- Drawing --
//--------------------------------------------------------------//

- (void)drawInteriorWithFrame:(NSRect)cellFrame inView:(NSView*)controlView
{
    // First, inovke super to load itself without drawing
    [[NSGraphicsContext currentContext] saveGraphicsState];
    [NSBezierPath clipRect:NSZeroRect];
    [super drawInteriorWithFrame:cellFrame inView:controlView];
    [[NSGraphicsContext currentContext] restoreGraphicsState];
    
    //
    // Draw highlight
    //
    if ([self state] == NSOnState || [self isHighlighted]) {
        NSColor*    highlightColor;
        highlightColor = [self highlightColorInView:controlView];
        [highlightColor set];
        NSRectFill(cellFrame);
    }
    
    // Get branch image
    NSImage*    branchImage = nil;
    if (![self isLeaf]) {
        if ([self state] == NSOnState || [self isHighlighted]) {
            branchImage = [NSBrowserCell highlightedBranchImage];
        }
        else {
            branchImage = [NSBrowserCell branchImage];
        }
    }
    
    //
    // Draw image
    //
    NSImage*    image;
    image = [self image];
    
    // Calc image and text frame
    NSRect  imageFrame, textFrame, branchFrame;
    if (image) {
        if (branchImage) {
            NSSize  branchSize;
            branchSize = [branchImage size];
            
            NSDivideRect(
                    cellFrame, &imageFrame, &textFrame, 
                    HMMarginImage + HMMarginBetweenImageAndText + [image size].width, NSMinXEdge);
            NSDivideRect(
                    textFrame, &branchFrame, &textFrame, 
                    branchSize.width + 4, NSMaxXEdge);
            
            branchFrame.origin.x += 2;
            branchFrame.origin.y += floor((branchFrame.size.height - branchSize.height) / 2);
            branchFrame.size = branchSize;
        }
        else {
            NSDivideRect(
                    cellFrame, &imageFrame, &textFrame, 
                    HMMarginImage + HMMarginBetweenImageAndText + [image size].width, NSMinXEdge);
        }
    }
    else {
        imageFrame = NSZeroRect;
        textFrame = cellFrame;
    }
    
    // Draw image
    if (image) {
        // Decide image frame
        NSSize	size;
        NSRect	frame;
        size = [image size];
        frame.origin = imageFrame.origin;
        frame.origin.x += HMMarginImage;
        frame.size = size;
        if ([controlView isFlipped]) {
            frame.origin.y += floor((cellFrame.size.height + frame.size.height) / 2);
        }
        else {
            frame.origin.y += floor((cellFrame.size.height - frame.size.height) / 2);
        }
        
        // Draw image
        [image compositeToPoint:frame.origin operation:NSCompositeSourceOver];
    }
    
    //
    // Draw text
    //
    NSAttributedString* attrStr;
    attrStr = [self attributedStringValue];
    if (attrStr) {
        NSSize  size;
        size = [attrStr size];
        
        textFrame.origin.y += floor((textFrame.size.height - size.height) / 2);
        textFrame.size.height = size.height;
        [attrStr drawInRect:textFrame];
    }
    
    //
    // Draw branch
    //
    if (branchImage) {
        NSRect  srcRect;
        srcRect.origin = NSZeroPoint;
        srcRect.size = [branchImage size];
        [branchImage drawInRect:branchFrame fromRect:srcRect operation:NSCompositeSourceOver fraction:1.0f];
    }
}

@end
