/*
SRSourceWindowController.m

Author: Makoto Kinoshita

Copyright 2004-2006 The Shiira Project. All rights reserved.

Redistribution and use in source and binary forms, with or without modification, are permitted 
provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice, this list of conditions 
  and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright notice, this list of 
  conditions and the following disclaimer in the documentation and/or other materials provided 
  with the distribution.

THIS SOFTWARE IS PROVIDED BY THE SHIIRA PROJECT ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, 
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE SHIIRA PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
POSSIBILITY OF SUCH DAMAGE.
*/

#import "SRSourceWindowController.h"

#import "SRPrefDefaultKeys.h"

// Frame auto save name
NSString*   SRSourceWindowFrameAutoSaveName = @"SRSourceWindowFrameAutoSaveName";

@implementation SRSourceWindowController

//--------------------------------------------------------------//
#pragma mark -- Initialize --
//--------------------------------------------------------------//

- (id)init
{
    self = [super initWithWindowNibName:@"Source"];
    if (!self) {
        return nil;
    }
    
    // Initialize instance variables
    _httpHeaders = [[NSMutableDictionary dictionary] retain];
    _domNodes = [[NSMutableArray array] retain];
    
    // Load window
    [self loadWindow];
    
    return self;
}

- (void)awakeFromNib
{
    NSUserDefaults* defaults;
    defaults = [NSUserDefaults standardUserDefaults];
    
    // Configure window
    NSWindow*   window;
    window = [self window];
    [window setFrameAutosaveName:SRSourceWindowFrameAutoSaveName];
    
    // Set back color
    NSData*     data;
    NSColor*    backColor = nil;
    data = [defaults objectForKey:SRSourceBackgroundColor];
    if (data) {
        backColor = [NSUnarchiver unarchiveObjectWithData:data];
    }
    if (!backColor) {
        backColor = [NSColor whiteColor];
    }
    [_htmlTextView setBackgroundColor:backColor];
    [_domTextView setBackgroundColor:backColor];
    [_xmlTextView setBackgroundColor:backColor];
    
    // Select source style
    [self selectSourceStyleAction:_typePopUp];
}

- (void)dealloc
{
    [_httpHeaders release], _httpHeaders = nil;
    
    [super dealloc];
}

//--------------------------------------------------------------//
#pragma mark -- Page source --
//--------------------------------------------------------------//

- (NSArray*)_createFieldItemsWithHeaderFields:(NSDictionary*)headerFields
{
    NSMutableArray*    items;
    items = [NSMutableArray array];
    
    NSEnumerator*   enumerator;
    NSString*       key;
    enumerator = [headerFields keyEnumerator];
    while (key = [enumerator nextObject]) {
        NSString*   value;
        value = [headerFields objectForKey:key];
        if (!value) {
            continue;
        }
        
        NSDictionary*   item;
        item = [NSDictionary dictionaryWithObjectsAndKeys:
                key, @"key", 
                value, @"value", nil];
        
        [items addObject:item];
    }
    
    return items;
}

- (NSAttributedString*)_colorizedSourceOfString:(NSString*)source
{
    if (!source) {
        return nil;
    }
    
    NSUserDefaults* defaults;
    defaults = [NSUserDefaults standardUserDefaults];
    
    // Get colors
    static NSColor* _textColor = nil;
    static NSColor* _tagColor = nil;
    static NSColor* _commentColor = nil;
    if (!_textColor) {
        _textColor = [[NSColor blackColor] retain];
        _tagColor = [[NSColor colorWithDeviceRed:0.1 green:0.1 blue:0.648 alpha:1.0] retain];
        _commentColor = [[NSColor colorWithDeviceRed:0.632 green:0 blue:0 alpha:1.0] retain];
    }
    
    NSData*     data;
    NSColor*    textColor = nil;
    NSColor*    tagColor = nil;
    NSColor*    commentColor = nil;
    data = [defaults objectForKey:SRSourceDefaultColor];
    if (data) {
        textColor = [NSUnarchiver unarchiveObjectWithData:data];
    }
    if (!textColor) {
        textColor = _textColor;
    }
    
    data = [defaults objectForKey:SRSourceTagColor];
    if (data) {
        tagColor = [NSUnarchiver unarchiveObjectWithData:data];
    }
    if (!tagColor) {
        tagColor = _tagColor;
    }
    
    data = [defaults objectForKey:SRSourceCommentColor];
    if (data) {
        commentColor = [NSUnarchiver unarchiveObjectWithData:data];
    }
    if (!commentColor) {
        commentColor = _commentColor;
    }
    
    // Get font
    NSString*   fontName;
    int         fontSize;
    NSFont*     font;
    fontName = [defaults stringForKey:SRSourceFontName];
    if (!fontName) {
        fontName = @"Monaco";
    }
    fontSize = [defaults integerForKey:SRSourceFontSize];
    if (fontSize < 4) {
        fontSize = 12;
    }
    
    font = [NSFont fontWithName:fontName size:fontSize];
    
    // Create attributed string
    NSMutableAttributedString*  attrString;
    attrString = [[NSMutableAttributedString alloc] initWithString:source];
    [attrString autorelease];
    [attrString addAttribute:NSForegroundColorAttributeName 
            value:textColor range:NSMakeRange(0, [attrString length])];
    
    // Scan source
    NSScanner*  scanner;
    scanner = [NSScanner scannerWithString:source];
    while (![scanner isAtEnd]) {
        // Find next '<' character
        int tagStart, tagEnd;
        [scanner scanUpToString:@"<" intoString:nil];
        tagStart = [scanner scanLocation];		
        
        // For '<!--'
        if ([source length] > tagStart + 6 && 
            [[source substringWithRange:NSMakeRange(tagStart + 1, 3)] isEqualToString:@"!--"])
        {
            // Find '-->'
            if ([scanner scanUpToString:@"-->" intoString:nil]) {
                tagEnd = [scanner scanLocation] + 3;
                if ([source length] > tagEnd) {
                    [attrString addAttribute:NSForegroundColorAttributeName 
                            value:commentColor range:NSMakeRange(tagStart, tagEnd - tagStart)];
                }
            }
        }
        else if ([source length] > tagStart) {
            // Find '>'
            if ([scanner scanUpToString:@">" intoString:nil]) {
                tagEnd = [scanner scanLocation] + 1;
                [attrString addAttribute:NSForegroundColorAttributeName
                        value:tagColor range:NSMakeRange(tagStart, tagEnd - tagStart)];
            }
        }
    }
    
    // Set font
    [attrString addAttribute:NSFontAttributeName
            value:font range:NSMakeRange(0, [attrString length])];
    
    return attrString;
}

- (void)_setHTMLSource:(NSString*)source
{
    // Check argument
    if (!source) {
        return;
    }
    
    // Get colorized source string
    NSAttributedString* attrString;
    attrString = [self _colorizedSourceOfString:source];
    if (!attrString) {
        return;
    }
    
    // Set attributed string
    [[_htmlTextView textStorage] setAttributedString:attrString];
    [_htmlTextView setSelectedRange:NSMakeRange(0, 0)];
}

- (void)_copyDOMNode:(DOMNode*)node inArray:(NSMutableArray*)array
{
    // Copy DOM node attributes to dictionary
    NSMutableDictionary*    dict;
    dict = [NSMutableDictionary dictionary];
    
    NSString*   nodeName;
    if (nodeName = [node nodeName]) {
        [dict setObject:nodeName forKey:@"name"];
    }
    
    [dict setObject:node forKey:@"node"];
    
    // Copy children
    if ([node hasChildNodes]) {
        NSMutableArray* childrenArray;
        childrenArray = [NSMutableArray array];
        
        // Get children
        DOMNodeList*    children;
        children = [node childNodes];
        
        unsigned long   i;
        for (i = 0; i < [children length]; i++) {
            // Get child
            DOMNode*    child;
            child = [children item:i];
            
            // Copy child
            [self _copyDOMNode:child inArray:childrenArray];
        }
        
        [dict setObject:childrenArray forKey:@"children"];
    }
    
    // Add copied node
    [array addObject:dict];
}

- (void)_setDOMDocument:(DOMDocument*)domDocument
{
    // Copy DOM document
    [_domNodes removeAllObjects];
    [self _copyDOMNode:domDocument inArray:_domNodes];
    
    // Reload data
    [_domOutline reloadData];
}

- (void)_setDOMSource:(NSString*)source
{
    // Check argument
    if (!source) {
        return;
    }
    
    // Get colorized source string
    NSAttributedString* attrString;
    attrString = [self _colorizedSourceOfString:source];
    if (!attrString) {
        return;
    }
    
    // Set attributed string
    [[_domTextView textStorage] setAttributedString:attrString];
    [_domTextView setSelectedRange:NSMakeRange(0, 0)];
}

- (void)setWebDataSource:(WebDataSource*)dataSource
{
    // Get request header fields
    NSURLRequest*   request;
    NSDictionary*   headerFields;
    request = [dataSource request];
    headerFields = [request allHTTPHeaderFields];
    if (headerFields) {
        [_httpHeaders setObject:[self _createFieldItemsWithHeaderFields:headerFields] 
                forKey:@"Request"];
    }
    
    // Get response header fields
    NSURLResponse*  response;
    response = [dataSource response];
    if ([response respondsToSelector:@selector(allHeaderFields)]) {
        headerFields = [(NSHTTPURLResponse*)response allHeaderFields];
        if (headerFields) {
            [_httpHeaders setObject:[self _createFieldItemsWithHeaderFields:headerFields] 
                    forKey:@"Response"];
        }
    }
    
    // Reload data
    [_httpHeaderOutline reloadData];
    
    // Get HTML source
    NSString*   source;
    source = [[dataSource representation] documentSource];
    if (source) {
        [self _setHTMLSource:source];
    }
    
    // Get DOM document
    DOMDocument*    domDocument;
    domDocument = [[dataSource webFrame] DOMDocument];
    if (domDocument) {
        [self _setDOMDocument:domDocument];
    }
}

- (void)setXMLDocument:(NSXMLDocument*)document
{
}

//--------------------------------------------------------------//
#pragma mark -- Appearance --
//--------------------------------------------------------------//

- (void)_updateDOMText
{
    // Get selected item
    id  item;
    item = [_domOutline itemAtRow:[_domOutline selectedRow]];
    if (!item) {
        [self _setDOMSource:@""];
        return;
    }
    
    // Get node
    id  node = nil;
    if ([item respondsToSelector:@selector(objectForKey:)]) {
        node = [item objectForKey:@"node"];
    }
    if (!node) {
        [self _setDOMSource:@""];
        return;
    }
    
    // Set outer HTML text
    if ([node respondsToSelector:@selector(outerHTML)]) {
        NSString*   outerHTML;
        outerHTML = [(DOMHTMLElement*)node outerHTML];
        if (!outerHTML) {
            outerHTML = @"";
        }
        
        [self _setDOMSource:outerHTML];
        return;
    }
    
    // Set character data
    if ([node respondsToSelector:@selector(data)]) {
        NSString*   data;
        data = [(DOMCharacterData*)node data];
        if (!data) {
            data = @"";
        }
        
        [self _setDOMSource:data];
        return;
    }
    
    // Set XML string
    if ([node respondsToSelector:@selector(XMLString)]) {
        NSString*   data;
        data = [node XMLStringWithOptions:NSXMLNodePrettyPrint];
        if (!data) {
            data = @"";
        }
        
        [self _setDOMSource:data];
        return;
    }
    
    [self _setDOMSource:@""];
}

//--------------------------------------------------------------//
#pragma mark -- Actions --
//--------------------------------------------------------------//

- (void)selectSourceStyleAction:(id)sender
{
    // Get selected tag
    int tag;
    tag = [[sender selectedItem] tag];
    
    // Select tab
    [_tabView selectTabViewItemAtIndex:tag];
}

- (void)outlineSelectedAction:(id)sender
{
    // For DOM outline
    if (sender == _domOutline) {
        [self _updateDOMText];
    }
}

//--------------------------------------------------------------//
#pragma mark -- NSOutlineView data source --
//--------------------------------------------------------------//

- (int)outlineView:(NSOutlineView*)outlineView 
        numberOfChildrenOfItem:(id)item
{
    // For HTTP headers
    if (outlineView == _httpHeaderOutline) {
        if (!item) {
            return [_httpHeaders count];
        }
        if ([item isKindOfClass:[NSString class]]) {
            return [[_httpHeaders objectForKey:item] count];
        }
    }
    
    // For DOM
    if (outlineView == _domOutline) {
        if (!item) {
            return [_domNodes count];
        }
        
        id  children;
        children = [item objectForKey:@"children"];
        if ([children respondsToSelector:@selector(count)]) {
            return [children count];
        }
    }
    
    return 0;
}

- (id)outlineView:(NSOutlineView*)outlineView 
        child:(int)index 
        ofItem:(id)item
{
    // For HTTP headers
    if (outlineView == _httpHeaderOutline) {
        if (!item) {
            // Get request header fields
            NSDictionary*   headerFields;
            headerFields = [_httpHeaders objectForKey:@"Request"];
            
            if (index == 0 && headerFields) {
                return @"Request";
            }
            return @"Response";
        }
        if ([item isKindOfClass:[NSString class]]) {
            return [[_httpHeaders objectForKey:item] objectAtIndex:index];
        }
    }
    
    // For DOM
    if (outlineView == _domOutline) {
        if (!item) {
            return [_domNodes objectAtIndex:index];
        }
        
        id  children;
        children = [item objectForKey:@"children"];
        if ([children respondsToSelector:@selector(objectAtIndex:)]) {
            return [children objectAtIndex:index];
        }
    }
    
    return nil;
}

- (BOOL)outlineView:(NSOutlineView*)outlineView 
        isItemExpandable:(id)item
{
    // For HTTP headers
    if (outlineView == _httpHeaderOutline) {
        if ([item isKindOfClass:[NSString class]]) {
            return YES;
        }
        return NO;
    }
    
    // For DOM
    if (outlineView == _domOutline) {
        id  children;
        children = [item objectForKey:@"children"];
        if ([children respondsToSelector:@selector(count)]) {
            return [children count] > 0;
        }
    }
    
    return NO;
}

- (id)outlineView:(NSOutlineView*)outlineView 
        objectValueForTableColumn:(NSTableColumn*)tableColumn 
        byItem:(id)item
{
    // Get identifier
    id  identifier;
    identifier = [tableColumn identifier];
    
    // For HTTP headers
    if (outlineView == _httpHeaderOutline) {
        if ([item isKindOfClass:[NSString class]]) {
            if ([identifier isEqualToString:@"key"]) {
                return item;
            }
            return nil;
        }
        return [item objectForKey:identifier];
    }
    
    // For DOM
    if (outlineView == _domOutline) {
        if ([item respondsToSelector:@selector(objectForKey:)]) {
            return [item objectForKey:identifier];
        }
    }
    
    return nil;
}

//--------------------------------------------------------------//
#pragma mark -- NSOutlineView delegate --
//--------------------------------------------------------------//

- (void)outlineViewSelectionDidChange:(NSNotification*)notification
{
    id  object;
    object = [notification object];
    
    // For DOM outline
    if (object == _domOutline) {
        [self _updateDOMText];
    }
}

@end
