/*
SRSearchField.m

Author: Makoto Kinoshita

Copyright 2004 The Shiira Project. All rights reserved.

Redistribution and use in source and binary forms, with or without modification, are permitted 
provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice, this list of conditions 
  and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright notice, this list of 
  conditions and the following disclaimer in the documentation and/or other materials provided 
  with the distribution.

THIS SOFTWARE IS PROVIDED BY THE SHIIRA PROJECT ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, 
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE SHIIRA PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
POSSIBILITY OF SUCH DAMAGE.
*/

#import "SRDefaultsKey.h"

#import "SRSearchField.h"

#import "AppKitEx.h"

int  SRSearchFieldMinWidth = 80;
int  SRSearchFieldMaxWidth = 480;

static int  _SRSearcFieldTrackingWidth = 5;

@implementation SRSearchField

+ (Class)cellClass
{
    return [SRSearchFieldCell class];
}

- (id)initWithFrame:(NSRect)frame
{
    self = [super initWithFrame:frame];
    if (!self) {
        return nil;
    }
    
    // Initialize instance variables
    _leftTrackingRectTag = -1;
    _rightTrackingRectTag = -1;
    
    return self;
}

- (BOOL)isFlexible
{
    return _isFlexible;
}

- (void)setFlexible:(BOOL)flag
{
    _isFlexible = flag;
}

- (NSRect)leftTrackingArea
{
    NSRect  frame;
    frame = [self frame];
    return NSMakeRect(0, 0, _SRSearcFieldTrackingWidth, frame.size.height);
}

- (NSRect)rightTrackingArea
{
    NSRect  frame;
    frame = [self frame];
    return NSMakeRect(
            frame.size.width - _SRSearcFieldTrackingWidth, 0, _SRSearcFieldTrackingWidth, frame.size.height);
}

- (void)updateTrackingArea
{
    if (!_isFlexible) {
        return;
    }
    
    // Remove old tracking rect
    if (_leftTrackingRectTag > 0) {
        [self removeTrackingRect:_leftTrackingRectTag];
    }
    if (_rightTrackingRectTag > 0) {
        [self removeTrackingRect:_rightTrackingRectTag];
    }
    
    // Add tracking rect
    _leftTrackingRectTag = [self addTrackingRect:[self leftTrackingArea] 
            owner:self 
            userData:&_leftTrackingRectTag 
            assumeInside:NO];
    _rightTrackingRectTag = [self addTrackingRect:[self rightTrackingArea] 
            owner:self 
            userData:&_rightTrackingRectTag 
            assumeInside:NO];
}

- (void)resetCursorRects
{
    if (!_isFlexible) {
        return;
    }
    
    [self addCursorRect:[self leftTrackingArea] cursor:[NSCursor resizeLeftRightCursor]];
    [self addCursorRect:[self rightTrackingArea] cursor:[NSCursor resizeLeftRightCursor]];
}

- (void)mouseDown:(NSEvent*)event
{
    if (!_isFlexible) {
        [super mouseDown:event];
        return;
    }
    
    // Get frame
    NSRect  frame;
    frame = [self frame];
    
    // Get mouse location
    NSPoint mouseLoc;
    mouseLoc = [self convertPoint:[event locationInWindow] fromView:nil];
    
    // Track mouse
    NSRect  leftTrackingArea, rightTrackingArea;
    BOOL    inLeft, inRight;
    leftTrackingArea = [self leftTrackingArea];
    rightTrackingArea = [self rightTrackingArea];
    inLeft = [self mouse:mouseLoc inRect:leftTrackingArea];
    inRight = [self mouse:mouseLoc inRect:rightTrackingArea];
    if (inLeft || inRight) {
        // Event loop
        while (YES) {
            // Wait next event
            NSEvent*    waitingEvent;
            waitingEvent = [NSApp nextEventMatchingMask:(NSLeftMouseDraggedMask | NSLeftMouseUpMask) 
                    untilDate:[NSDate distantFuture] 
                    inMode:NSEventTrackingRunLoopMode 
                    dequeue:YES];
            if (!waitingEvent) {
                return;
            }
            
            // For dragging
            if ([waitingEvent type] == NSLeftMouseDragged) {
                // Calc dragging distant
                NSPoint newMouseLoc;
                int     delta = 0;
                int     newWidth;
                newMouseLoc = [self convertPoint:[waitingEvent locationInWindow] fromView:nil];
                if (inLeft) {
                    delta = [event locationInWindow].x - [waitingEvent locationInWindow].x;
                }
                if (inRight) {
                    delta = [waitingEvent locationInWindow].x - [event locationInWindow].x;
                }
                newWidth = frame.size.width + delta;
                
                if (delta != 0) {
                    if (newWidth < SRSearchFieldMinWidth) {
                        newWidth = SRSearchFieldMinWidth;
                    }
                    if (newWidth > SRSearchFieldMaxWidth) {
                        newWidth = SRSearchFieldMaxWidth;
                    }
                    
                    // Set size
                    NSToolbarItem*  item;
                    item = [[[self window] toolbar] toolbarItemWithIdentifier:@"SRSearch"];
                    if (item) {
                        NSSize  size;
                        size = NSMakeSize(newWidth, frame.size.height);
                        [item setMinSize:size];
                        [item setMaxSize:size];
                    }
                    
                    // Update tracking area
                    [self updateTrackingArea];
                    
                    // Store width
                    [[NSUserDefaults standardUserDefaults] setInteger:newWidth forKey:SRSearchFieldWidth];
                }
                
                continue;
            }
            
            // For mouse up
            if ([waitingEvent type] == NSLeftMouseUp) {
                return;
            }
        }
    }
    
    [super mouseDown:event];
}

@end

#pragma mark -

@implementation SRSearchFieldCell

- (NSRect)searchButtonRectForBounds:(NSRect)rect
{
    // Move 1 pixel to right
    NSRect  searchButtonRect;
    searchButtonRect = [super searchButtonRectForBounds:rect];
    searchButtonRect.origin.x += 2;
    
    return searchButtonRect;
}

@end
