// <file>
//     <copyright see="prj:///doc/copyright.txt"/>
//     <license see="prj:///doc/license.txt"/>
//     <owner name="none" email=""/>
//     <version>$Revision: 1963 $</version>
// </file>

using System;
using System.IO;
using System.Windows.Forms;

using ICSharpCode.Core;
using ICSharpCode.SharpDevelop;
using ICSharpCode.SharpDevelop.Project;
using NSvn.Common;
using NSvn.Core;

namespace ICSharpCode.Svn.Commands
{
	/// <summary>
	/// Description of AutostartCommands.
	/// </summary>
	public class RegisterEventsCommand : AbstractCommand
	{
		public override void Run()
		{
			FileService.FileRemoving += FileRemoving;
			FileService.FileRenaming += FileRenaming;
			
			FileUtility.FileSaved += new FileNameEventHandler(FileSaved);
			AbstractProjectBrowserTreeNode.AfterNodeInitialize += TreeNodeInitialized;
		}
		
		SvnProjectBrowserVisitor visitor = new SvnProjectBrowserVisitor();
		
		void TreeNodeInitialized(object sender, TreeViewEventArgs e)
		{
			AbstractProjectBrowserTreeNode node = e.Node as AbstractProjectBrowserTreeNode;
			node.AcceptVisitor(visitor, null);
		}
		
		internal static bool CanBeVersionControlledFile(string fileName)
		{
			return CanBeVersionControlledDirectory(Path.GetDirectoryName(fileName));
		}
		
		internal static bool CanBeVersionControlledDirectory(string directory)
		{
			return Directory.Exists(Path.Combine(directory, ".svn")) || Directory.Exists(Path.Combine(directory, "_svn"));
		}
		
		void FileSaved(object sender, FileNameEventArgs e)
		{
			ProjectBrowserPad pad = ProjectBrowserPad.Instance;
			if (pad == null) return;
			string fileName = e.FileName;
			if (!CanBeVersionControlledFile(fileName)) return;
			FileNode node = pad.ProjectBrowserControl.FindFileNode(fileName);
			if (node == null) return;
			OverlayIconManager.Enqueue(node);
			SubversionStateCondition.ResetCache();
		}
		
		void FileAdded(object sender, FileEventArgs e)
		{
			try {
				if (AddInOptions.AutomaticallyAddFiles) {
					if (!CanBeVersionControlledFile(e.FileName)) return;
					SvnClient.Instance.Client.Add(Path.GetFullPath(e.FileName), Recurse.None);
				}
			} catch (Exception ex) {
				MessageService.ShowError("File added exception: " + ex);
			}
		}
		
		void FileRemoving(object sender, FileCancelEventArgs e)
		{
			if (e.Cancel) return;
			string fullName = Path.GetFullPath(e.FileName);
			if (!CanBeVersionControlledFile(fullName)) return;
			if (e.IsDirectory) {
				// show "cannot delete directories" message even if
				// AutomaticallyDeleteFiles (see below) is off!
				Status status = SvnClient.Instance.Client.SingleStatus(fullName);
				switch (status.TextStatus) {
					case StatusKind.None:
					case StatusKind.Unversioned:
						break;
					default:
						MessageService.ShowMessage("SubversionAddIn cannot delete directories, the directory is only removed from the project.");
						e.OperationAlreadyDone = true;
						break;
				}
				return;
			}
			if (!AddInOptions.AutomaticallyDeleteFiles) return;
			try {
				Status status = SvnClient.Instance.Client.SingleStatus(fullName);
				switch (status.TextStatus) {
					case StatusKind.None:
					case StatusKind.Unversioned:
					case StatusKind.Deleted:
						return; // nothing to do
					case StatusKind.Normal:
						// remove without problem
						break;
					case StatusKind.Modified:
					case StatusKind.Replaced:
						if (MessageService.AskQuestion("The file has local modifications. Do you really want to remove it?")) {
							// modified files cannot be deleted, so we need to revert the changes first
							SvnClient.Instance.Client.Revert(new string[] { fullName }, e.IsDirectory ? Recurse.Full : Recurse.None);
						} else {
							e.Cancel = true;
							return;
						}
						break;
					case StatusKind.Added:
						if (status.Copied) {
							if (!MessageService.AskQuestion("The file has just been moved to this location, do you really want to remove it?")) {
								e.Cancel = true;
								return;
							}
						}
						SvnClient.Instance.Client.Revert(new string[] { fullName }, e.IsDirectory ? Recurse.Full : Recurse.None);
						return;
					default:
						MessageService.ShowError("The file/directory cannot be removed because it is in subversion status '" + status.TextStatus + "'.");
						e.Cancel = true;
						return;
				}
				SvnClient.Instance.Client.Delete(new string [] { fullName }, true);
				e.OperationAlreadyDone = true;
			} catch (Exception ex) {
				MessageService.ShowError("File removed exception: " + ex);
			}
		}
		
		void FileRenaming(object sender, FileRenamingEventArgs e)
		{
			if (e.Cancel) return;
			string fullSource = Path.GetFullPath(e.SourceFile);
			if (!CanBeVersionControlledFile(fullSource)) return;
			try {
				Status status = SvnClient.Instance.Client.SingleStatus(fullSource);
				switch (status.TextStatus) {
					case StatusKind.Unversioned:
					case StatusKind.None:
						return; // nothing to do
					case StatusKind.Normal:
					case StatusKind.Modified:
						// rename without problem
						break;
					case StatusKind.Added:
					case StatusKind.Replaced:
						if (status.Copied) {
							MessageService.ShowError("The file was moved/copied and cannot be renamed without losing it's history.");
							e.Cancel = true;
						} else if (e.IsDirectory) {
							goto default;
						} else {
							SvnClient.Instance.Client.Revert(new string[] { fullSource }, Recurse.None);
							FileService.FileRenamed += new AutoAddAfterRenameHelper(e).Renamed;
						}
						return;
					default:
						MessageService.ShowError("The file/directory cannot be renamed because it is in subversion status '" + status.TextStatus + "'.");
						e.Cancel = true;
						return;
				}
				SvnClient.Instance.Client.Move(fullSource,
				                               Path.GetFullPath(e.TargetFile),
				                               true
				                              );
				e.OperationAlreadyDone = true;
			} catch (Exception ex) {
				MessageService.ShowError("File renamed exception: " + ex);
			}
		}
		
		class AutoAddAfterRenameHelper
		{
			FileRenamingEventArgs args;
			
			public AutoAddAfterRenameHelper(FileRenamingEventArgs args) {
				this.args = args;
			}
			
			public void Renamed(object sender, FileRenameEventArgs e)
			{
				FileService.FileRenamed -= Renamed;
				if (args.Cancel || args.OperationAlreadyDone)
					return;
				if (args.SourceFile != e.SourceFile || args.TargetFile != e.TargetFile)
					return;
				SvnClient.Instance.Client.Add(e.TargetFile, Recurse.None);
			}
		}
	}
}
