/*
 *  linux/drivers/video/sh7785fb.c SH7785/SH7786 Display Unit frame buffer driver
 *
 *     Copyright (C) 2010 ALPHAPROJECT Co.,Ltd. 
 *     Copyright (C) 2002 Lineo Solutions, Inc.
 *
 *  This file is subject to the terms and conditions of the GNU General Public
 *  License. See the file COPYING in the main directory of this archive for
 *  more details.
 *
 * 0.01
 *  - initial beta version (ks)
 */

#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/string.h>
#include <linux/mm.h>
#include <linux/tty.h>
#include <linux/slab.h>
#include <linux/vmalloc.h>
#include <linux/delay.h>
#include <linux/fb.h>
#include <linux/init.h>
#include <linux/interrupt.h>
#include <linux/wait.h>
#include <linux/platform_device.h>
#include <linux/fcntl.h>
#include <linux/cdev.h>
#include <linux/ioctl.h>
#include <linux/dma-mapping.h>
#include <asm/io.h>
#include <asm/uaccess.h>

#include "sh7785fb.h"

//extern int  soft_cursor(struct fb_info *info, struct fb_cursor *cursor);

#define PFX "sh7785fb: "

#define	WAIT_TIMEOUT	1000
DECLARE_WAIT_QUEUE_HEAD(wait);

static struct fb_var_screeninfo sh7785fb_var __initdata = {
	.xres		= XRES,
	.yres		= YRES,
	.xres_virtual	= V_XRES,
	.yres_virtual	= V_YRES,
	.xoffset	= 0,
	.yoffset	= 0,
	.grayscale	= 0,
	.bits_per_pixel	= BPP,
	.red		= {11, 5, 0},
	.green		= { 5, 6, 0},
	.blue		= { 0, 5, 0},
	.transp		= {0, 0, 0},
	.nonstd		= 0,
	.activate	= FB_ACTIVATE_NOW,
	.height		= -1,
	.width		= -1,
	.accel_flags	= 0,
	.pixclock	= DOT_CLOCK,
	.left_margin	= H_BPORCH,
	.right_margin	= H_FPORCH,
	.upper_margin	= V_BPORCH,
	.lower_margin	= V_FPORCH,
	.hsync_len	= H_SYNC_W,
	.vsync_len	= V_SYNC_W,
	.sync		= 0,
	.vmode		= FB_VMODE_NONINTERLACED
};

static struct fb_fix_screeninfo sh7785fb_fix __initdata = {
	.id		= "SH7785FB", 
	.type		= FB_TYPE_PACKED_PIXELS,
	.xpanstep	= 0,
	.ypanstep	= 0,
	.ywrapstep	= 0,
	.type_aux	= 0,
	.visual		= FB_VISUAL_TRUECOLOR,
	.accel		= FB_ACCEL_NONE,
	.line_length	= XRES * BPP_BYTE,
	.smem_len	= MAX_FRAMEBUFFER_MEM_SIZE * MAX_BLINK_NUM,
};

static struct sh7785fb_par current_par[MAX_PLANE_NUM];
static struct sh7785fb_hw_par current_hw_par;
static unsigned int pseudo_palette[MAX_PLANE_NUM][16];

static struct fb_info fbinfo[MAX_PLANE_NUM];

static void sh7785fb_hw_set_var2par(struct fb_info *info);
static void sh7785fb_hw_load_par(struct sh7785fb_par *par);
static void sh7785fb_hw_load_hw_par(struct fb_info *info);
static void sh7785fb_hw_enable(struct sh7785fb_par *par);
static void sh7785fb_hw_disable(struct sh7785fb_par *par);

static int sh7785fb_check_var(struct fb_var_screeninfo *var, struct fb_info *info)
{
	int nom, den;

	if (var->bits_per_pixel != 16)
		var->bits_per_pixel = 16;

	switch (var->bits_per_pixel) {
	case 1 ... 8: /* Not Support */
		var->red.offset = var->green.offset = var->blue.offset = 0;
		var->red.length = var->green.length = var->blue.length = 8;
		var->bits_per_pixel = 8;
		nom = den = 1;
		break;
	case 9 ... 15:
		var->green.length = 5;
		/* fall through */
	case 16:
		var->bits_per_pixel = 16;
		if (var->green.length == 5) {
			/* 0rrrrrgg gggbbbbb */
			var->red.offset = 10;
			var->green.offset = 5;
			var->green.offset = 5;
			var->blue.offset = 0;
			var->red.length = 5;
			var->green.length = 5;
			var->blue.length = 5;
		} else {
			/* rrrrrggg gggbbbbb */
			var->red.offset = 11;
			var->green.offset = 5;
			var->blue.offset = 0;
			var->red.length = 5;
			var->green.length = 6;
			var->blue.length = 5;
		}
		nom = 2;
		den = 1;
		break;
	default:
		printk(KERN_ERR PFX
			"mode %dx%dx%d rejected...color depth not supported.\n",
			var->xres, var->yres, var->bits_per_pixel);
		return -EINVAL;
	}

	var->xres_virtual = ((var->xres_virtual + 15 )/ 16) * 16;
	var->xres = ((var->xres + 15 )/ 16) * 16;
#if 1
	if (var->xres_virtual > V_XRES)
		return -EINVAL;
	if (var->yres_virtual > V_YRES)
		return -EINVAL;

	if (var->xres_virtual < var->xres)
		return -EINVAL;
	if (var->yres_virtual < var->yres)
		return -EINVAL;
#else
	if (var->xres_virtual > V_XRES)
		var->xres_virtual = V_XRES;
	if (var->yres_virtual > V_YRES)
		var->yres_virtual = V_YRES;

	if (var->xres_virtual < var->xres)
		var->xres_virtual = var->xres;
	if (var->yres_virtual < var->yres)
		var->yres_virtual = var->yres;
#endif

	if (var->xoffset < 0)
		var->xoffset = 0;
	if (var->yoffset < 0)
		var->yoffset = 0;
	/* truncate xoffset and yoffset to maximum if too high */
	var->red.msb_right =
	var->green.msb_right =
	var->blue.msb_right =
	var->transp.offset = var->transp.length = var->transp.msb_right = 0;

	return 0;
}

static int sh7785fb_set_par(struct fb_info *info)
{
	sh7785fb_hw_set_var2par(info);
	sh7785fb_hw_load_hw_par(info);
	sh7785fb_hw_load_par(info->par);

	info->fix.line_length = (info->var.xres_virtual * (info->var.bits_per_pixel >> 3));
	info->fix.visual = (info->var.bits_per_pixel == 8) ?
				FB_VISUAL_PSEUDOCOLOR : FB_VISUAL_TRUECOLOR;

	return 0;
}

static int sh7785fb_pan_display(struct fb_var_screeninfo *var,
			       struct fb_info *info) 
{
	if (var->xoffset + info->var.xres > info->var.xres_virtual ||
	    var->yoffset + info->var.yres > info->var.yres_virtual)
		return -EINVAL;

	info->var.xoffset = var->xoffset;
	info->var.yoffset = var->yoffset;
	info->var.vmode &= ~FB_VMODE_YWRAP;

	return 0;
}

static int sh7785fb_setcolreg(unsigned regno, unsigned red, unsigned green,
			   unsigned blue, unsigned transp,
			   struct fb_info *info)
{

	if (regno >= CMAP_LEN)
		return -EINVAL;

	if (info->var.grayscale) {
		/* gray = 0.30*R + 0.59*G + 0.11*B */
		red = green = blue =
				(red * 77 + green * 151 + blue * 28) >> 8;
	}

	switch (info->var.bits_per_pixel) {
	case 8: /* NO suport */
		/* "transparent" stuff is completely ignored. */
		break;
	case 16:
		if (info->var.green.length == 5) {
			if (regno < 16) {
				/* 0rrrrrgg gggbbbbb */
				((u32 *)info->pseudo_palette)[regno] =
				((red & 0xf800) >> 1) |
				((green & 0xf800) >> 6) |
				((blue & 0xf800) >> 11);
			}
		} else {
			if (regno < 16) {
				/* rrrrrggg gggbbbbb */
				((u32 *)info->pseudo_palette)[regno] =
				((red & 0xf800) >> 0) |
				((green & 0xf800) >> 5) |
				((blue & 0xf800) >> 11);
			}
		}
		break;
	default:
		/* do nothing */
		break;
	}

	return 0;
}

static int sh7785fb_blank(int blank_mode, struct fb_info *info)
{
	return 0;
}

static int sh7785fb_ioctl(struct fb_info* info, unsigned int cmd, unsigned long arg)
{
	static  long *po;
	int *wk;
	struct sh7785fb_overlay *ov;
	struct sh7785fb_par *par;

	if (cmd == SH7785FB_IOCTL_DEBUG_ADD) {
		po = (long *)arg;
		return 0;
	} else if (cmd == SH7785FB_IOCTL_DEBUG_GET) {
		wk = (int *)arg;
		*wk = *po;
		return 0;
	} else if (cmd == SH7785FB_IOCTL_DEBUG_PUT) {
		*po = arg;
		return 0;
	} else if (cmd == SH7785FB_IOCTL_ENABLE) {
		if (arg == 0)
			sh7785fb_hw_disable(info->par);
		else
			sh7785fb_hw_enable(info->par);
		return 0;
	} else if (cmd == SH7785FB_IOCTL_OVERLAY){
		ov = (struct sh7785fb_overlay *)arg;
		par=info->par;
		par->pxmr &= 0x0000F000;
		if(ov->yuv_type == 0) {
			par->pxmr |= 0x00000001;
		}
		else{
			par->pxmr |= 0x00100003;
		}
		par->pxmwr  = ov->pxmwr;
		par->pxmlr  = ov->pxmlr;
		par->pxdsxr = ov->pxdsxr;
		par->pxdsyr = ov->pxdsyr;
		par->pxdpxr = ov->pxdpxr;
		par->pxdpyr = ov->pxdpyr;
		par->pxspxr = ov->pxspxr;
		par->pxspyr = ov->pxspyr;
		sh7785fb_hw_load_par(par);

		return 0;
	} else if (cmd == SH7785FB_IOCTL_PRIORITY){
		struct sh7785fb_hw_par *hw_par;
		struct sh7785fb_priority *pri = (struct sh7785fb_priority *)arg;
		unsigned long data;
		par=info->par;
		hw_par = par->hw_par;
		
		data = ((pri->pri_3rd & 0x0007) | 0x8);
		data = data<<4;
		data |= ((pri->pri_2nd & 0x0007) | 0x8);
		data = data<<4;
		data |= ((pri->pri_1st & 0x0007) | 0x8);
		hw_par->dppr = data | 0x76000000;
		__raw_writel(hw_par->dppr, SH7785FB_DPPR);
		return 0;
	} else if (cmd == SH7785FB_IOCTL_TRANS_COLOR){
		unsigned short color;
		unsigned long flg;
		par=info->par;
		color = ((arg & 0x000000ff) >> 3);
		color |= (((arg & 0x0000ff00) >> 5) & 0x07e0);
		color |= (((arg & 0x00ff0000) >> 8) & 0xf800);
		flg = (arg & 0xff000000);
		par->pxmr &= 0xFFFF0FFF;
		if(flg){
			par->pxmr |= 0x00004000;
		}
		par->pxtc2r = (unsigned long)color;
		SH7785FB_OUTL(par->pxtc2r, par->plane_number, SH7785FB_PX_OFFSET, SH7785FB_PXTC2R);
		SH7785FB_OUTL(par->pxmr, par->plane_number, SH7785FB_PX_OFFSET, SH7785FB_PXMR);
		return 0;
	}

	return -EINVAL;
}

/*
 *  Frame buffer operations
 */
static struct fb_ops sh7785fb_ops = {
	.owner		= THIS_MODULE,
	.fb_check_var	= sh7785fb_check_var,
	.fb_set_par	= sh7785fb_set_par,
	.fb_setcolreg	= sh7785fb_setcolreg,
	.fb_pan_display = sh7785fb_pan_display,
	.fb_blank	= sh7785fb_blank,
	.fb_fillrect    = cfb_fillrect,
        .fb_copyarea    = cfb_copyarea,
        .fb_imageblit   = cfb_imageblit,
	.fb_ioctl	= sh7785fb_ioctl,
};


/* ------------------------------------------------------------------------- */

static void sh7785fb_hw_set_var2par(struct fb_info *info)
{
	struct fb_var_screeninfo *var = &info->var;
	struct sh7785fb_par *par = info->par;
	struct sh7785fb_hw_par *hw_par = par->hw_par;

	par->pxmwr = var->xres_virtual;
	par->pxmlr = var->yres_virtual;
	par->pxdsxr = var->xres;
	par->pxdsyr = var->yres;
	par->pxdpxr = var->xoffset;
	par->pxdpyr = var->yoffset;

	hw_par->hdsr = HDS(var->hsync_len, var->left_margin);
	hw_par->hder = HDE(var->hsync_len, var->left_margin, var->xres);
	hw_par->vdsr = VDS(var->upper_margin);
	hw_par->vder = VDE(var->upper_margin, var->yres);
	hw_par->hswr = HSW(var->hsync_len);
	hw_par->hcr  = HC(var->hsync_len, var->left_margin, var->xres, var->right_margin);
	hw_par->vcr  = VC(var->vsync_len, var->upper_margin, var->yres, var->lower_margin);
	hw_par->vspr = VSP(var->upper_margin, var->yres, var->lower_margin);
	hw_par->desr = DES(var->hsync_len, var->left_margin);
	hw_par->dewr = DEW(var->xres);

}

static int __init sh7785fb_hw_set_default_par(struct sh7785fb_par *par, dma_addr_t addr)
{
//	par->pxdsa0r = virt_to_phys(SH7785FB_VRAM_TOP + (ALLOCATED_FB_MEM_SIZE * par->plane_number));
	par->pxdsa0r = addr;
	par->pxdsa1r = virt_to_phys(par->pxdsa0r + MAX_FRAMEBUFFER_MEM_SIZE);
	par->pxdsa2r = par->pxdsa1r + MAX_FRAMEBUFFER_MEM_SIZE;

	par->pxmr = 0x00004001;
	par->pxmwr = V_XRES;
	par->pxmlr = V_YRES;
	par->pxdsxr = XRES;
	par->pxdsyr = YRES;
	par->pxdpxr = 0;
	par->pxdpyr = 0;
	par->pxwaspr = 0;
	par->pxwamwr = 0;
	par->pxalphar = ALPH;
	par->pxbtr = 0;
	par->pxtc1r = 0;
	par->pxtc2r = 0;
	par->pxspxr = 0;
	par->pxspyr = 0;

	par->hw_par = &current_hw_par;
	
	return 0;
}

static int __init sh7785fb_hw_set_default_hw_par(struct sh7785fb_par *par)
{
	struct sh7785fb_hw_par *hw_par;

	hw_par = par->hw_par;

	hw_par->dsysr_enable	= 0x00000100;
	hw_par->dsysr_disable	= 0x00000240;
	hw_par->dsmr		= 0x07001000;
	hw_par->dssr		= 0;
	hw_par->dsrcr		= 0x0000CB0F;
	hw_par->dier		= 0x00000000;
	hw_par->cpcr		= 0x00000000;
	hw_par->eqwr		= 0;
	hw_par->spwr		= 0;
	hw_par->clampsr		= 0;
	hw_par->clampwr		= 0;
	hw_par->desr		= 0;
	hw_par->dewr		= 0;
	hw_par->cp1tr		= 0x00000000;
	hw_par->cp2tr		= 0x00000000;
	hw_par->cp3tr		= 0x00000000;
	hw_par->cp4tr		= 0x00000000;
	hw_par->rintofsr	= 0x00000000;
	hw_par->dppr		= 0x00000218;
	hw_par->defr		= 0x00000020;

	hw_par->hdsr = HDS(H_SYNC_W, H_BPORCH);
	hw_par->hder = HDE(H_SYNC_W, H_BPORCH, XRES);
	hw_par->vdsr = VDS(V_BPORCH);
	hw_par->vder = VDE(V_BPORCH, YRES);
	hw_par->hswr = HSW(H_SYNC_W);
	hw_par->hcr  = HC(H_SYNC_W, H_BPORCH, XRES, H_FPORCH);
	hw_par->vcr  = VC(V_SYNC_W, V_BPORCH, YRES, V_FPORCH);
	hw_par->vspr = VSP(V_BPORCH, YRES, V_FPORCH);
	hw_par->desr = DES(H_SYNC_W, H_BPORCH);
	hw_par->dewr = DEW(XRES);

	/* base color */
	hw_par->door = DOO;
	hw_par->bpor = BPO;

	hw_par->escr = FREQ_SELECT;
	hw_par->otar = 0;
	
	return 0;
}

static void sh7785fb_hw_enable(struct sh7785fb_par *par)
{
	__raw_writel(par->hw_par->dsysr_enable, SH7785FB_DSYSR);
}

static void sh7785fb_hw_disable(struct sh7785fb_par *par)
{
	__raw_writel(par->hw_par->dsysr_disable, SH7785FB_DSYSR);
}

static void sh7785fb_hw_load_par(struct sh7785fb_par *par)
{
	int i;

	i = par->plane_number;

	/* set VRAM ADDRESS */
	SH7785FB_OUTL(par->pxdsa0r,  i, SH7785FB_PX_OFFSET, SH7785FB_PXDSA0R);
	SH7785FB_OUTL(par->pxdsa1r,  i, SH7785FB_PX_OFFSET, SH7785FB_PXDSA1R);
	SH7785FB_OUTL(par->pxdsa2r,  i, SH7785FB_PX_OFFSET, SH7785FB_PXDSA2R);
	SH7785FB_OUTL(par->pxmr,     i, SH7785FB_PX_OFFSET, SH7785FB_PXMR);
	SH7785FB_OUTL(par->pxmwr,    i, SH7785FB_PX_OFFSET, SH7785FB_PXMWR);
	SH7785FB_OUTL(par->pxdsxr,   i, SH7785FB_PX_OFFSET, SH7785FB_PXDSXR);
	SH7785FB_OUTL(par->pxdsyr,   i, SH7785FB_PX_OFFSET, SH7785FB_PXDSYR);
	SH7785FB_OUTL(par->pxdpxr,   i, SH7785FB_PX_OFFSET, SH7785FB_PXDPXR);
	SH7785FB_OUTL(par->pxdpyr,   i, SH7785FB_PX_OFFSET, SH7785FB_PXDPYR);
	SH7785FB_OUTL(par->pxwaspr,  i, SH7785FB_PX_OFFSET, SH7785FB_PXWASPR);
	SH7785FB_OUTL(par->pxwamwr,  i, SH7785FB_PX_OFFSET, SH7785FB_PXWAMWR);
	SH7785FB_OUTL(par->pxmlr,    i, SH7785FB_PX_OFFSET, SH7785FB_PXMLR);
	SH7785FB_OUTL(par->pxalphar, i, SH7785FB_PX_OFFSET, SH7785FB_PXALPHAR);
	SH7785FB_OUTL(par->pxbtr,    i, SH7785FB_PX_OFFSET, SH7785FB_PXBTR);
	SH7785FB_OUTL(par->pxtc1r,   i, SH7785FB_PX_OFFSET, SH7785FB_PXTC1R);
	SH7785FB_OUTL(par->pxtc2r,   i, SH7785FB_PX_OFFSET, SH7785FB_PXTC2R);
	SH7785FB_OUTL(par->pxspxr,   i, SH7785FB_PX_OFFSET, SH7785FB_PXSPXR);
	SH7785FB_OUTL(par->pxspyr,   i, SH7785FB_PX_OFFSET, SH7785FB_PXSPYR);
}

static void sh7785fb_hw_load_hw_par(struct fb_info *info)
{
	struct sh7785fb_par *par;
	struct sh7785fb_hw_par *hw_par;

	par = info->par;
	hw_par = par->hw_par;

	__raw_writel(hw_par->dsmr, SH7785FB_DSMR);
	__raw_writel(hw_par->dssr, SH7785FB_DSSR);
	__raw_writel(hw_par->dsrcr, SH7785FB_DSRCR);
	__raw_writel(hw_par->dier, SH7785FB_DIER);
	__raw_writel(hw_par->cpcr, SH7785FB_CPCR);
	__raw_writel(hw_par->eqwr, SH7785FB_EQWR);
	__raw_writel(hw_par->spwr, SH7785FB_SPWR);
	__raw_writel(hw_par->clampsr, SH7785FB_CLAMPSR);
	__raw_writel(hw_par->clampwr, SH7785FB_CLAMPWR);
	__raw_writel(hw_par->desr, SH7785FB_DESR);
	__raw_writel(hw_par->dewr, SH7785FB_DEWR);
	__raw_writel(hw_par->cp1tr, SH7785FB_CP1TR);
	__raw_writel(hw_par->cp2tr, SH7785FB_CP2TR);
	__raw_writel(hw_par->cp3tr, SH7785FB_CP3TR);
	__raw_writel(hw_par->cp4tr, SH7785FB_CP4TR);
	__raw_writel(hw_par->rintofsr, SH7785FB_RINTOFSR);
	__raw_writel(hw_par->defr, SH7785FB_DEFR);

	__raw_writel(hw_par->hdsr, SH7785FB_HDSR);
	__raw_writel(hw_par->hder, SH7785FB_HDER);
	__raw_writel(hw_par->vdsr, SH7785FB_VDSR);
	__raw_writel(hw_par->vder, SH7785FB_VDER);
	__raw_writel(hw_par->hcr, SH7785FB_HCR);
	__raw_writel(hw_par->hswr, SH7785FB_HSWR);
	__raw_writel(hw_par->vcr, SH7785FB_VCR);
	__raw_writel(hw_par->vspr, SH7785FB_VSPR);

	__raw_writel(hw_par->door, SH7785FB_DOOR);
	__raw_writel(hw_par->bpor, SH7785FB_BPOR);
	__raw_writel(hw_par->escr, SH7785FB_ESCR);
	__raw_writel(hw_par->otar, SH7785FB_OTAR);
	__raw_writel(hw_par->dppr, SH7785FB_DPPR);
}

int __init sh7785fb_init(void)
{
	int cmap_len, i;
	unsigned long addr, size;


	cmap_len = CMAP_LEN;

	for ( i = 0; i < MAX_PLANE_NUM; i++){
		dma_addr_t fbdma;

		addr = dma_alloc_coherent(NULL, ALLOCATED_FB_MEM_SIZE, &fbdma, GFP_KERNEL);
		//addr = SH7785FB_VRAM_TOP + (ALLOCATED_FB_MEM_SIZE * i);
		size = ALLOCATED_FB_MEM_SIZE;
//		printk("=== %x %x %x\n",addr,size,addr+size);
		printk("%s: [%d] %08lx %08lx\n", __func__, i, addr, fbdma);


		fbinfo[i].screen_base = ioremap(addr, ALLOCATED_FB_MEM_SIZE);

		memset(fbinfo[i].screen_base, 0, size);

		fbinfo[i].fbops = &sh7785fb_ops;
		fbinfo[i].fix = sh7785fb_fix;
		fbinfo[i].fix.smem_start = addr;
		fbinfo[i].pseudo_palette = pseudo_palette[i];
		fbinfo[i].flags = FBINFO_FLAG_DEFAULT;
		fbinfo[i].par = (void *)&current_par[i];

		fb_alloc_cmap(&fbinfo[i].cmap, cmap_len, 0);

		fbinfo[i].var = sh7785fb_var;

		INFO_PAR(fbinfo[i],plane_number) = i;
		INFO_PAR(fbinfo[i],enable) = 1; /* ON */

		sh7785fb_hw_set_default_par(fbinfo[i].par, addr);
	}

	sh7785fb_hw_set_default_hw_par(fbinfo[SH7785FB_BASE_PLANE].par);
	sh7785fb_hw_disable(fbinfo[SH7785FB_BASE_PLANE].par);
	for ( i = 0; i < MAX_PLANE_NUM; i++){
		INFO_HWPAR(fbinfo[i],dppr) &= ~(0xf << (i*4));
		if (INFO_PAR(fbinfo[i],enable))
			INFO_HWPAR(fbinfo[i],dppr) |= ((0x8 | i) << (i*4));
		else
			INFO_HWPAR(fbinfo[i],dppr) |= (i << (i*4));
		sh7785fb_hw_load_par(fbinfo[i].par);
	}
	sh7785fb_hw_load_hw_par(&fbinfo[SH7785FB_BASE_PLANE]);
	sh7785fb_hw_enable(fbinfo[SH7785FB_BASE_PLANE].par);

	for ( i = 0; i < MAX_PLANE_NUM; i++){
		if (register_framebuffer(&fbinfo[i]) < 0)
			return -EINVAL;
		printk(KERN_INFO "fb%d: %s frame buffer device\n", fbinfo[i].node,
			fbinfo[i].fix.id);
	}

	return 0;
}

module_init(sh7785fb_init);

MODULE_LICENSE("GPL");
