using System;
using System.Diagnostics;
using System.IO;
using PdfSharp.Drawing;
using PdfSharp.Pdf;
using PdfSharp.Pdf.IO;
using PdfSharp.Pdf.Security;

namespace UnprotectDocument
{
  /// <summary>
  /// This sample shows how to unprotect a document (if you know the password).
  /// </summary>
  class Program
  {
    [STAThread]
    static void Main(string[] args)
    {
      // Get a fresh copy of the sample PDF file.
      // The passwords are 'user' and 'owner' in this sample.
      string filename = "HelloWorld (protected).pdf";
      File.Copy(Path.Combine("../../../../PDFs/", filename), 
        Path.Combine(Directory.GetCurrentDirectory(), filename), true);

      PdfDocument document;

      // Opening a document will fail with an invalid password.
      try
      {
        document = PdfReader.Open(filename, "invalid password");
      }
      catch (Exception ex)
      {
        Debug.WriteLine(ex.Message);
      }

      // You can specifiy a delegate, which is called if the document needs a
      // password. If you want to modify the document, you must provide the
      // owner password.
      document = PdfReader.Open(filename, PdfDocumentOpenMode.Modify,
        new PdfPasswordProvider(PasswordProvider));

      // Open the document with the user password.
      document = PdfReader.Open(filename, "user", PdfDocumentOpenMode.ReadOnly);

      // Use the property HasOwnerPermissions to decide whether the used password
      // was the user or the owner password. In both cases PDFsharp provides full
      // access to the PDF document. It is up to the programmer who uses PDFsharp
      // to honor the access rights. PDFsharp doesn't try to protect the document
      // because this make little sence for an open source library.
      bool hasOwnerAccess = document.SecuritySettings.HasOwnerPermissions;

      // Open the document with the owenr password.
      document = PdfReader.Open(filename, "owner");
      hasOwnerAccess = document.SecuritySettings.HasOwnerPermissions;

      // A document opened with the owner password is completely unprotected
      // and can be modified.
      XGraphics gfx = XGraphics.FromPdfPage(document.Pages[0]);
      gfx.DrawString("Some text...",
        new XFont("Times", 12), XBrushes.Firebrick, 50, 100);

      // The modified document is saved without any protection applied.
      PdfDocumentSecurityLevel level = document.SecuritySettings.DocumentSecurityLevel;

      // If you want to save it protected, you must set the DocumentSecurityLevel
      // or apply new passwords.
      // In the current implementation the old passwords are not automatically
      // reused. See 'ProtectDocument' sample for further information.

      // Save the document...
      document.Save(filename);
      // ...and start a viewer.
      Process.Start(filename);
    }

    /// <summary>
    /// The 'get the password' call back function.
    /// </summary>
    static void PasswordProvider(PdfPasswordProviderArgs args)
    {
      // Show a dialog here in a real application
      args.Password = "owner";
    }
  }
}