using System;
using System.Diagnostics;
using System.IO;
using PdfSharp;
using PdfSharp.Pdf;
using PdfSharp.Pdf.IO;
using PdfSharp.Drawing;

namespace CompareDocuments
{
  /// <summary>
  /// This samples shows how to create a new document from two existing PDF files.
  /// The pages are inserted alternately from two documents. This may be useful
  /// for visual comparision.
  /// </summary>
  class Program
  {
    [STAThread]
    static void Main(string[] args)
    {
      // Choose a variant
      //Variant1();
      Variant2();
    }

    /// <summary>
    /// Imports pages 'as they are' from an external document.
    /// </summary>
    static void Variant1()
    {
      // Information: The current implementation of PDFsharp creates a complete
      // deep copy of an external page, which leads to large documents if many
      // pages are imported. Currently use Variant2.

      // Get a fresh copies of the sample PDF files
      string filename1 = "Portable Document Format.pdf";
      File.Copy(Path.Combine("../../../../PDFs/", filename1), 
        Path.Combine(Directory.GetCurrentDirectory(), filename1), true);
      string filename2 = "Portable Document Format.pdf";  // use other file here
      File.Copy(Path.Combine("../../../../PDFs/", filename2), 
        Path.Combine(Directory.GetCurrentDirectory(), filename2), true);

      //
      //filename1 = "PDFReference.pdf";
      //filename2 = "1000 Pages Manual.pdf";

      // Open the input files
      PdfDocument inputDocument1 = PdfReader.Open(filename1, PdfDocumentOpenMode.ReadOnly);
      PdfDocument inputDocument2 = PdfReader.Open(filename2, PdfDocumentOpenMode.ReadOnly);

      // Create the output document
      PdfDocument outputDocument = new PdfDocument();

      // Show consecutive pages facing
      outputDocument.PageLayout = PdfPageLayout.TwoColumnLeft;

      XFont font = new XFont("Verdana", 10, XFontStyle.Bold);
      XStringFormat format = new XStringFormat();
      format.Alignment = XStringAlignment.Center;
      format.LineAlignment = XLineAlignment.Far;
      XGraphics gfx;
      XRect box;
      int count = Math.Max(inputDocument1.PageCount, inputDocument2.PageCount);
      for (int idx = 0; idx < count; idx++)
      {
        PdfPage page1 = inputDocument1.PageCount > idx ? 
          inputDocument1.Pages[idx] : new PdfPage();
        PdfPage page2 = inputDocument2.PageCount > idx ? 
          inputDocument2.Pages[idx] : new PdfPage();

        // Add the pages to  the output document
        page1 = outputDocument.AddPage(page1);
        page2 = outputDocument.AddPage(page2);

        // Write document file name and page number on each page
        gfx = XGraphics.FromPdfPage(page1);
        box = page1.MediaBox.ToXRect();
        box.Inflate(0, -10);
        gfx.DrawString(String.Format("{0}  {1}", filename1, idx + 1),
          font, XBrushes.Red, box, format);

        gfx = XGraphics.FromPdfPage(page2);
        box = page2.MediaBox.ToXRect();
        box.Inflate(0, -10);
        gfx.DrawString(String.Format("{0}  {1}", filename2, idx + 1),
          font, XBrushes.Red, box, format);
      }

      // Save the document...
      string filename = "CompareDocument1.pdf";
      outputDocument.Save(filename);
      // ...and start a viewer.
      Process.Start(filename);
    }

    /// <summary>
    /// Imports the pages as form X objects.
    /// </summary>
    static void Variant2()
    {
      // Get fresh copies of the sample PDF files
      string filename1 = "Portable Document Format.pdf";
      File.Copy(Path.Combine("../../../../PDFs/", filename1), 
        Path.Combine(Directory.GetCurrentDirectory(), filename1), true);
      string filename2 = "Portable Document Format.pdf";
      File.Copy(Path.Combine("../../../../PDFs/", filename2), 
        Path.Combine(Directory.GetCurrentDirectory(), filename2), true);

      // Create the output document
      PdfDocument outputDocument = new PdfDocument();

      // Show consecutive pages facing
      outputDocument.PageLayout = PdfPageLayout.TwoPageLeft;

      XFont font = new XFont("Verdana", 10, XFontStyle.Bold);
      XStringFormat format = new XStringFormat();
      format.Alignment = XStringAlignment.Center;
      format.LineAlignment = XLineAlignment.Far;
      XGraphics gfx;
      XRect box;

      // Open the external documents as XPdfForm objects. Such objects are
      // treated like images. By default the first page of the document is
      // referenced by a new XPdfForm.
      XPdfForm form1 = XPdfForm.FromFile(filename1);
      XPdfForm form2 = XPdfForm.FromFile(filename2);

      int count = Math.Max(form1.PageCount, form2.PageCount);
      for (int idx = 0; idx < count; idx++)
      {
        // Add two new pages to the output document
        PdfPage page1 = outputDocument.AddPage();
        PdfPage page2 = outputDocument.AddPage();

        if (form1.PageCount > idx)
        {
          gfx = XGraphics.FromPdfPage(page1);

          // Set page number (which is one-based)
          form1.PageNumber = idx + 1;

          // Draw the page identified by the page number like an image
          gfx.DrawImage(form1, new XRect(0, 0, form1.Width, form1.Height));

          // Write document file name and page number on each page
          box = page1.MediaBox.ToXRect();
          box.Inflate(0, -10);
          gfx.DrawString(String.Format("{0}  {1}", filename1, idx + 1),
            font, XBrushes.Red, box, format);
        }

        // Same as above for second page
        if (form2.PageCount > idx)
        {
          gfx = XGraphics.FromPdfPage(page2);

          form2.PageNumber = idx + 1;
          gfx.DrawImage(form2, new XRect(0, 0, form2.Width, form2.Height));

          box = page2.MediaBox.ToXRect();
          box.Inflate(0, -10);
          gfx.DrawString(String.Format("{0}  {1}", filename2, idx + 1),
            font, XBrushes.Red, box, format);
        }
      }

      // Save the document...
      string filename = "CompareDocument2.pdf";
      outputDocument.Save(filename);
      // ...and start a viewer.
      Process.Start(filename);
    }
  }
}
