//
// PDFSharp - A library for processing PDF
//
// Authors:
//   Stefan Lange (mailto:Stefan.Lange@pdfsharp.com)
//
// Copyright (c) 2005 empira Software GmbH, Cologne (Germany)
//
// http://www.pdfsharp.com
//
// Permission is hereby granted, free of charge, to any person obtaining
// a copy of this software and associated documentation files (the
// "Software"), to deal in the Software without restriction, including
// without limitation the rights to use, copy, modify, merge, publish,
// distribute, sublicense, and/or sell copies of the Software, and to
// permit persons to whom the Software is furnished to do so, subject to
// the following conditions:
// 
// The above copyright notice and this permission notice shall be
// included in all copies or substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
// EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
// MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
// NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
// LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
// OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
// WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

using System;
using System.Diagnostics;
using System.Collections;
using System.Text;
using System.IO;
using PdfSharp.Internal;
using PdfSharp.Pdf.Advanced;
using PdfSharp.Pdf.Internal;
using PdfSharp.Pdf.IO;

namespace PdfSharp.Pdf
{
  /// <summary>
  /// Represents the cross reference table of a PDF document. It contains all indirect objects of
  /// a document.
  /// </summary>
  internal sealed class PdfXRefTable // do not derive from PdfObject
  {
    public PdfXRefTable(PdfDocument document)
    {
      this.document = document;
    }
    PdfDocument document;

    /// <summary>
    /// Represents the relation between PdfObjectID and PdfXRef for a PdfDocument.
    /// </summary>
    public Hashtable objectTable = new Hashtable();

    internal bool IsUnderConstruction
    {
      get {return this.isUnderConstruction;}
      set {this.isUnderConstruction = value;}
    }
    bool isUnderConstruction;

    /// <summary>
    /// Adds a cross reference entry to the table. Used when parsing the trailer.
    /// </summary>
    public void Add(PdfXRef xref)
    {
      if (xref.ObjectID.IsEmpty)
        xref.ObjectID = new PdfObjectID(GetNewObjectNumber());

      if (this.objectTable.Contains(xref.ObjectID))
        throw new InvalidOperationException("Object already in table.");

      this.objectTable.Add(xref.ObjectID, xref);
    }

    /// <summary>
    /// Adds a PdfObject to the table.
    /// </summary>
    public void Add(PdfObject value)
    {
      if (value.ObjectID.IsEmpty)
        value.SetObjectID(GetNewObjectNumber(), 0);

      if (this.objectTable.Contains(value.ObjectID))
        throw new InvalidOperationException("Object already in table.");

      this.objectTable.Add(value.ObjectID, value.XRef);
    }

    public void Remove(PdfXRef xref)
    {
      this.objectTable.Remove(xref.ObjectID);
    }

    /// <summary>
    /// Gets a cross reference entry from an object identifier.
    /// </summary>
    public PdfXRef this[PdfObjectID objectID]
    {
      get { return (PdfXRef)this.objectTable[objectID]; }
    }

    /// <summary>
    /// Indicates whether the specified object identifier is in the table.
    /// </summary>
    public bool Contains(PdfObjectID objectID)
    {
      return this.objectTable.Contains(objectID);
    }

    //public PdfObject GetObject(PdfObjectID objectID)
    //{
    //  return this[objectID].Value;
    //}

//    /// <summary>
//    /// Gets the entry for the specified object, or null, if the object is not in
//    /// this XRef table.
//    /// </summary>
//    internal PdfXRef GetEntry(PdfObjectID objectID)
//    {
//      return this[objectID];
//    }

    /// <summary>
    /// Returns the next free object number.
    /// </summary>
    public int GetNewObjectNumber()
    {
      // New objects are numbered consecutively. If a document is imported, maxObjectNumber is
      // set to the highest object number used in the document.
      return ++this.maxObjectNumber;
    }
    internal int maxObjectNumber;

    /// <summary>
    /// Writes the xref section in pdf stream.
    /// </summary>
    internal void WriteObject(PdfWriter writer)
    {
      writer.WriteRaw("xref\n");

//      PdfObjectID[] objectIDs2 = AllObjectIDs;
      PdfXRef[] xrefs = AllXRefs;
//      PdfObjectID id1 = objectIDs2[0];
//      PdfXRef     xref1 = xrefs[0];
//      id1.GetType();
//      xref1.GetType();
      //ArrayList list = new ArrayList(AllObjectIDs);
      //list.Sort();
      //list.CopyTo(objectIDs);

      int count = xrefs.Length;
      writer.WriteRaw(String.Format("0 {0}\n", count + 1));
      writer.WriteRaw(String.Format("{0:0000000000} {1:00000} {2} \n", 0, 65535, "f"));
      //PdfEncoders.WriteAnsi(stream, text);

      for (int idx = 0; idx < count; idx++)
      {
        PdfXRef xref = xrefs[idx];

        //text = String.Format("{0} {1}\n", xref.ObjectID.ObjectNumber, 1);
        //PdfEncoders.WriteAnsi(stream, text);

        // Acrobat is very pedantic; it must be exactly 20 bytes per line.
        writer.WriteRaw(
          String.Format("{0:0000000000} {1:00000} {2} \n", xref.Position, xref.GenerationNumber, "n"));
        //PdfEncoders.WriteAnsi(stream, text);
      }
    }

    /// <summary>
    /// Gets an array of all object identifier. For debugging purposes only.
    /// </summary>
    internal PdfObjectID[] AllObjectIDs
    {
      get
      {
        ICollection collection = this.objectTable.Keys;
        PdfObjectID[] objectIDs = new PdfObjectID[collection.Count];
        collection.CopyTo(objectIDs, 0);
        return objectIDs;
      }
    }

    /// <summary>
    /// Gets an array of all cross references ordered increasing by their object identifier.
    /// </summary>
    internal PdfXRef[] AllXRefs
    {
      get
      {
        ICollection collection = this.objectTable.Values;
        ArrayList list = new ArrayList(collection);
        list.Sort(PdfXRef.Comparer);
        PdfXRef[] xrefs = new PdfXRef[collection.Count];
        list.CopyTo(xrefs, 0);
        return xrefs;
      }
    }

    internal void HandleOrphanedReferences()
    {
    }

    /// <summary>
    /// Removes all objects that cannot be reached from the trailer. Returns the number of removed
    /// objects.
    /// </summary>
    internal int Compact()
    {
      // TODO: remove PdfBooleanObject, PdfIntegerObject etc.
      int removed = this.objectTable.Count;
      //CheckConsistence();
      // TODO: Is this really so easy?
      PdfXRef[] xrefs = TransitiveClosure(this.document.trailer);

#if DEBUG_
      foreach (PdfXRef xref in this.objectTable.Values)
      {
        if (xref.Value == null)
          this.GetType();
        Debug.Assert(xref.Value != null);
      }

      foreach (PdfXRef xref in xrefs)
      {
        if (!this.objectTable.Contains(xref.ObjectID))
          this.GetType();
        Debug.Assert(this.objectTable.Contains(xref.ObjectID));

        if (xref.Value == null)
          this.GetType();
        Debug.Assert(xref.Value != null);
      }
#endif

      this.maxObjectNumber = 0;
      this.objectTable.Clear();
      foreach (PdfXRef xref in xrefs)
      {
        this.objectTable.Add(xref.ObjectID, xref);
        this.maxObjectNumber = Math.Max(this.maxObjectNumber, xref.ObjectNumber);
      }
      //CheckConsistence();
      removed -= this.objectTable.Count;
      return removed;
    }

    /// <summary>
    /// Renumbers the objects starting at 1.
    /// </summary>
    internal void Renumber()
    {
      //CheckConsistence();
      PdfXRef[] xrefs = AllXRefs;
      this.objectTable.Clear();
      // Give all objects a new number
      int count = xrefs.Length;
      for (int idx = 0; idx < count; idx++)
      {
        PdfXRef xref = xrefs[idx];
#if DEBUG_
        if (xref.ObjectNumber == 1108)
          GetType();
#endif
        xref.ObjectID = new PdfObjectID(idx + 1);
        // Rehash with new number
        this.objectTable.Add(xref.ObjectID, xref);
      }
      this.maxObjectNumber = count;
      //CheckConsistence();
    }

    /// <summary>
    /// Checks the locical consistence for debugging purposes (useful after reconstruction work).
    /// </summary>
    [Conditional("DEBUG_")]
    public void CheckConsistence()
    {
      Hashtable ht = new Hashtable();
      foreach (PdfXRef xref in this.objectTable.Values)
      {
        Debug.Assert(!ht.Contains(xref), "Duplicate xref.");
        Debug.Assert(xref.Value != null);
        ht.Add(xref, null);
      }

      ht.Clear();
      foreach (PdfXRef xref in this.objectTable.Values)
      {
        Debug.Assert(!ht.Contains(xref.ObjectID), "Duplicate xref.");
        ht.Add(xref.ObjectID, null);
      }

      ICollection collection = this.objectTable.Values;
      int count = collection.Count;
      PdfXRef[] xrefs = new PdfXRef[count];
      collection.CopyTo(xrefs, 0);
#if true_
      for (int i = 0; i < count; i++)
        for (int j = 0; j < count; j++)
          if (i != j)
          {
            Debug.Assert(Object.ReferenceEquals(xrefs[i].Document, this.document));
            Debug.Assert(xrefs[i] != xrefs[j]);
            Debug.Assert(!Object.ReferenceEquals(xrefs[i], xrefs[j]));
            Debug.Assert(!Object.ReferenceEquals(xrefs[i].Value, xrefs[j].Value));
            Debug.Assert(!Object.ReferenceEquals(xrefs[i].ObjectID, xrefs[j].Value.ObjectID));
            Debug.Assert(xrefs[i].ObjectNumber != xrefs[j].Value.ObjectNumber);
            Debug.Assert(Object.ReferenceEquals(xrefs[i].Document, xrefs[j].Document));
            GetType();
          }
#endif
    }

    ///// <summary>
    ///// The garbage collector for PDF objects.
    ///// </summary>
    //public sealed class GC
    //{
    //  PdfXRefTable xrefTable;
    //
    //  internal GC(PdfXRefTable xrefTable)
    //  {
    //    this.xrefTable = xrefTable;
    //  }
    //
    //  public void Collect()
    //  {
    //  }
    //
    //  public PdfXRef[] ReachableObjects()
    //  {
    //    Hashtable objects = new Hashtable();
    //    TransitiveClosure(objects, this.xrefTable.document.trailer);
    //  }

    /// <summary>
    /// Calculates the transitive closure of the specifed PdfObject, i.e. all indirect objects
    /// recursively reachable from the specified object.
    /// </summary>
    public PdfXRef[] TransitiveClosure(PdfObject pdfObject)
    {
      CheckConsistence();
      Hashtable objects = new Hashtable();
      TransitiveClosureImplementation(objects, pdfObject);
      CheckConsistence();

      ICollection collection = objects.Keys;
      int count = collection.Count;
      PdfXRef[] xrefs = new PdfXRef[count];
      collection.CopyTo(xrefs, 0);

#if true_
      for (int i = 0; i < count; i++)
        for (int j = 0; j < count; j++)
          if (i != j)
          {
            Debug.Assert(Object.ReferenceEquals(xrefs[i].Document, this.document));
            Debug.Assert(xrefs[i] != xrefs[j]);
            Debug.Assert(!Object.ReferenceEquals(xrefs[i], xrefs[j]));
            Debug.Assert(!Object.ReferenceEquals(xrefs[i].Value, xrefs[j].Value));
            Debug.Assert(!Object.ReferenceEquals(xrefs[i].ObjectID, xrefs[j].Value.ObjectID));
            Debug.Assert(xrefs[i].ObjectNumber != xrefs[j].Value.ObjectNumber);
            Debug.Assert(Object.ReferenceEquals(xrefs[i].Document, xrefs[j].Document));
            GetType();
          }
#endif
      return xrefs;
    }

    void TransitiveClosureImplementation(Hashtable objects2, PdfObject pdfObject27)
    {
      if (!Object.ReferenceEquals(pdfObject27.Document, this.document))
        GetType();
//////Debug.Assert(Object.ReferenceEquals(pdfObject27.Document, this.document));
//      if (item is PdfObject && ((PdfObject)item).ObjectID.ObjectNumber == 5)
//        Debug.WriteLine("items: " + ((PdfObject)item).ObjectID.ToString());
      if (pdfObject27.ObjectNumber == 5)
        GetType();

      IEnumerable enumerable = null; //(IEnumerator)pdfObject;
      if (pdfObject27 is PdfDictionary)
        enumerable = ((PdfDictionary)pdfObject27).Elements.Values;
      else if (pdfObject27 is PdfArray)
        enumerable = ((PdfArray)pdfObject27).Elements;
      if (enumerable != null)
      {
        foreach (PdfItem item in enumerable)
        {
          PdfXRef xref = item as PdfXRef;
          if (xref != null)
          {
            // Is this an indirect reference to an object that not exists?
            //if (xref.Document == null)
            //{
            //  Debug.WriteLine("Dead object dedected: " + xref.ObjectID.ToString());
            //  PdfXRef dead = DeadObject;
            //  xref.ObjectID = dead.ObjectID;
            //  xref.Document = this.document;
            //  xref.SetObject(dead.Value);
            //  PdfDictionary dict = (PdfDictionary)dead.Value;
            //
            //  dict.Elements["/DeadObjectCount"] = 
            //    new PdfInteger(dict.Elements.GetInteger("/DeadObjectCount") + 1);
            //
            //  xref = dead;
            //}

            if (!Object.ReferenceEquals(xref.Document, this.document))
            {
              GetType();
              Debug.WriteLine(String.Format("Bad xref: {0}", xref.ObjectID.ToString()));
            }
            Debug.Assert(Object.ReferenceEquals(xref.Document, this.document) || xref.Document == null, "External object detected!");
#if DEBUG
            if (xref.ObjectID.ObjectNumber == 23)
              GetType();
#endif
            if (!objects2.Contains(xref))
            {
              PdfObject value = xref.Value;

              // HACK ignore unreachable objets
              if (xref.Document != null)
              {
                // ... from trailer hack
                if (value == null)  
                {
                  xref = (PdfXRef)this.objectTable[xref.ObjectID];
                  Debug.Assert(xref.Value != null);
                  value = xref.Value;
                }
                Debug.Assert(Object.ReferenceEquals(xref.Document, this.document));
                objects2.Add(xref, null);
                if (value is PdfArray || value is PdfDictionary)
                  TransitiveClosureImplementation(objects2, value);
              }
              //else
              //{
              //  objects2.Add(this[xref.ObjectID], null);
              //}
            }
          }
          else
          {
            PdfObject pdfObject28 = item as PdfObject;
            //if (pdfObject28 != null)
            //  Debug.Assert(Object.ReferenceEquals(pdfObject28.Document, this.document));
            if (pdfObject28 != null && (pdfObject28 is PdfDictionary || pdfObject28 is PdfArray))
              TransitiveClosureImplementation(objects2, pdfObject28);
          }
        }
      }
    }

    /// <summary>
    /// Gets the cross reference to an objects used for undefined indirect references.
    /// </summary>
    public PdfXRef DeadObject
    {
      get
      {
        if (false || this.deadObject == null)
        {
          this.deadObject = new PdfDictionary(this.document);
          Add(this.deadObject);
          this.deadObject.Elements.Add("/DeadObjectCount", new PdfInteger());
        }
        return this.deadObject.XRef;
      }
    }
    PdfDictionary deadObject;
  }
}
