//
// PDFSharp - A library for processing PDF
//
// Authors:
//   Stefan Lange (mailto:Stefan.Lange@pdfsharp.com)
//
// Copyright (c) 2005 empira Software GmbH, Cologne (Germany)
//
// http://www.pdfsharp.com
//
// Permission is hereby granted, free of charge, to any person obtaining
// a copy of this software and associated documentation files (the
// "Software"), to deal in the Software without restriction, including
// without limitation the rights to use, copy, modify, merge, publish,
// distribute, sublicense, and/or sell copies of the Software, and to
// permit persons to whom the Software is furnished to do so, subject to
// the following conditions:
// 
// The above copyright notice and this permission notice shall be
// included in all copies or substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
// EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
// MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
// NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
// LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
// OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
// WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

using System;
using System.Diagnostics;
using System.Collections;
using System.Text;
using System.IO;
using PdfSharp.Internal;
using PdfSharp.Pdf.IO;
using PdfSharp.Pdf;
using PdfSharp.Pdf.Advanced;

namespace PdfSharp.Pdf
{
  /// <summary>
  /// Represents the pages of the document.
  /// </summary>
  public sealed class PdfPages : PdfDictionary, IEnumerable
  {
    internal PdfPages(PdfDocument document) : base(document)
    {
      Elements.Type = "/Pages";
      Elements[Keys.Count] = new PdfInteger(0);
    }

    PdfPages(PdfDictionary dictionary) : base(dictionary)
    {
    }

    /// <summary>
    /// Gets the number of pages.
    /// </summary>
    public int Count
    {
      get {return this.PagesArray.Elements.Count;}
    }

    /// <summary>
    /// Gets the page with the specified index.
    /// </summary>
    public PdfPage this[int index]
    { 
      get
      {
        if (index < 0 || index >= Count)
          throw new ArgumentOutOfRangeException("index", index, PSSR.PageIndexOutOfRange);

        PdfDictionary dict = (PdfDictionary)((PdfXRef)PagesArray.Elements[index]).Value;
        if (!(dict is PdfPage))
          dict = new PdfPage(dict);
        return (PdfPage)dict;
      }
      //set
      //{
      //  if (index < 0 || index >= this.pages.Count)
      //    throw new ArgumentOutOfRangeException("index", index, PSSR.PageIndexOutOfRange);
      //  this.pages[index] = value;
      //}
    }

    public PdfPage Add()
    {
      PdfPage page = new PdfPage();
      Insert(Count, page);
      return page;
    }

    public PdfPage Add(PdfPage page)
    {
      return Insert(Count, page);
    }

    public PdfPage Insert(int index)
    {
      PdfPage page = new PdfPage();
      Insert(index, page);
      return page;
    }

    public PdfPage Insert(int index, PdfPage page)
    {
      if (page == null)
        throw new ArgumentNullException("page");
      if (page.Document == this.Document)
        throw new InvalidOperationException(PSSR.MultiplePageInsert);

      // All page insertions come here
      if (page.Document == null)
      {
        page.Document = this.Document;

        Document.xrefTable.Add(page);
        PagesArray.Elements.Insert(index, page.XRef);
        Elements.SetInteger(PdfPages.Keys.Count, PagesArray.Elements.Count);
      }
      else
      {
        // Import from other document
        page = ImportExternalPage(page);

        Document.xrefTable.Add(page);
        PagesArray.Elements.Insert(index, page.XRef);
        Elements.SetInteger(PdfPages.Keys.Count, PagesArray.Elements.Count);
      }
      return page;
    }

    public void Remove(PdfPage page)
    {
      PagesArray.Elements.Remove(page.XRef);
      Elements.SetInteger(PdfPages.Keys.Count, PagesArray.Elements.Count);
    }

    public void RemoveAt(int index)
    {
      PagesArray.Elements.RemoveAt(index);
      Elements.SetInteger(PdfPages.Keys.Count, PagesArray.Elements.Count);
    }

    /// <summary>
    /// Imports an external page. The elements of the imported page are cloned and added to this
    /// document.
    /// Important: In contrast to PdfFormXObject adding an external page always make a deep copy
    /// of their transitive closure. Any reuse of already imported objects is not intended because
    /// any modification of an imported page must not change another page.
    /// </summary>
    PdfPage ImportExternalPage(PdfPage importPage)
    {
      PdfPage page = new PdfPage(this.document);

      // TODO: dont always deep copy resources
      CloneElement(page, importPage, PdfPage.Keys.Resources);
      CloneElement(page, importPage, PdfPage.Keys.Contents);
      CloneElement(page, importPage, PdfPage.Keys.MediaBox);
      CloneElement(page, importPage, PdfPage.Keys.CropBox);
      CloneElement(page, importPage, PdfPage.Keys.Rotate);
      CloneElement(page, importPage, PdfPage.Keys.BleedBox);
      CloneElement(page, importPage, PdfPage.Keys.TrimBox);
      CloneElement(page, importPage, PdfPage.Keys.ArtBox);
      CloneElement(page, importPage, PdfPage.Keys.Annots);
      // TODO more elements?
      return page;
    }

    /// <summary>
    /// Helper function for ImportExternalPage.
    /// </summary>
    void CloneElement(PdfPage page, PdfPage importPage, string key)
    {
      Debug.Assert(page.Document == this.document);
      PdfItem item = importPage.Elements[key];
      if (item != null)
      {
        // The item can be indirect. If so, replace it by its value.
        if (item is PdfXRef)
          item = ((PdfXRef)item).Value;
        if (item is PdfObject)
        {
          // Objects are deep copied.
          PdfObject root = (PdfObject)item;
          root = PdfObject.DeepCopyClosure(this.document, root);
          if (root.XRef == null)
            page.Elements[key] = root;
          else
            page.Elements[key] = root.XRef;
        }
        else
        {
          // Simple items are just cloned.
          page.Elements[key] = item.Clone();
        }
      }
    }

    /// <summary>
    /// 
    /// </summary>
    internal PdfArray PagesArray
    {
      get 
      {
        if (this.pagesArray == null)
          this.pagesArray = (PdfArray)Elements.GetValue(Keys.Kids, VCF.Create);
        return this.pagesArray;
      }
    }
    PdfArray pagesArray;

    /// <summary>
    /// Replaces the page tree by a flat array of indirect references to the the pages objects.
    /// </summary>
    internal void FlattenPageTree()
    {
      // Acrobat creates a balanced tree if the number of pages is rougly more than ten. This is
      // not difficult but obviously also not neccessary. I created a document with 50000 pages with
      // PDF4NET and Acrobat opened it in less than 2 seconds.

      //PdfXRef xrefRoot = this.Document.Catalog.Elements[PdfCatalog.Keys.Pages] as PdfXRef;
      //PdfDictionary[] pages = GetKids(xrefRoot, null);

      // Promote inheritable values down the page tree
      PdfPage.InheritedValues values = new PdfPage.InheritedValues();
      PdfPage.InheritValues(this, ref values);
      PdfDictionary[] pages = GetKids(this.XRef, values, null);

      // Replace /Pages in catalog by this object
//      xrefRoot.Value = this;

      PdfArray array = new PdfArray(this.Document);
      foreach (PdfDictionary page in pages)
      {
        // Fix the parent
        page.Elements[PdfPage.Keys.Parent] = this.XRef;
        array.Elements.Add(page.XRef);
      }

      Elements.SetName(Keys.Type, "/Pages");
#if true 
      // direct array
      Elements.SetValue(Keys.Kids, array);
#else
      // incdirect array
      this.Document.xrefTable.Add(array);
      Elements.SetValue(Keys.Kids, array.XRef);
#endif
      Elements.SetInteger(Keys.Count, array.Elements.Count);
    }

    /// <summary>
    /// Recursively converts the page tree into a flat array.
    /// </summary>
    PdfDictionary[] GetKids(PdfXRef xref, PdfPage.InheritedValues values, PdfDictionary parent)
    {
      // TODO: inherit inheritable keys...
      PdfDictionary kid = (PdfDictionary)xref.Value;

      if (kid.Elements.Type == "/Page")
      {
        PdfPage.InheritValues(kid, values);
        return new PdfDictionary[]{kid};
      }
      else
      {
        Debug.Assert(kid.Elements.Type == "/Pages");
        PdfPage.InheritValues(kid, ref values);
        ArrayList list = new ArrayList();
        PdfArray kids = kid.Elements["/Kids"] as PdfArray;
        foreach (PdfXRef xref2 in kids)
          list.AddRange(GetKids(xref2, values, kid));
        int count = list.Count;
        Debug.Assert(count == kid.Elements.GetInteger("/Count"));
        return (PdfDictionary[])list.ToArray(typeof(PdfDictionary));
      }
    }

    /// <summary>
    /// TODO: Create the page tree.
    /// </summary>
    internal override void PrepareForSave()
    {
      // TODO: Close all open content streams

      // TODO: Create the page tree.
      // Arrays have a limit of 8192 entries, but I successfully tested documents
      // with 50000 pages and no page tree.
      // ==> wait for bug report.
    }

//    public IEnumerator GetEnumerator()
//    {
//      return new PdfPagesEnumerator(this);
//    }

//    private string BuildPageRefs()
//    {
//      StringBuilder builder = new StringBuilder();
//      for (int idx = 0; idx < this.pages.Count; idx++)
//      {
//        builder.AppendFormat("{0} 0 R ", this[idx].ObjectID);
//        if (idx % 10 == 0 && idx != 0)
//          builder.Append("\n");
//      }
//      return builder.ToString();
//    }

    //internal PdfPage.PdfPageCollection pages;

    private class PdfPagesEnumerator : IEnumerator
    {
      private PdfPage  currentElement;
      private int      index;
      private PdfPages list;

      internal PdfPagesEnumerator(PdfPages list)
      {
        this.list  = list;
        this.index = -1;
      }

      public bool MoveNext()
      {
        if (this.index < this.list.Count - 1)
        {
          this.index++;
          this.currentElement = this.list[this.index];
          return true;
        }
        this.index = this.list.Count;
        return false;
      }

      public void Reset()
      {
        this.currentElement = null;
        this.index = -1;
      }

      object IEnumerator.Current
      {
        get {return this.Current;}
      }

      public PdfPage Current
      { 
        get
        {
          if (this.index == -1 || this.index >= this.list.Count)
            throw new InvalidOperationException(PSSR.ListEnumCurrentOutOfRange);
          return this.currentElement;
        }
      }
    }

    /// <summary>
    /// Predefined keys of this dictionary.
    /// </summary>
    internal sealed class Keys : PdfPage.InheritablePageKeys
    {
      /// <summary>
      /// (Required) The type of PDF object that this dictionary describes; 
      /// must be Pages for a page tree node.
      /// </summary>
      [KeyInfo(KeyType.Name | KeyType.Required, FixedValue = "Pages")]
      public const string Type = "/Type";

      /// <summary>
      /// (Required except in root node; must be an indirect reference)
      /// The page tree node that is the immediate parent of this one.
      /// </summary>
      [KeyInfo(KeyType.Dictionary | KeyType.Required)]
      public const string Parent = "/Parent";

      /// <summary>
      /// (Required) An array of indirect references to the immediate children of this node.
      /// The children may be page objects or other page tree nodes.
      /// </summary>
      [KeyInfo(KeyType.Array | KeyType.Required)]
      public const string Kids = "/Kids";

      /// <summary>
      /// (Required) The number of leaf nodes (page objects) that are descendants of this node 
      /// within the page tree.
      /// </summary>
      [KeyInfo(KeyType.Integer | KeyType.Required)]
      public const string Count = "/Count";

      /// <summary>
      /// Gets the KeysMeta for these keys.
      /// </summary>
      public static KeysMeta Meta
      {
        get
        {
          if (Keys.meta == null)
            Keys.meta = CreateMeta(typeof(Keys));
          return Keys.meta;
        }
      }
      static KeysMeta meta;
    }

    /// <summary>
    /// Gets the KeysMeta of this dictionary type.
    /// </summary>
    internal override KeysMeta Meta
    {
      get {return Keys.Meta;}
    }
  }
}
