//
// PDFSharp - A library for processing PDF
//
// Authors:
//   Stefan Lange (mailto:Stefan.Lange@pdfsharp.com)
//
// Copyright (c) 2005 empira Software GmbH, Cologne (Germany)
//
// http://www.pdfsharp.com
//
// Permission is hereby granted, free of charge, to any person obtaining
// a copy of this software and associated documentation files (the
// "Software"), to deal in the Software without restriction, including
// without limitation the rights to use, copy, modify, merge, publish,
// distribute, sublicense, and/or sell copies of the Software, and to
// permit persons to whom the Software is furnished to do so, subject to
// the following conditions:
// 
// The above copyright notice and this permission notice shall be
// included in all copies or substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
// EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
// MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
// NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
// LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
// OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
// WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

using System;
using System.Diagnostics;
using System.Collections;
using System.Globalization;
using System.Text;
using System.IO;
using PdfSharp.Internal;
using PdfSharp.Pdf.Advanced;
using PdfSharp.Pdf.Internal;
using PdfSharp.Pdf.IO;

namespace PdfSharp.Pdf
{
  /// <summary>
  /// Base class of all composite PDF objects.
  /// </summary>
  public abstract class PdfObject : PdfItem
  {
    protected PdfObject()
    {
    }

    protected PdfObject(PdfDocument document)
    {
      this.Document = document;
    }

    /// <summary>
    /// Initializes a new instance from an existing object. Used for object type transformation.
    /// </summary>
    protected PdfObject(PdfObject obj)
    {
      this.Document = obj.Document;
      if (obj.xref != null)
        obj.xref.Value = this;
    }

    /// <summary>
    /// Creates a copy of this object. The clone does not belong the a document.
    /// </summary>
    public new PdfObject Clone()
    {
      return (PdfObject)Copy();
    }

    protected override object Copy()
    {
      PdfObject obj = (PdfObject)base.Copy();
      obj.document = null;
      obj.xref = null;
      return obj;
    }

    /// <summary>
    /// ...
    /// Setting the object identifier makes this object an indirect object, i.e. the object gets
    /// a PdfXRef entry in the PdfXRefTable.
    /// </summary>
    internal void SetObjectID(int objectNumber, int generationNumber)
    {
      PdfObjectID objectID = new PdfObjectID(objectNumber, generationNumber);

      // TODO: check imported
      if (this.xref == null)
        this.xref = this.document.xrefTable[objectID];
      if (this.xref == null)
      {
        this.xref = new PdfXRef(this);
        this.xref.ObjectID = objectID;
      }
      this.xref.Value = this;
      this.xref.Document = this.document;
    }

    internal void SetObjectID(PdfObjectID objectID)
    {
      if (this.xref == null)
        this.xref = this.document.xrefTable[objectID];
      //this.xref = new PdfXRef(this);
      //this.xref.ObjectID = objectID;
      this.xref.Value = this;
    }

    /// <summary>
    /// Gets or sets the PdfDocument this object belongs to.
    /// </summary>
    internal virtual PdfDocument Document
    {
      get {return this.document;}
      set 
      {
        if (this.document != value)
        {
          if (this.document != null)
            throw new InvalidOperationException("Cannot change document.");
          this.document = value;
          if (this.xref != null)
            this.xref.Document = value;
        }
      }
    }
    protected PdfDocument document;

    /// <summary>
    /// Indicates whether the object is an indirect object.
    /// </summary>
    internal bool IsIndirect
    {
      // An object is an indirect object if and only if is has cross reference value.
      get { return this.xref != null; }
    }

    /// <summary>
    /// When overridden in a derived class, prepares the object to get saved.
    /// </summary>
    internal virtual void PrepareForSave()
    {
    }

    /// <summary>
    /// Saves the stream position. 2nd Edition.
    /// </summary>
    internal override void WriteObject(PdfWriter writer)
    {
      //      Debug.Assert(this.inStreamOffset <= 0);
      //      if (this.inStreamOffset == 0)
      //      {
      //        //this.InStreamOffset = stream.Position;
      //        this.document.xrefTable.AddObject(this);
      //        return Format("{0} {1} obj\n", this.objectID, this.generation);
      //      }
      //      else if (this.inStreamOffset == -1)
      //      {
      //
      //      }
      //      return null;
      //
    }

//    /// <summary>
//    /// Writes the address of the object.
//    /// </summary>
//    protected void WriteObjectAddress(StringBuilder pdf)
//    {
//      pdf.AppendFormat("{0} {1} obj\n", this.ObjectNumber, this.GenerationNumber);
//    }

//    /// <summary>
//    /// Writes the xref entry for the object.
//    /// </summary>
//    internal void WriteXRefEntry(Stream stream)
//    {
//      // Must be exactly 20 bytes
//      WriteString(stream, String.Format("{0:0000000000} 00000 n \n", this.InStreamOffset));
//    }

    /// <summary>
    /// Gets the object identifier.
    /// </summary>
    internal PdfObjectID ObjectID
    {
      get {return this.xref != null ? this.xref.ObjectID : PdfObjectID.Empty;}
    }

    /// <summary>
    /// Gets the object number.
    /// </summary>
    internal int ObjectNumber
    {
      get {return this.ObjectID.ObjectNumber;}
    }

    /// <summary>
    /// Gets the generation number.
    /// </summary>
    internal int GenerationNumber
    {
      get {return this.ObjectID.GenerationNumber;}
    }

    /// <summary>
    /// Creates a deep copy of the specified value and its transitive closure and adds the
    /// new objects to the specified owner document.
    /// </summary>
    internal static PdfObject DeepCopyClosure(PdfDocument owner, PdfObject value)
    {
      // Get transitive closure
      PdfObject[] elements = value.Document.Internals.GetClosure(value);
      int count = elements.Length;
#if DEBUG_
        for (int idx = 0; idx < count; idx++)
        {
          Debug.Assert(elements[idx].XRef != null);
          Debug.Assert(elements[idx].XRef.Document != null);
          Debug.Assert(elements[idx].Document != null);
          if (elements[idx].ObjectID.ObjectNumber == 12)
            GetType();
        }
#endif
      // 1st step. Replace all objects by their clones.
      ImportedObjectTable iot = new ImportedObjectTable(owner);
      for (int idx = 0; idx < count; idx++)
      {
        PdfObject obj = elements[idx];
        PdfObject clone = obj.Clone();
        Debug.Assert(clone.XRef == null);
        clone.Document = owner;
        if (obj.XRef != null)
        {
          // add clone to new owner document
          owner.xrefTable.Add(clone);
          // the clone gets an xref by adding it to its new owner
          Debug.Assert(clone.XRef != null);
          // save association from old object identifier to new xref
          iot.Add(obj.ObjectID, clone.XRef);
        }
        else
        {
          // only the root object can be a direct object
          Debug.Assert(idx == 0);
          ////////// add clone to new owner document
          ////////owner.xrefTable.Add(clone);
          ////////// the clone gets an xref by adding it to its new owner
          ////////Debug.Assert(clone.XRef != null);
        }
        // replace external object by its clone
        elements[idx] = clone;
      }
#if DEBUG_
        for (int idx = 0; idx < count; idx++)
        {
          Debug.Assert(elements[idx].XRef != null);
          Debug.Assert(elements[idx].XRef.Document != null);
          Debug.Assert(resources[idx].Document != null);
          if (elements[idx].ObjectID.ObjectNumber == 12)
            GetType();
        }
#endif

      // 2nd step. Fix up all indirect references that still refers to the import document.
      for (int idx = 0; idx < count; idx++)
      {
        PdfObject obj = elements[idx];
        Debug.Assert(obj.Document == owner);
        FixUpObject(iot, owner, obj);
      }
      // return the root object
      return elements[0];
    }

    /// <summary>
    /// Replace all indirect references to external objects by their cloned counterparts
    /// owned by the importer document.
    /// </summary>
    static void FixUpObject(ImportedObjectTable iot, PdfDocument owner, PdfObject value)
    {
      // TODO: merge with PdfFormXObject.FixUpObject
      PdfDictionary dict;
      PdfArray array;
      if ((dict = value as PdfDictionary) != null)
      {
        // Set document for cloned direct objects
        if (dict.Document == null)
          dict.Document = owner;
        else
          Debug.Assert(dict.Document == owner);

        // Search for indirect references in all keys
        PdfName[] names = dict.Elements.Keys;
        foreach (PdfName name in names)
        {
          PdfItem item = dict.Elements[name];
          // Is item an xref?
          PdfXRef xref = item as PdfXRef;
          if (xref != null)
          {
            // Does the xref already belongs to the owner?
            if (xref.Document == owner)
            {
              // Yes: fine. Happens when an already cloned object is reused.
              continue;
            }
            else
            {
              //Debug.Assert(xref.Document == iot.Document);
              // No: replace with xref of cloned object
              PdfXRef newXRef = iot[xref.ObjectID];
              Debug.Assert(newXRef != null);
              Debug.Assert(newXRef.Document == owner);
              dict.Elements[name] = newXRef;
            }
          }
          else if (item is PdfObject)
          {
            // Fix up inner objects
            FixUpObject(iot, owner, (PdfObject)item);
          }
        }
      }
      else if ((array = value as PdfArray) != null)
      {
        // Set document for cloned direct objects
        if (array.Document == null)
          array.Document = owner;
        else
          Debug.Assert(array.Document == owner);

        // Search for indirect references in all array elements
        int count = array.Elements.Count;
        for (int idx = 0; idx < count; idx++)
        {
          PdfItem item = array.Elements[idx];
          // Is item an xref?
          PdfXRef xref = item as PdfXRef;
          if (xref != null)
          {
            // Does the xref already belongs to the owner?
            if (xref.Document == owner)
            {
              // Yes: fine. Happens when an already cloned object is reused.
              continue;
            }
            else
            {
              //Debug.Assert(xref.Document == iot.Document);
              // No: replace with xref of cloned object
              PdfXRef newXRef = iot[xref.ObjectID];
              Debug.Assert(newXRef != null);
              Debug.Assert(newXRef.Document == owner);
              array.Elements[idx] = newXRef;
            }
          }
          else if (item is PdfObject)
          {
            // Fix up inner objects
            FixUpObject(iot, owner, (PdfObject)item);
          }
        }
      }
    }

    ///// <summary>
    ///// Gets or sets the generation.
    ///// </summary>
    //internal int ObjectGeneration
    //{
    //  get {return this.o.generation;}
    //  set {this.generation = value;}
    //}

    internal PdfXRef XRef
    {
      get {return this.xref;}
      set 
      {
        //Debug.Assert(value.Value == null);
        //Debug.Assert(value.Value == null);
        this.xref = value;
      }
    }
    protected PdfXRef xref;
  }
}
