//
// PDFSharp - A library for processing PDF
//
// Authors:
//   Stefan Lange (mailto:Stefan.Lange@pdfsharp.com)
//
// Copyright (c) 2005 empira Software GmbH, Cologne (Germany)
//
// http://www.pdfsharp.com
//
// Permission is hereby granted, free of charge, to any person obtaining
// a copy of this software and associated documentation files (the
// "Software"), to deal in the Software without restriction, including
// without limitation the rights to use, copy, modify, merge, publish,
// distribute, sublicense, and/or sell copies of the Software, and to
// permit persons to whom the Software is furnished to do so, subject to
// the following conditions:
// 
// The above copyright notice and this permission notice shall be
// included in all copies or substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
// EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
// MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
// NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
// LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
// OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
// WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

using System;
using System.Diagnostics;
using System.Collections;
using System.Reflection;
using System.Text;
using System.IO;
using PdfSharp.Internal;
using PdfSharp.Pdf;
using PdfSharp.Pdf.Advanced;
using PdfSharp.Pdf.Internal;
using PdfSharp.Pdf.IO;
using PdfSharp.Pdf.Security;

namespace PdfSharp.Pdf
{
  enum DocumentState
  {
    Created  = 0x0001,
    Imported = 0x0002,
    //Modifyable,
    //ReadOnly,
    //Information,
  }

  /// <summary>
  /// Represents a PDF document.
  /// </summary>
  public sealed class PdfDocument : PdfObject
  {
    internal DocumentState state;

    /// <summary>
    /// Creates a new PDF document in memory.
    /// To open an existing PDF file, use the PdfReader class.
    /// </summary>
    public PdfDocument()
    {
      this.creation = DateTime.Now;
      this.state = DocumentState.Created;
      this.version = 14;
      Initialize();
      Info.CreationDate = this.creation;
    }

    /// <summary>
    /// Creates a new PDF document with the specified file name.
    /// To open an existing PDF file, use the PdfReader class.
    /// </summary>
    public PdfDocument(string filename)
    {
      this.creation = DateTime.Now;
      this.state = DocumentState.Created;
      this.version = 14;
      Initialize();
      Info.CreationDate = this.creation;

      this.outStream = new FileStream(filename, FileMode.Create);
    }

    /// <summary>
    /// Creates a new PDF document using the specified stream.
    /// To open an existing PDF file, use the PdfReader class.
    /// </summary>
    public PdfDocument(Stream outputStream)
    {
      this.creation = DateTime.Now;
      this.state = DocumentState.Created;
      Initialize();
      Info.CreationDate = this.creation;

      this.outStream = outputStream;
    }

    internal PdfDocument(Lexer lexer)
    {
      this.creation = DateTime.Now;
      this.state = DocumentState.Imported;

      //this.info = new PdfInfo(this);
      //this.pages = new PdfPages(this);
      //this.fontTable = new PdfFontTable();
      //this.catalog = new PdfCatalog(this);
      ////this.font = new PdfFont();
      //this.objects = new PdfObjectTable(this);
      //this.trailer = new PdfTrailer(this);
      this.xrefTable = new PdfXRefTable(this);
      this.lexer = lexer;
    }

    void Initialize()
    {
      //this.info       = new PdfInfo(this);
      this.fontTable  = new PdfFontTable(this);
      this.imageTable = new PdfImageTable(this);
      this.trailer    = new PdfTrailer(this);
      this.xrefTable  = new PdfXRefTable(this);
      this.trailer.CreateNewDocumentIDs();
    }

    internal bool CanModify
    {
      //get {return this.state == DocumentState.Created || this.state == DocumentState.Modifyable;}
      get {return true;}
    }

    public void Close()
    {
      if (!CanModify)
        throw new InvalidOperationException(PSSR.CannotModify);

      if (this.outStream != null)
      {
        // Get security handler if document gets encrypted
        PdfStandardSecurityHandler securityHandler = null;
        if (SecuritySettings.DocumentSecurityLevel != PdfDocumentSecurityLevel.None)
          securityHandler = SecuritySettings.SecurityHandler;

        PdfWriter writer = new PdfWriter(this.outStream, securityHandler);
        try
        {
          DoSave(writer);
        }
        finally
        {
          if (writer != null)
            writer.Close();
        }
      }
    }

    /// <summary>
    /// Saves the document to the specified path. If a file already exists, it will be overwritten.
    /// </summary>
    public void Save(string path)
    {
      if (!CanModify)
        throw new InvalidOperationException(PSSR.CannotModify);

      Stream stream = new FileStream(path, FileMode.Create, FileAccess.Write);
      Save(stream);
    }

    /// <summary>
    /// Saves the document to the specified stream.
    /// </summary>
    public void Save(Stream stream, bool closeStream)
    {
      if (!CanModify)
        throw new InvalidOperationException(PSSR.CannotModify);

      // TODO: more diagnostic checks
      string message = "";
      if (!CanSave(ref message))
        throw new PdfSharpException(message);

      // Get security handler if document gets encrypted
      PdfStandardSecurityHandler securityHandler = null;
      if (SecuritySettings.DocumentSecurityLevel != PdfDocumentSecurityLevel.None)
        securityHandler = SecuritySettings.SecurityHandler;

      PdfWriter writer = new PdfWriter(stream, securityHandler);
      try
      {
        DoSave(writer);
      }
      finally
      {
        if (stream != null && closeStream)
          stream.Close();
        if (writer != null)
          writer.Close(closeStream);
      }
    }

    /// <summary>
    /// Saves the document to the specified stream and closes the stream.
    /// </summary>
    public void Save(Stream stream)
    {
      Save(stream, true);
    }

    void DoSave(PdfWriter writer)
    {
      try
      {
        bool encrypt = this.securitySettings.DocumentSecurityLevel != PdfDocumentSecurityLevel.None;
        if (encrypt)
        {
          PdfStandardSecurityHandler securityHandler = this.securitySettings.SecurityHandler;
          if (securityHandler.XRef == null)
            this.xrefTable.Add(securityHandler);
          else
            Debug.Assert(this.xrefTable.Contains(securityHandler.ObjectID));
          this.trailer.Elements[PdfTrailer.Keys.Encrypt] = this.securitySettings.SecurityHandler.XRef;
        }
        else
          this.trailer.Elements.Remove(PdfTrailer.Keys.Encrypt);

        PrepareForSave();

        if (encrypt)
          this.securitySettings.SecurityHandler.PrepareEncryption();

        writer.WriteFileHeader(this);
        PdfXRef[] xrefs = this.xrefTable.AllXRefs;
        int count = xrefs.Length;
        for (int idx = 0; idx < count; idx++)
        {
          PdfXRef xref = xrefs[idx];
#if DEBUG_
          if (xref.ObjectNumber == 378)
            GetType();
#endif
          xref.Position = writer.Position;
          xref.Value.WriteObject(writer);
        }
        int startxref = writer.Position;
        this.xrefTable.WriteObject(writer);
        writer.WriteRaw("trailer\n");
        this.trailer.Elements.SetInteger("/Size", count + 1);
        this.trailer.WriteObject(writer);
        writer.WriteEof(this, startxref);

        //if (encrypt)
        //{
        //  this.state &= ~DocumentState.SavingEncrypted;
        //  //this.securitySettings.SecurityHandler.EncryptDocument();
        //}
      }
      finally
      {
        if (writer != null)
        {
          writer.Stream.Flush();
          // DO NOT CLOSE WRITER HERE
          //writer.Close();
        }
      }
    }

    /// <summary>
    /// Dispatches PrepareForSave to the objects that need it.
    /// </summary>
    internal override void PrepareForSave()
    {
      PdfDocumentInformation info = this.Info;

      // Set Creator if value is undefined
      if (info.Elements[PdfDocumentInformation.Keys.Creator] == null)
        info.Creator = VersionInfo.Producer;

      // Keep original producer if file was imported
      string producer = info.Producer;
      if (producer.Length == 0)
        producer = VersionInfo.Producer;
      else
      {
        // Prevent endless concatenation if file is edited with PDFsharp more than once
        if (!producer.StartsWith(VersionInfo.Title))
          producer = VersionInfo.Producer + " (Original: " + producer + ")";
      }
      info.Elements.SetString(PdfDocumentInformation.Keys.Producer, producer);

      // Let catalog do the rest
      this.catalog.PrepareForSave();

#if true
      // Remove all unreachable objects (e.g. from deleted pages)
      int removed = this.xrefTable.Compact();
      if (removed != 0)
        Debug.WriteLine("PrepareForSave: Number of deleted unreachable objects: " + removed.ToString());
      this.xrefTable.Renumber();
#endif
    }

    /// <summary>
    /// Determines whether the document can be saved.
    /// </summary>
    public bool CanSave(ref string message)
    {
      if (!SecuritySettings.CanSave(ref message))
        return false;

      return true;
    }

    internal bool HasVersion(string version)
    {
      return String.Compare(Catalog.Version, version) >= 0;
    }

    public PdfDocumentOptions Options
    {
      get 
      {
        if (this.options == null)
           this.options = new PdfDocumentOptions(this);
        return this.options;
      }
    }
    PdfDocumentOptions options;

    /// <summary>
    /// NYI Indicates whether large objects are written immediately to the output stream to relieve
    /// memory consumption.
    /// </summary>
    internal bool EarlyWrite
    {
      get {return false;}
    }

    // Some old test code -> delete
#if false
    public static void Test01()
    {
      FileStream filestream = null;
      try
      {
        //PdfPages.Keys.Meta;
        //filestream = new FileStream("..\\..\\Dokument1.pdf", FileMode.Open);
        //filestream = new FileStream("..\\..\\HelloWorld.pdf", FileMode.Open);
        filestream = new FileStream("..\\..\\HelloWorld.pdf", FileMode.Open);
        //filestream = new FileStream("..\\..\\Linearized.pdf", FileMode.Open);
        int length = (int)filestream.Length;
        byte[] bytes = new byte[length];
        filestream.Read(bytes, 0, length);
        char[] chars = new char[length];
        //Decoder decoder = Encoding.UTF8.GetDecoder();
        Decoder decoder = PdfEncoders.WinAnsiEncoding.GetDecoder();
        decoder.GetChars(bytes, 0, length, chars, 0);
        string pdf = new string(chars);
        int pdflen = pdf.Length;
        Lexer lexer = new Lexer(filestream);
        PdfDocument document = new PdfDocument(lexer);
        document.ReadTrailer();
        object o;
        o = document.trailer.Elements["/info"];
        o = document.trailer.Elements["/Info"];

        PdfXRef xref = (PdfXRef)o;
        o = xref.Value;

        Parser parser = new Parser(document);
        PdfCatalog catalog = new PdfCatalog(document);
        //o = parser.ReadObject(catalog, "2.0", false);

        //PdfCatalog catalog = (PdfCatalog)o;
        KeysMeta meta = PdfCatalog.Keys.Meta;
        KeyDescriptor desc = meta["/Pages"];
        Type objType = desc.ObjectType;
        ConstructorInfo ctorInfo = objType.GetConstructor(
          BindingFlags.Public | BindingFlags.NonPublic | BindingFlags.Instance, null,
          new Type[]{typeof(PdfDocument)}, null);
        o = ctorInfo.Invoke(new object[]{document});

        //o = desc.CreateObject(objType, document);

//        object[] attribs = objType.GetCustomAttributes(typeof(PdfObjectInfoAttribute), false);
//        if (attribs.Length == 1)
//          o = attribs[0];

        //o = parser.ReadObject(null, "4", false);
        //object page = PdfPage.FromDictionary((PdfDictionary)o);

        //o = parser.ReadObject(null, "5", false);
        //object page = PdfPage.FromDictionary((PdfDictionary)o);

        meta.GetType();


        o.GetType();
//
//        document.catalog = document.ReadCatalog(document.trailer[PdfTrailer.Keys.Root]);
//        PdfPages pages = document.catalog.Pages;
//        int count = pages.Count;
//        for (int idx = 0; idx < count; idx++)
//        {
//          PdfPage page = pages[idx];
//          page.ReadContents();
//          PdfFontMap map = page.Resources.Fonts;
//        }
//        lexer.GetType();
      }
      finally
      {
        if (filestream != null)
          filestream.Close();
      }
    }

    public static void Test02()
    {
      FileStream filestream = null;
      try
      {
        string s;
        s = PSSR.GetString(PSMsgID.SampleMessage1);
        s = PSSR.GetString(PSMsgID.SampleMessage2);

        s.GetType();


        GC.Collect();
        int allocated = (int)GC.GetTotalMemory(true);

        //PdfPages.Keys.Meta;
        //filestream = new FileStream("..\\..\\HalloPDF.pdf", FileMode.Open);
        //filestream = new FileStream("..\\..\\Dokumentobjektmodell.pdf", FileMode.Open);
        filestream = new FileStream("..\\..\\Zeichenstift.pdf", FileMode.Open);
        //filestream = new FileStream("..\\..\\Linearized.pdf", FileMode.Open);
        int length = (int)filestream.Length;
        byte[] bytes = new byte[length];
        filestream.Read(bytes, 0, length);
        //char[] chars = new char[length];
        ////Decoder decoder = Encoding.UTF8.GetDecoder();
        //Decoder decoder = PdfEncoders.WinAnsiEncoding.GetDecoder();
        //decoder.GetChars(bytes, 0, length, chars, 0);
        //string pdf = new string(chars);
        //int pdflen = pdf.Length;
        Lexer lexer = new Lexer(filestream);
        PdfDocument document = new PdfDocument(lexer);
        document.ReadTrailer();

        string[] addresses = null; //document.xrefTable.AllAddresses;
        int count = addresses.Length;
        PdfObject[] objects = new PdfObject[count];
        GC.Collect();
        Debug.WriteLine((GC.GetTotalMemory(true) - allocated).ToString());
        allocated = (int)GC.GetTotalMemory(true);

        FileStream fs = new FileStream("outfile.pdf", FileMode.Create, FileAccess.Write, FileShare.ReadWrite);
        PdfWriter writer = new PdfWriter(fs);
        Parser parser = new Parser(document);
        PdfObject pdfObject = null;
        int startxref;
        writer.WriteRaw("%PDF-1.3\n%xxxxx\n");
        for (int idx = 0; idx < count; idx++)
        {
          //pdfObject = parser.ReadObject(null, addresses[idx], false);
          //objects[idx] = pdfObject;
//          pdfObject.InStreamOffset = writer.Position + 1;
          //document.xrefTable.GetEntry(idx + 1).Offset = writer.Position;
          //pdfObject.WriteObject(writer);
          //string pdfText = pdf.ToString();
          //fs.Write(PdfEncoders.RawEncoding.GetBytes(pdfText), 0, pdfText.Length);
          pdfObject.GetType();
        }
        startxref = writer.Position;
        document.xrefTable.WriteObject(writer.Stream);
        //document.trailer.ObjectID = -1;
        writer.WriteRaw("trailer\n");
        document.trailer.WriteObject(writer);
        writer.WriteRaw("startxref\n");
        writer.WriteRaw(startxref.ToString());
        writer.WriteRaw("\n%%EOF\n");
        writer.Close();
        GC.Collect();
        Debug.WriteLine((GC.GetTotalMemory(true) - allocated).ToString());

        objects.GetType();
        addresses.GetType();
      }
      finally
      {
        if (filestream != null)
          filestream.Close();
      }
    }

    public static void Test03()
    {
      FileStream filestream = null;
      try
      {
        int allocated = (int)GC.GetTotalMemory(true);
        //filestream = new FileStream("..\\..\\HalloPDF.pdf", FileMode.Open);
        filestream = new FileStream("..\\..\\HelloWorld.pdf", FileMode.Open);
        //filestream = new FileStream("..\\..\\1000 Seiten Handbuch.pdf", FileMode.Open);
        //filestream = new FileStream("..\\..\\Dokumentobjektmodell.pdf", FileMode.Open);
        //filestream = new FileStream("..\\..\\Zeichenstift.pdf", FileMode.Open);
        //filestream = new FileStream("..\\..\\Linearized.pdf", FileMode.Open);
        int length = (int)filestream.Length;
        byte[] bytes = new byte[length];
        filestream.Read(bytes, 0, length);
        Lexer lexer = new Lexer(filestream);
        PdfDocument inputDocument = new PdfDocument(lexer);
        PdfDocument outputDocument = new PdfDocument();
        outputDocument.xrefTable = new PdfXRefTable(outputDocument);
        inputDocument.ReadTrailer();

        PdfObjectID[] objectIDs = inputDocument.xrefTable.AllObjectIDs;
        int count = objectIDs.Length;
        PdfObject[] objects = new PdfObject[count];
        int[] positions = new int[count];
        GC.Collect();
        Debug.WriteLine((GC.GetTotalMemory(true) - allocated).ToString());
        allocated = (int)GC.GetTotalMemory(true);

        FileStream fs = new FileStream("outfile.pdf", FileMode.Create, FileAccess.Write, FileShare.ReadWrite);
        PdfWriter writer = new PdfWriter(fs);
        try
        {
          Parser parser = new Parser(inputDocument);
          PdfObject pdfObject = null;
          int startxref;
          writer.WriteRaw("%PDF-1.3\n%xxxxx\n");
          for (int idx = 0; idx < count; idx++)
          {
            pdfObject = parser.ReadObject(null, objectIDs[idx], false);
            pdfObject.XRef = inputDocument.xrefTable[objectIDs[idx]];
            objects[idx] = pdfObject;
            //pdfObject.InStreamOffset = writer.Position + 1;
            //outputDocument.xrefTable.GetEntry(objectIDs[idx]).Position = writer.Position;
            positions[idx] = writer.Position;
            pdfObject.WriteObject(writer);
            //string pdfText = pdf.ToString();
            //fs.Write(PdfEncoders.RawEncoding.GetBytes(pdfText), 0, pdfText.Length);
            pdfObject.GetType();
          
          }
          PdfXRef[] reachables = inputDocument.xrefTable.TransitiveClosure(inputDocument.trailer);
          reachables.GetType();

          for (int idx = 0; idx < count; idx++)
          {
            PdfXRef xref = inputDocument.xrefTable[objectIDs[idx]];
            xref.Position = positions[idx];
          }
          startxref = writer.Position;
          inputDocument.xrefTable.WriteObject(writer.Stream);
          //document.trailer.ObjectID = -1;
          writer.WriteRaw("trailer\n");
          inputDocument.trailer.Elements.SetInteger("/Size", count + 1);
          inputDocument.trailer.WriteObject(writer);
          writer.WriteRaw("startxref\n");
          writer.WriteRaw(startxref.ToString());
          writer.WriteRaw("\n%%EOF\n");
        }
        finally
        {
          writer.Close();
        }
        GC.Collect();
        Debug.WriteLine((GC.GetTotalMemory(true) - allocated).ToString());

        objects.GetType();
        objectIDs.GetType();
      }
      finally
      {
        if (filestream != null)
          filestream.Close();
      }
    }

    internal static int TotalMemory
    {
      get {return (int)GC.GetTotalMemory(false);}
    }

    public static PdfDocument Import(string path, PdfImportOption options)
    {
      PdfDocument document = null;
      Stream stream = new FileStream(path, FileMode.Open, FileAccess.Read);
      try
      {
        document = PdfDocument.Import(stream);
      }
      finally
      {
        if (stream != null)
          stream.Close();
      }
      return document;
    }

    public static PdfDocument Import(string path)
    {
      return Import(path, PdfImportOption.ReadForExtraction);
    }

    public static PdfDocument Import(Stream stream, PdfImportOption options)
    {
      PdfDocument document = null;
      try
      {
        Lexer lexer = new Lexer(stream);
        document = new PdfDocument(lexer);
        document.state |= PdfDocumentState.Imported;

        // Read all trailers
        document.xrefTable.IsUnderConstruction = true;
        document.ReadTrailer();

        PdfXRef[] xrefs = document.xrefTable.AllXRefs;
        int count = xrefs.Length;

        // Read all indirect objects
        Parser parser = new Parser(document);
        for (int idx = 0; idx < count; idx++)
        {
          PdfXRef xref = xrefs[idx];
          PdfObject pdfObject = parser.ReadObject(null, xref.ObjectID, false);
          Debug.Assert(pdfObject.XRef == xref);
          pdfObject.XRef = xref;
          Debug.Assert(pdfObject.XRef.Value != null, "something got wrong");
        }

        // Fix references of trailer values and then objects and xrefs are consistent.
        document.trailer.Finish();

        // Some tests...
        PdfXRef[] reachables = document.xrefTable.TransitiveClosure(document.trailer);
        reachables.GetType();
        reachables = document.xrefTable.AllXRefs;
        document.xrefTable.CheckConsistence();

        // Remove all unreachable objects
        int removed = document.xrefTable.Compact();
        if (removed != 0)
          Debug.WriteLine("Number of deleted unreachable objects: " + removed.ToString());

        // Fore flattening of page tree
        PdfPages pages = document.Pages;

        document.xrefTable.CheckConsistence();
        document.xrefTable.Renumber();
        document.xrefTable.CheckConsistence();
      }
      finally
      {
        //if (filestream != null)
        //  filestream.Close();
      }
      return document;
    }

    public static PdfDocument Import(Stream stream)
    {
      return Import(stream, PdfImportOption.ReadForExtraction);
    }
#endif
    /// <summary>
    /// Gets or sets the PDF version number.
    /// </summary>
    public int Version
    {
      get {return this.version;}
      set
      {
        if (!CanModify)
          throw new InvalidOperationException(PSSR.CannotModify);
        if (value < 12 || value > 16) // TODO not realy implemented
          throw new ArgumentException(PSSR.InvalidVersionNumber, "value");
        this.version = value;
      }
    }
    internal int version;

    /// <summary>
    /// Gets the number of pages in the document.
    /// </summary>
    public int PageCount
    {
      get 
      {
        if (CanModify)
          return Pages.Count;
        // PdfOpenMode is InformationOnly
        PdfDictionary pageTreeRoot = (PdfDictionary)Catalog.Elements.GetIndirectObject(PdfCatalog.Keys.Pages);
        return pageTreeRoot.Elements.GetInteger(PdfPages.Keys.Count);
      }
    }

    /// <summary>
    /// Gets the file size of the document.
    /// </summary>
    public long FileSize
    {
      get { return this.fileSize; }
    }
    internal long fileSize;

    public bool IsImported
    {
      get {return (this.state & DocumentState.Imported) != 0;}
    }

    internal Exception DocumentNotImported()
    {
      return new InvalidOperationException("Document not imported.");
    }

    /// <summary>
    /// Gets information about the document.
    /// </summary>
    public PdfDocumentInformation Info
    {
      get 
      {
        if (this.info == null)
          this.info = this.trailer.Info;
        return this.info;
      }
    }
    PdfDocumentInformation info;  // never changes if once created

    /// <summary>
    /// Get the pages dictionary.
    /// </summary>
    public PdfPages Pages
    { 
      get
      {
        if (this.pages == null)
          this.pages = Catalog.Pages;
        return this.pages;
      }
    }
    PdfPages pages;  // never changes if once created

    /// <summary>
    /// Gets or sets a value specifying the page layout to be used when the document is opened.
    /// </summary>
    public PdfPageLayout PageLayout
    {
      get {return Catalog.PageLayout;}
      set 
      {
        if (!CanModify)
          throw new InvalidOperationException(PSSR.CannotModify);
        Catalog.PageLayout = value;
      }
    }

    /// <summary>
    /// Gets or sets a value specifying how the document should be displayed when opened.
    /// </summary>
    public PdfPageMode PageMode
    {
      get {return Catalog.PageMode;}
      set 
      {
        if (!CanModify)
          throw new InvalidOperationException(PSSR.CannotModify);
        Catalog.PageMode = value;
      }
    }

    /// <summary>
    /// Gets the viewer preverences of this document.
    /// </summary>
    public PdfViewerPreferences ViewerPreferences
    {
      get {return Catalog.ViewerPreferences;}
    }

    /// <summary>
    /// Gets the root of the outline (or bookmark) tree.
    /// </summary>
    public PdfOutline.PdfOutlineCollection Outlines
    {
      get {return Catalog.Outlines;}
    }

    /// <summary>
    /// Gets the security settings of this document.
    /// </summary>
    public PdfSecuritySettings SecuritySettings
    {
      get
      {
        if (this.securitySettings == null)
          this.securitySettings = new PdfSecuritySettings(this);
        return this.securitySettings;
      }
    }
    internal PdfSecuritySettings securitySettings;

    /// <summary>
    /// Gets the document font table that holds all fonts used in the current document.
    /// </summary>
    internal PdfFontTable FontTable
    {
      get 
      {
        if (this.fontTable == null)
          this.fontTable = new PdfFontTable(this);
        return this.fontTable;
      }
    }
    PdfFontTable fontTable;

    /// <summary>
    /// Gets the document image table that holds all images used in the current document.
    /// </summary>
    internal PdfImageTable ImageTable
    {
      get 
      {
        if (this.imageTable == null)
          this.imageTable = new PdfImageTable(this);
        return this.imageTable;
      }
    }
    PdfImageTable imageTable;

    /// <summary>
    /// Gets the document form table that holds all form external objects used in the current document.
    /// </summary>
    internal PdfFormXObjectTable FormTable
    {
      get 
      {
        if (this.formTable == null)
          this.formTable = new PdfFormXObjectTable(this);
        return this.formTable;
      }
    }
    PdfFormXObjectTable formTable;

    /// <summary>
    /// Gets the document ExtGState table that holds all form state objects used in the current document.
    /// </summary>
    internal PdfExtGStateTable ExtGStateTable
    {
      get 
      {
        if (this.extGStateTable == null)
          this.extGStateTable = new PdfExtGStateTable(this);
        return this.extGStateTable;
      }
    }
    PdfExtGStateTable extGStateTable;

    /// <summary>
    /// Gets the PdfCatalog of the current document.
    /// </summary>
    internal PdfCatalog Catalog
    {
      get 
      {
        if (this.catalog == null)
          this.catalog = this.trailer.Root;
        return catalog;
      }
    }
    PdfCatalog catalog;  // never changes if once created

    /// <summary>
    /// Gets the PdfInternals object of this document, that grants access to some internal structures
    /// which are not part of the public interface of PdfDocument.
    /// </summary>
    public PdfInternals Internals
    {
      get
      {
        if (this.internals == null)
          this.internals = new PdfInternals(this);
        return this.internals;
      }
    }
    PdfInternals internals;

    /// <summary>
    /// Creates a new page and adds it to this document.
    /// </summary>
    public PdfPage AddPage()
    {
      if (!CanModify)
        throw new InvalidOperationException(PSSR.CannotModify);
      return Catalog.Pages.Add();
    }

    /// <summary>
    /// Adds the specified page to this document. If the page is from an external document,
    /// it is imported to this document. In this case the returned page is not the same
    /// object as the specified one.
    /// </summary>
    public PdfPage AddPage(PdfPage page)
    {
      if (!CanModify)
        throw new InvalidOperationException(PSSR.CannotModify);
      return Catalog.Pages.Add(page);
    }

    /// <summary>
    /// Creates a new page and inserts it in this document at the specified position.
    /// </summary>
    public PdfPage InsertPage(int index)
    {
      if (!CanModify)
        throw new InvalidOperationException(PSSR.CannotModify);
      return Catalog.Pages.Insert(index);
    }

    /// <summary>
    /// Inserts the specified page in this document. If the page is from an external document,
    /// it is imported to this document. In this case the returned page is not the same
    /// object as the specified one.
    /// </summary>
    public PdfPage InsertPage(int index, PdfPage page)
    {
      if (!CanModify)
        throw new InvalidOperationException(PSSR.CannotModify);
      return Catalog.Pages.Insert(index, page);
    }

    /// <summary>
    /// Gets the security handler.
    /// </summary>
    public PdfStandardSecurityHandler SecurityHandler
    {
      get {return this.trailer.SecurityHandler;}
    }

    internal PdfTrailer trailer;
    internal PdfXRefTable xrefTable;
    internal Stream outStream;
    
    // Imported Document
    internal Lexer lexer;

    internal DateTime creation;

    /// <summary>
    /// Gets the ThreadLocalStorage object. It is used for caching objects that should created
    /// only once.
    /// </summary>
    internal static ThreadLocalStorage Tls
    {
      get
      {
        if (PdfDocument.tls == null)
          PdfDocument.tls = new ThreadLocalStorage();
        return PdfDocument.tls;
      }
    }
    [ThreadStatic]
    static ThreadLocalStorage tls;
  }
}
