//
// PDFSharp - A library for processing PDF
//
// Authors:
//   Stefan Lange (mailto:Stefan.Lange@pdfsharp.com)
//
// Copyright (c) 2005 empira Software GmbH, Cologne (Germany)
//
// http://www.pdfsharp.com
//
// Permission is hereby granted, free of charge, to any person obtaining
// a copy of this software and associated documentation files (the
// "Software"), to deal in the Software without restriction, including
// without limitation the rights to use, copy, modify, merge, publish,
// distribute, sublicense, and/or sell copies of the Software, and to
// permit persons to whom the Software is furnished to do so, subject to
// the following conditions:
// 
// The above copyright notice and this permission notice shall be
// included in all copies or substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
// EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
// MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
// NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
// LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
// OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
// WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

using System;
using System.Diagnostics;
using System.Collections;
using System.Globalization;
using System.Text;
using System.IO;
using PdfSharp.Pdf;
using PdfSharp.Pdf.Advanced;
using PdfSharp.Pdf.Security;
using PdfSharp.Pdf.Internal;
using PdfSharp.Internal;

namespace PdfSharp.Pdf.IO
{
  /// <summary>
  /// Encapsulates the arguments of the PdfPasswordProvider delegate.
  /// </summary>
  public class PdfPasswordProviderArgs
  {
    /// <summary>
    /// Sets the password to open the document with.
    /// </summary>
    public string Password;

    /// <summary>
    /// When set to true the PdfReader.Open function returns null indicating that no PdfDocument was created.
    /// </summary>
    public bool Abort;
  }

  /// <summary>
  /// A delegated used by the PdfReader.Open function to retrieve a password if the document is protected.
  /// </summary>
  public delegate void PdfPasswordProvider(PdfPasswordProviderArgs args);

  /// <summary>
  /// Represents the functionality for reading PDF documents.
  /// </summary>
  public sealed class PdfReader
  {
    // Makes this class static
    PdfReader()
    {
    }

    /// <summary>
    /// Determines whether the specified path is a PDF file by inspecting the first eight bytes
    /// of the file. If the file header has the form %PDF-x.y the function returns the version
    /// number as integer (e.g. 14 for PDF 1.4). If the file header is invalid or inaccessible
    /// for any reason, 0 is returned. The function never throws an exception. 
    /// </summary>
    public static int TestPdfFile(string path)
    {
      FileStream stream = null;
      try
      {
        stream = new FileStream(path, FileMode.Open, FileAccess.Read, FileShare.ReadWrite);
        byte[] bytes = new byte[1024];
        stream.Read(bytes, 0, 1024);
        return GetPdfFileVersion(bytes);
      }
      catch
      {
        // just return 0
      }
      finally
      {
        if (stream != null)
          stream.Close();
      }
      return 0;
    }

    /// <summary>
    /// Implements scanning the PDF file version.
    /// </summary>
    internal static int GetPdfFileVersion(byte[] bytes)
    {
      try
      {
        // Acrobat accepts headers like %!PS-Adobe-N.n PDF-M.m...
        string header = Encoding.ASCII.GetString(bytes);
        if (header[0] == '%')
        {
          int ich = header.IndexOf("PDF-");
          if (ich > 0 && header[ich + 5] == (byte)'.')
          {
            char major = header[ich + 4];
            char minor = header[ich + 6];
            if (major >= '1' && major < '2' && minor >= '0' && minor <= '9')
              return ((int)(major - '0')) * 10 + (int)(minor - '0');
          }
        }
      }
      catch {}
      return 0;
    }

    /// <summary>
    /// Opens an existing PDF document.
    /// </summary>
    public static PdfDocument Open(string path, PdfDocumentOpenMode openmode)
    {
      return Open(path, null, openmode, null);
    }

    /// <summary>
    /// Opens an existing PDF document.
    /// </summary>
    public static PdfDocument Open(string path, PdfDocumentOpenMode openmode, PdfPasswordProvider provider)
    {
      return Open(path, null, openmode, provider);
    }

    /// <summary>
    /// Opens an existing PDF document.
    /// </summary>
    public static PdfDocument Open(string path, string password, PdfDocumentOpenMode openmode)
    {
      return Open(path, password, openmode, null);
    }

    /// <summary>
    /// Opens an existing PDF document.
    /// </summary>
    public static PdfDocument Open(string path, string password, PdfDocumentOpenMode openmode, PdfPasswordProvider provider)
    {
      PdfDocument document = null;
      Stream stream = new FileStream(path, FileMode.Open, FileAccess.Read);
      try
      {
        document = PdfReader.Open(stream, password, openmode, provider);
      }
      finally
      {
        if (stream != null)
          stream.Close();
      }
      return document;
    }

    /// <summary>
    /// Opens an existing PDF document.
    /// </summary>
    public static PdfDocument Open(string path)
    {
      return Open(path, null, PdfDocumentOpenMode.Modify, null);
    }

    /// <summary>
    /// Opens an existing PDF document.
    /// </summary>
    public static PdfDocument Open(string path, string password)
    {
      return Open(path, password, PdfDocumentOpenMode.Modify, null);
    }

    /// <summary>
    /// Opens an existing PDF document.
    /// </summary>
    public static PdfDocument Open(Stream stream, PdfDocumentOpenMode openmode)
    {
      return Open(stream, null, openmode);
    }

    /// <summary>
    /// Opens an existing PDF document.
    /// </summary>
    public static PdfDocument Open(Stream stream, PdfDocumentOpenMode openmode, PdfPasswordProvider passwordProvider)
    {
      return Open(stream, null, openmode);
    }
    /// <summary>
    /// Opens an existing PDF document.
    /// </summary>
    public static PdfDocument Open(Stream stream, string password, PdfDocumentOpenMode openmode)
    {
      return Open(stream, password, openmode, null);
    }

    /// <summary>
    /// Opens an existing PDF document.
    /// </summary>
    public static PdfDocument Open(Stream stream, string password, PdfDocumentOpenMode openmode, PdfPasswordProvider passwordProvider)
    {
      PdfDocument document = null;
      try
      {
        Lexer lexer = new Lexer(stream);
        document = new PdfDocument(lexer);
        document.state |= DocumentState.Imported;
        document.fileSize = stream.Length;

        // Get file version
        byte[] header = new byte[1024];
        stream.Position = 0;
        stream.Read(header, 0, 1024);
        document.version = GetPdfFileVersion(header);
        if (document.version == 0)
          throw new InvalidOperationException(PSSR.InvalidPdf);

        // Read all trailers
        document.xrefTable.IsUnderConstruction = true;
        Parser parser = new Parser(document);
        document.trailer = parser.ReadTrailer();

        document.xrefTable.IsUnderConstruction = false;

        // Is document encrypted?
        PdfXRef xrefEncrypt = document.trailer.Elements[PdfTrailer.Keys.Encrypt] as PdfXRef;
        if (xrefEncrypt != null)
        {
          //xrefEncrypt.Value = parser.ReadObject(null, xrefEncrypt.ObjectID, false);
          PdfObject encrypt = parser.ReadObject(null, xrefEncrypt.ObjectID, false);

          encrypt.XRef = xrefEncrypt;
          xrefEncrypt.Value = encrypt;
          PdfStandardSecurityHandler securityHandler = document.SecurityHandler;
        TryAgain:
          int v = securityHandler.ValidatePassword(password);
          if (v == 0)
          {
            if (passwordProvider != null)
            {
              PdfPasswordProviderArgs args = new PdfPasswordProviderArgs();
              passwordProvider(args);
              if (args.Abort)
                return null;
              password = args.Password;
              goto TryAgain;
            }
            else
            {
              if (password == null)
                throw new PdfReaderException(PSSR.PasswordRequired);
              else
                throw new PdfReaderException(PSSR.InvalidPassword);
            }
          }
          else if (v == 1 && openmode == PdfDocumentOpenMode.Modify)
          {
            throw new PdfReaderException(PSSR.OwnerPasswordRequired);
          }
        }
        else
        {
          if (password != null)
          {
            // Password specified but document is not encrypted.
            // ignore
          }
        }

        PdfXRef[] xrefs = document.xrefTable.AllXRefs;
        int count = xrefs.Length;

        // Read all indirect objects
        for (int idx = 0; idx < count; idx++)
        {
          PdfXRef xref = xrefs[idx];
          if (xref.Value == null)
          {
            Debug.Assert(document.xrefTable.Contains(xref.ObjectID));
            PdfObject pdfObject = parser.ReadObject(null, xref.ObjectID, false);
            Debug.Assert(pdfObject.XRef == xref);
            pdfObject.XRef = xref;
            Debug.Assert(pdfObject.XRef.Value != null, "something got wrong");
          }
          else
          {
            Debug.Assert(document.xrefTable.Contains(xref.ObjectID));
            xref.GetType();
          }
          // Set maximum object number
          document.xrefTable.maxObjectNumber = Math.Max(document.xrefTable.maxObjectNumber, xref.ObjectNumber);
        }
        // Encrypt all objects
        if (xrefEncrypt != null)
        {
          document.SecurityHandler.EncryptDocument();
          // HACK
          ////document.trailer.Elements.Remove(PdfTrailer.Keys.Encrypt);
        }

        // Fix references of trailer values and then objects and xrefs are consistent.
        document.trailer.Finish();

#if DEBUG_
        // Some tests...
        PdfXRef[] reachables = document.xrefTable.TransitiveClosure(document.trailer);
        reachables.GetType();
        reachables = document.xrefTable.AllXRefs;
        document.xrefTable.CheckConsistence();
#endif

        if (openmode == PdfDocumentOpenMode.Modify)
        {
          // Create new or change existing document IDs
          if (document.Internals.SecondDocumentID == "")
            document.trailer.CreateNewDocumentIDs();
          else
            document.Internals.SecondDocumentID = PdfEncoders.RawEncoding.GetString(Guid.NewGuid().ToByteArray());

          // Change modification date
          document.Info.ModificationDate = DateTime.Now;

          // Remove all unreachable objects
          int removed = document.xrefTable.Compact();
          if (removed != 0)
            Debug.WriteLine("Number of deleted unreachable objects: " + removed.ToString());

          // Force flattening of page tree
          PdfPages pages = document.Pages;

          bool b = document.xrefTable.Contains(new PdfObjectID(1108));
          b.GetType();

          document.xrefTable.CheckConsistence();
          document.xrefTable.Renumber();
          document.xrefTable.CheckConsistence();
        }
      }
      finally
      {
        //if (filestream != null)
        //  filestream.Close();
      }
      return document;
    }

    /// <summary>
    /// Opens an existing PDF document.
    /// </summary>
    public static PdfDocument Open(Stream stream)
    {
      return PdfReader.Open(stream, PdfDocumentOpenMode.Modify);
    }
  }
}