//
// PDFSharp - A library for processing PDF
//
// Authors:
//   Stefan Lange (mailto:Stefan.Lange@pdfsharp.com)
//
// Copyright (c) 2005 empira Software GmbH, Cologne (Germany)
//
// http://www.pdfsharp.com
//
// Permission is hereby granted, free of charge, to any person obtaining
// a copy of this software and associated documentation files (the
// "Software"), to deal in the Software without restriction, including
// without limitation the rights to use, copy, modify, merge, publish,
// distribute, sublicense, and/or sell copies of the Software, and to
// permit persons to whom the Software is furnished to do so, subject to
// the following conditions:
// 
// The above copyright notice and this permission notice shall be
// included in all copies or substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
// EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
// MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
// NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
// LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
// OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
// WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

using System;
using System.Diagnostics;
using System.Collections;
using PdfSharp.Pdf;
using PdfSharp.Internal;
using PdfSharp.Drawing;


namespace PdfSharp.Pdf.Annotations
{
  /// <summary>
  /// Represents the base class for all annotations.
  /// </summary>
  public abstract class PdfAnnotation : PdfDictionary
  {
    public PdfAnnotation()
    {
      Elements.Type = "/Annot";
    }

    public PdfAnnotation(PdfDocument document) : base(document)
    {
      Elements.Type = "/Annot";
    }

    public PdfRectangle Rectangle
    {
      get {return Elements.GetRectangle(Keys.Rect, true);}
      set {Elements.SetRectangle(Keys.Rect, value);}
    }

    /// <summary>
    /// Predefined keys of this dictionary.
    /// </summary>
    internal class Keys : KeysBase
    {
      /// <summary>
      /// (Optional) The type of PDF object that this dictionary describes; if present,
      /// must be Annot for an annotation dictionary.
      /// </summary>
      [KeyInfo(KeyType.Name | KeyType.Optional, FixedValue = "Annot")]
      public const string Type = "/Type";

      /// <summary>
      /// (Required) The type of annotation that this dictionary describes.
      /// </summary>
      [KeyInfo(KeyType.Name | KeyType.Required)]
      public const string Subtype = "/Subtype";

      /// <summary>
      /// (Required) The annotation rectangle, defining the location of the annotation
      /// on the page in default user space units.
      /// </summary>
      [KeyInfo(KeyType.Rectangle | KeyType.Required)]
      public const string Rect = "/Rect";

      // Contents
      // P
      // NM
      // M

      /// <summary>
      /// (Optional; PDF 1.1) A set of flags specifying various characteristics of the annotation.
      /// Default value: 0.
      /// </summary>
      [KeyInfo("1.1", KeyType.Integer | KeyType.Optional)]
      public const string F = "/F";

      /// <summary>
      /// (Optional; PDF 1.2) A border style dictionary specifying the characteristics of
      /// the annotations border.
      /// </summary>
      [KeyInfo("1.2", KeyType.Dictionary | KeyType.Optional)]
      public const string BS = "/BS";

      // AP
      // AS

      /// <summary>
      /// (Optional) An array specifying the characteristics of the annotations border.
      /// The border is specified as a rounded rectangle.
      /// In PDF 1.0, the array consists of three numbers defining the horizontal corner 
      /// radius, vertical corner radius, and border width, all in default user space units.
      /// If the corner radii are 0, the border has square (not rounded) corners; if the border 
      /// width is 0, no border is drawn.
      /// In PDF 1.1, the array may have a fourth element, an optional dash array defining a 
      /// pattern of dashes and gaps to be used in drawing the border. The dash array is 
      /// specified in the same format as in the line dash pattern parameter of the graphics state.
      /// For example, a Border value of [0 0 1 [3 2]] specifies a border 1 unit wide, with
      /// square corners, drawn with 3-unit dashes alternating with 2-unit gaps. Note that no
      /// dash phase is specified; the phase is assumed to be 0.
      /// Note: In PDF 1.2 or later, this entry may be ignored in favor of the BS entry.
      /// </summary>
      [KeyInfo(KeyType.Array| KeyType.Optional)]
      public const string Border = "/Border";

      /// <summary>
      /// (Optional; PDF 1.1) An array of three numbers in the range 0.0 to 1.0, representing
      /// the components of a color in the DeviceRGB color space. This color is used for the
      /// following purposes:
      ///  The background of the annotations icon when closed
      ///  The title bar of the annotations pop-up window
      ///  The border of a link annotation
      /// </summary>
      [KeyInfo("1.1", KeyType.Array | KeyType.Optional)]
      public const string C = "/C";

      /// <summary>
      /// (Optional; PDF 1.1) An action to be performed when the annotation is activated.
      /// Note: This entry is not permitted in link annotations if a Dest entry is present.
      /// Also note that the A entry in movie annotations has a different meaning.
      /// </summary>
      [KeyInfo("1.1", KeyType.Dictionary | KeyType.Optional)]
      public const string A = "/A";

      // AA
      // StructParent
      // OC
    }
  }
}
