//
// PDFSharp - A library for processing PDF
//
// Authors:
//   Stefan Lange (mailto:Stefan.Lange@pdfsharp.com)
//
// Copyright (c) 2005 empira Software GmbH, Cologne (Germany)
//
// http://www.pdfsharp.com
//
// Permission is hereby granted, free of charge, to any person obtaining
// a copy of this software and associated documentation files (the
// "Software"), to deal in the Software without restriction, including
// without limitation the rights to use, copy, modify, merge, publish,
// distribute, sublicense, and/or sell copies of the Software, and to
// permit persons to whom the Software is furnished to do so, subject to
// the following conditions:
// 
// The above copyright notice and this permission notice shall be
// included in all copies or substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
// EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
// MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
// NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
// LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
// OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
// WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

using System;
using System.Diagnostics;
using System.Collections;
using System.Globalization;
using System.Text;
using System.IO;
using PdfSharp.Drawing;
using PdfSharp.Internal;
using PdfSharp.Fonts.TrueType;

namespace PdfSharp.Pdf.Advanced
{
  /// <summary>
  /// Contains all external PDF files from which PdfFormXObjects are imported into the current document.
  /// </summary>
  internal sealed class PdfFormXObjectTable : PdfResourceTable
  {
    // The name PdfFormXObjectTable is technically not correct, because in contrast to PdfFontTable
    // or PdfImageTable this class holds no PdfFormXObject objects. Actually it holds instances of
    // the class ImportedObjectTable, one for each external document. The PdfFormXObject instances
    // are not cached, because the hold a transformation matrix that make them unique. If the user
    // wants to use a particual page of a PdfFormXObject more than once, he must reuse the object
    // before he changes the PageNumber or the transformation matrix. In other words this class
    // caches the indirect objects of an external form, not the form itself.

    /// <summary>
    /// Initializes a new instance of this class, which is a singleton for each document.
    /// </summary>
    public PdfFormXObjectTable(PdfDocument document) : base(document)
    {
    }

    /// <summary>
    /// Gets a PdfFormXObject from an XPdfForm. Because the returned objects must be unique, always
    /// a new instance of PdfFormXObject is created if none exists for the specifed form. 
    /// </summary>
    public PdfFormXObject GetForm(XPdfForm form)
    {
      // If the form already has a PdfFormXObject, return it.
      if (form.pdfForm != null)
        return form.pdfForm;

      // Is the external PDF file from which is imported already known for the current document?
      FormSelector selector = new FormSelector(form);
      ImportedObjectTable importDocument = this.forms[selector] as ImportedObjectTable;
      if (importDocument == null)
      {
        // No: Get the external document from the form and create ImportedObjectTable.
        PdfDocument doc = form.ExternalDocument;
        importDocument = new ImportedObjectTable(doc);
        this.forms[selector] = importDocument;
      }
      //Debug.Assert(form.importDocument == null); // TODO: realy true in any case???
      form.importDocument = importDocument;
      form.pdfForm = new PdfFormXObject(this.document, form);
      return form.pdfForm;
    }

    /// <summary>
    /// Map from FormSelector to ImportedObjectTable.
    /// </summary>
    Hashtable forms = new Hashtable();

    /// <summary>
    /// A collection of information that uniquely idendifies a particular ImportedObjectTable.
    /// </summary>
    public class FormSelector
    {
      /// <summary>
      /// Initializes a new instance of FormSelector from an XPdfForm.
      /// </summary>
      public FormSelector(XPdfForm form)
      {
        // HACK: just use full path to identify
        this.path = form.path.ToLower(CultureInfo.InvariantCulture);
      }

      public string Path
      {
        get {return this.path;}
        set {this.path = value;}
      }
      string path;

      public override bool Equals(object obj)
      {
        FormSelector selector = obj as FormSelector;
        if (obj == null)
          return false;
        return this.path == selector.path;;
      }

      public override int GetHashCode()
      {
        return this.path.GetHashCode();
      }
    }
  }
}
