//
// PDFSharp - A library for processing PDF
//
// Authors:
//   Stefan Lange (mailto:Stefan.Lange@pdfsharp.com)
//
// Copyright (c) 2005 empira Software GmbH, Cologne (Germany)
//
// http://www.pdfsharp.com
//
// Permission is hereby granted, free of charge, to any person obtaining
// a copy of this software and associated documentation files (the
// "Software"), to deal in the Software without restriction, including
// without limitation the rights to use, copy, modify, merge, publish,
// distribute, sublicense, and/or sell copies of the Software, and to
// permit persons to whom the Software is furnished to do so, subject to
// the following conditions:
// 
// The above copyright notice and this permission notice shall be
// included in all copies or substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
// EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
// MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
// NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
// LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
// OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
// WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

using System;
using System.Diagnostics;
using System.Collections;
using System.Globalization;
using System.Text;
using System.IO;
using PdfSharp.Drawing;
using PdfSharp.Internal;
using PdfSharp.Fonts.TrueType;

namespace PdfSharp.Pdf.Advanced
{
  /// <summary>
  /// Contains all used fonts of a document.
  /// </summary>
  internal sealed class PdfFontTable : PdfResourceTable
  {
    /// <summary>
    /// Initializes a new instance of this class, which is a singleton for each document.
    /// </summary>
    public PdfFontTable(PdfDocument document) : base(document)
    {
    }

    /// <summary>
    /// Gets a PdfFont from an XFont. If no PdfFont already exists, a new one is created.
    /// </summary>
    public PdfFont GetFont(XFont font)
    {
      string fontName = font.Name;

      PdfFontTable.FontSelector selector = font.selector;
      if (selector == null)
      {
        selector = new FontSelector(font);
        font.selector = selector;
      }
      PdfFont pdfFont = this.fonts[selector] as PdfFont;
      if (pdfFont == null)
      {
        pdfFont = new PdfFont(this.document, font);
        //pdfFont.Document = this.document;
        Debug.Assert(pdfFont.Document == this.document);
        this.fonts[selector] = pdfFont;
        if (this.document.EarlyWrite)
        {
          //pdfFont.Close(); delete 
          //pdfFont.AssignObjID(ref this.document.ObjectID); // BUG: just test code!!!!
          //pdfFont.WriteObject(null);
        }
      }
      return pdfFont;

#if false
      goto TrueTypeFont;
      switch (font.Name)
      {
        case "Times":
        case "Times New Roman":
          std = 0;
          break;

        case "Helvetica":
          std = 1;
          break;

        case "Courier":
          std = 2;
          break;

        case "Symbol":
          std = 3;
          break;

        case "ZapfDingbats":
          std = 4;
          break;
      }
      if (std != -1)
      {
        int idx = (int)font.Style & 0x3;
        string name = pdfStandardFonts[std][idx];
        PdfFont pdfFont = GetFont(name);
        if (pdfFont == null)
        {
          pdfFont = new PdfFont();
          pdfFont.SubType = "/Type1";
          pdfFont.BaseFont = name;
          pdfFont.DefaultName = string.Format("F{0}", PdfFontTable.fontNumber++);
        }
        return pdfFont;
      }
      else
      {
      TrueTypeFont:
        // TrueType font
        PdfFont pdfFont = new PdfFont();
        pdfFont.SubType = "/TrueType";
        pdfFont.FirstChar = 0;
        pdfFont.LastChar = 255;
        pdfFont.BaseFont = font.Name;
        pdfFont.DefaultName = string.Format("F{0}", PdfFontTable.fontNumber++);
      } 
#endif
      // TrueType font
//      PdfFont pdfFont = new PdfFont();
//      pdfFont.descriptor = new PdfFontDescriptor((TrueTypeDescriptor)FontDescriptorStock.Global.CreateDescriptor(font));
//      pdfFont.SubType = "/TrueType";
//      pdfFont.FirstChar = 0;
//      pdfFont.LastChar = 255;
//      pdfFont.BaseFont = font.Name;
//      pdfFont.BaseFont = pdfFont.BaseFont.Replace(" ", "");
//      switch (font.Style & (XFontStyle.Bold | XFontStyle.Italic))
//      {
//        case XFontStyle.Bold:
//          pdfFont.BaseFont += ",Bold";
//          break;
//
//        case XFontStyle.Italic:
//          pdfFont.BaseFont += ",Italic";
//          break;
//        
//        case XFontStyle.Bold | XFontStyle.Italic:
//          pdfFont.BaseFont += ",BoldItalic";
//          break;
//      }
//      pdfFont.descriptor.FontName = pdfFont.BaseFont;
//      pdfFont.DefaultName = string.Format("F{0}", PdfFontTable.fontNumber++);
    }

    //string[][] pdfStandardFonts =
    //{
    //  new string[]{"Times-Roman", "Times-Bold", "Times-Italic", "Times-BoldItalic"},
    //  new string[]{"Helvetica", "Helvetica-Bold", "Helvetica-Oblique", "Helvetica-BoldOblique"},
    //  new string[]{"Courier", "Courier-Bold", "Courier-Oblique", "Courier-BoldOblique"},
    //  new string[]{"Symbol", "Symbol", "Symbol", "Symbol"},
    //  new string[]{"ZapfDingbats", "ZapfDingbats", "ZapfDingbats", "ZapfDingbats"},
    //};

    /// <summary>
    /// Map from PdfFontSelector to PdfFont.
    /// </summary>
    Hashtable fonts = new Hashtable();

    /// <summary>
    /// A collection of information that uniquely idendifies a particular PDF font.
    /// ... more docu...
    /// Two PDF fonts are equal if and only if their font selector objects are equal.
    /// </summary>
    public class FontSelector
    {
      enum FontType
      {
        TrueType = 1,
      }

      /// <summary>
      /// Initializes a new instance of PdfFontSelector from an XFont.
      /// </summary>
      public FontSelector(XFont font)
      {
        this.name = font.Name;
        this.style = font.Style;
        this.fontType = FontType.TrueType;
      }

      public FontSelector(XFontFamily family, XFontStyle style)
      {
        throw new NotImplementedException("PdfFontSelector(XFontFamily family, XFontStyle style)");
      }

      /// <summary>
      /// Gets the (generated) resource name of the font. In our own PDF files equal fonts share the
      /// same resource name in all contents streams.
      /// </summary>
      public string Name // TODO: How to solve name clashs with imported contents streams?
      {
        get {return this.name;}
      }
      string name;

      public XFontStyle Style
      {
        get {return this.style;}
      }
      XFontStyle style;

      public bool IsTrueType
      {
        get {return this.fontType == FontType.TrueType;}
      }
      FontType fontType;

      public static bool operator == (FontSelector selector1, FontSelector selector2)
      {
        if (!Object.ReferenceEquals(selector1, null))
          selector1.Equals(selector2);
        return Object.ReferenceEquals(selector2, null);
      }

      public static bool operator != (FontSelector selector1, FontSelector selector2)
      {
        return !(selector1 == selector2);
      }

      public override bool Equals(object obj)
      {
        FontSelector selector = obj as FontSelector;
        if (obj != null && this.name == selector.name)
          return this.style == selector.style;
        return false;
      }

      public override int GetHashCode()
      {
        return this.name.GetHashCode() ^ this.style.GetHashCode();
      }

      /// <summary>
      /// Returns a string for diagnostic purposes only.
      /// </summary>
      public override string ToString()
      {
        string variation = "";
        switch (this.style)
        {
          case XFontStyle.Regular:
            variation = "(Regular)";
            break;

          case XFontStyle.Bold:
            variation = "(Bold)";
            break;

          case XFontStyle.Italic:
            variation = "(Italic)";
            break;

          case XFontStyle.Bold | XFontStyle.Italic:
            variation = "(BoldItalic)";
            break;
        }
        return this.name + variation;
      }
    }
  }
}
