//
// PDFSharp - A library for processing PDF
//
// Authors:
//   Stefan Lange (mailto:Stefan.Lange@pdfsharp.com)
//
// Copyright (c) 2005 empira Software GmbH, Cologne (Germany)
//
// http://www.pdfsharp.com
//
// Permission is hereby granted, free of charge, to any person obtaining
// a copy of this software and associated documentation files (the
// "Software"), to deal in the Software without restriction, including
// without limitation the rights to use, copy, modify, merge, publish,
// distribute, sublicense, and/or sell copies of the Software, and to
// permit persons to whom the Software is furnished to do so, subject to
// the following conditions:
// 
// The above copyright notice and this permission notice shall be
// included in all copies or substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
// EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
// MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
// NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
// LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
// OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
// WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

using System;
using System.Diagnostics;
using System.Collections;
using System.Text;
using System.IO;
using PdfSharp.Pdf.Advanced;
using PdfSharp.Drawing;
using PdfSharp.Fonts.TrueType;
using PdfSharp.Pdf.Filters;
using PdfSharp.Internal;

namespace PdfSharp.Pdf.Advanced
{
  /// <summary>
  /// Represents a font.
  /// </summary>
  internal class PdfFont : PdfDictionary
  {
    public PdfFont()
    {
    }
    
    /// <summary>
    /// Initializes a new instance of PdfFont from an XFont.
    /// </summary>
    public PdfFont(PdfDocument document, XFont font) : base(document)
    {
      // TrueType with WinAnsiEncoding only
      TrueTypeDescriptor ttDescriptor = (TrueTypeDescriptor)FontDescriptorStock.Global.CreateDescriptor(font);
      this.fontDescriptor = new PdfFontDescriptor(document, ttDescriptor);
      this.fontOptions = font.PdfOptions;
      Debug.Assert(this.fontOptions != null);

      Elements.Type = "/Font";
      Elements.Subtype = "/TrueType";
      FirstChar = 0;
      LastChar  = 255;

      if (this.fontOptions.BaseFont != "")
      {
        BaseFont = this.fontOptions.BaseFont;
      }
      else
      {
        BaseFont = font.Name.Replace(" ", "");
        switch (font.Style & (XFontStyle.Bold | XFontStyle.Italic))
        {
          case XFontStyle.Bold:
            this.BaseFont += ",Bold";
            break;

          case XFontStyle.Italic:
            this.BaseFont += ",Italic";
            break;
        
          case XFontStyle.Bold | XFontStyle.Italic:
            this.BaseFont += ",BoldItalic";
            break;
        }
      }
      this.fontDescriptor.FontName = this.BaseFont;

      if (this.fontOptions.Unicode)
      {
        // TODO 
        throw new NotImplementedException("Unicode encoding");
      }
      else
      {
        if (!IsSymbolFont)
          Encoding = "/WinAnsiEncoding";
      }

      // Embed the whole font file
      if (this.fontOptions.Embed)
      {
        if (this.fontOptions.FontFile != "")
        {
          byte[] fontProgram = ttDescriptor.image.Bytes;
          PdfDictionary fontStream = new PdfDictionary(this.Document);
          this.Document.xrefTable.Add(fontStream);
          this.fontDescriptor.Elements[PdfFontDescriptor.Keys.FontFile2] = fontStream.XRef;

          fontStream.Elements["/Length1"] = new PdfInteger(fontProgram.Length);
          if (!this.Document.Options.NoCompression)
          {
            fontProgram = Filtering.FlateDecode.Encode(fontProgram);
            fontStream.Elements["/Filter"] = new PdfName("/FlateDecode");
          }
          fontStream.Elements["/Length"] = new PdfInteger(fontProgram.Length);
          fontStream.CreateStream(fontProgram);
        }
        else
        {
#if true
          throw new NotImplementedException("font file specified");
#else
          FileStream stream = new FileStream("WAL____I.AFM", FileAccess.Read);
          int length = stream.Length;
          byte[] fontProgram = new byte[length];
          PdfDictionary fontStream = new PdfDictionary(this.Document);
          this.Document.xrefTable.Add(fontStream);
          this.fontDescriptor.Elements[PdfFontDescriptor.Keys.FontFile] = fontStream.XRef;

          fontStream.Elements["/Length1"] = new PdfInteger(fontProgram.Length);
          if (!this.Document.Options.NoCompression)
          {
            fontProgram = Filtering.FlateDecode.Encode(fontProgram);
            fontStream.Elements["/Filter"] = new PdfName("/FlateDecode");
          }
          fontStream.Elements["/Length"] = new PdfInteger(fontProgram.Length);
          fontStream.CreateStream(fontProgram);
#endif
        }
      }

      PdfArray width = Widths;
      for (int idx = 0; idx < 256; idx++)
        width.Elements.Add(new PdfInteger(this.fontDescriptor.descriptor.widths[idx]));

      this.Document.xrefTable.Add(fontDescriptor);
      this.Elements[Keys.FontDescriptor] = fontDescriptor.XRef;
    }

    internal override PdfDocument Document
    {
      get {return base.Document;}
//      set 
//      {
//        base.Document = value;
//        if (this.fontDescriptor != null)
//          this.fontDescriptor.Document = value;
//      }
    }

    public PdfFontDescriptor FontDescriptor
    {
      get 
      {
        Debug.Assert(this.fontDescriptor != null);
        //if (this.fontDescriptor2 == null)
        //  this.fontDescriptor2 = (PdfFontDescriptor)Elements.GetValue(Keys.FontDescriptor, VCF.CreateIndirect);
        return this.fontDescriptor;
      }
    }
    PdfFontDescriptor fontDescriptor;

    PdfFontOptions FontOptions
    {
      get {return this.fontOptions;}
    }
    PdfFontOptions fontOptions;

    public bool IsSymbolFont
    {
      get {return this.fontDescriptor.IsSymbolFont;}
    }

    public string BaseFont
    {
      get { return Elements.GetName(Keys.BaseFont); }
      set { Elements.SetName(Keys.BaseFont, value); }
    }

    public int FirstChar
    {
      get { return Elements.GetInteger(Keys.FirstChar); }
      set { Elements.SetInteger(Keys.FirstChar, value); }
    }

    public int LastChar
    {
      get { return Elements.GetInteger(Keys.LastChar); }
      set { Elements.SetInteger(Keys.LastChar, value); }
    }

    public PdfArray Widths
    {
      get { return (PdfArray)Elements.GetValue(Keys.Widths, VCF.Create); }
    }

    public string Encoding
    {
      get { return Elements.GetName(Keys.Encoding); }
      set { Elements.SetName(Keys.Encoding, value); }
    }

    /// <summary>
    /// Predefined keys of this dictionary.
    /// </summary>
    internal sealed class Keys : KeysBase
    {
      /// <summary>
      /// (Required) The type of PDF object that this dictionary describes;
      /// must be Font for a font dictionary.
      /// </summary>
      [KeyInfo(KeyType.Name | KeyType.Required, FixedValue = "Font")]
      public const string Type = "/Type";

      /// <summary>
      /// (Required) The type of font; must be Type1 for a Type 1 font.
      /// </summary>
      [KeyInfo(KeyType.Name | KeyType.Required)]
      public const string Subtype = "/Subtype";

      /// <summary>
      /// (Required in PDF 1.0; optional otherwise) The name by which this font is 
      /// referenced in the Font subdictionary of the current resource dictionary.
      /// </summary>
      [KeyInfo(KeyType.Name | KeyType.Optional)]
      public const string Name = "/Name";

      /// <summary>
      /// (Required) The PostScript name of the font. For Type 1 fonts, this is usually
      /// the value of the FontName entry in the font program; for more information.
      /// The Post-Script name of the font can be used to find the fonts definition in 
      /// the consumer application or its environment. It is also the name that is used when
      /// printing to a PostScript output device.
      /// </summary>
      [KeyInfo(KeyType.Name | KeyType.Required)]
      public const string BaseFont = "/BaseFont";

      /// <summary>
      /// (Required except for the standard 14 fonts) The first character code defined 
      /// in the fonts Widths array.
      /// </summary>
      [KeyInfo(KeyType.Integer)]
      public const string FirstChar = "/FirstChar";

      /// <summary>
      /// (Required except for the standard 14 fonts) The last character code defined
      /// in the fonts Widths array.
      /// </summary>
      [KeyInfo(KeyType.Integer)]
      public const string LastChar = "/LastChar";

      /// <summary>
      /// (Required except for the standard 14 fonts; indirect reference preferred)
      /// An array of (LastChar - FirstChar + 1) widths, each element being the glyph width
      /// for the character code that equals FirstChar plus the array index. For character
      /// codes outside the range FirstChar to LastChar, the value of MissingWidth from the 
      /// FontDescriptor entry for this font is used. The glyph widths are measured in units 
      /// in which 1000 units corresponds to 1 unit in text space. These widths must be 
      /// consistent with the actual widths given in the font program. 
      /// </summary>
      [KeyInfo(KeyType.Array, typeof(PdfArray))]
      public const string Widths = "/Widths";

      /// <summary>
      /// (Required except for the standard 14 fonts; must be an indirect reference)
      /// A font descriptor describing the fonts metrics other than its glyph widths.
      /// Note: For the standard 14 fonts, the entries FirstChar, LastChar, Widths, and 
      /// FontDescriptor must either all be present or all be absent. Ordinarily, they are
      /// absent; specifying them enables a standard font to be overridden.
      /// </summary>
      [KeyInfo(KeyType.Dictionary | KeyType.MustBeIndirect, typeof(PdfFontDescriptor))]
      public const string FontDescriptor = "/FontDescriptor";

      /// <summary>
      /// (Optional) A specification of the fonts character encoding if different from its
      /// built-in encoding. The value of Encoding is either the name of a predefined
      /// encoding (MacRomanEncoding, MacExpertEncoding, or WinAnsiEncoding, as described in 
      /// Appendix D) or an encoding dictionary that specifies differences from the fonts
      /// built-in encoding or from a specified predefined encoding.
      /// </summary>
      [KeyInfo(KeyType.Name | KeyType.Dictionary)]
      public const string Encoding = "/Encoding";

      /// <summary>
      /// (Optional; PDF 1.2) A stream containing a CMap file that maps character
      /// codes to Unicode values.
      /// </summary>
      [KeyInfo(KeyType.Stream | KeyType.Optional)]
      public const string ToUnicode = "/ToUnicode";

      /// <summary>
      /// Gets the KeysMeta for these keys.
      /// </summary>
      public static KeysMeta Meta
      {
        get
        {
          if (Keys.meta == null)
            Keys.meta = CreateMeta(typeof(Keys));
          return Keys.meta;
        }
      }
      static KeysMeta meta;
    }

    /// <summary>
    /// Gets the KeysMeta of this dictionary type.
    /// </summary>
    internal override KeysMeta Meta
    {
      get {return Keys.Meta;}
    }
  }
}
