//
// PDFSharp - A library for processing PDF
//
// Authors:
//   Stefan Lange (mailto:Stefan.Lange@pdfsharp.com)
//
// Copyright (c) 2005 empira Software GmbH, Cologne (Germany)
//
// http://www.pdfsharp.com
//
// Permission is hereby granted, free of charge, to any person obtaining
// a copy of this software and associated documentation files (the
// "Software"), to deal in the Software without restriction, including
// without limitation the rights to use, copy, modify, merge, publish,
// distribute, sublicense, and/or sell copies of the Software, and to
// permit persons to whom the Software is furnished to do so, subject to
// the following conditions:
// 
// The above copyright notice and this permission notice shall be
// included in all copies or substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
// EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
// MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
// NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
// LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
// OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
// WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

using System;
using System.Collections;
using System.Text;
using System.IO;
using PdfSharp.Drawing;
using PdfSharp.Drawing.Pdf;
using PdfSharp.Pdf;
using PdfSharp.Pdf.Internal;
using PdfSharp.Pdf.Filters;
using PdfSharp.Pdf.IO;

namespace PdfSharp.Pdf.Advanced
{
  /// <summary>
  /// Represents the content of a page. PDFsharp supports only one content stream per page.
  /// If an imported page has an array of content streams, the streams are concatenated to
  /// one single stream.
  /// </summary>
  public sealed class PdfContent : PdfDictionary
  {
    public PdfContent(PdfDocument document) : base(document)
    {
    }

    internal PdfContent(PdfPage page) : base(page != null ? page.Document : null)
    {
      //this.pageContent = new PageContent(page);
    }

    internal PdfContent(PdfDictionary dict) : base(dict)
    {
    }

    // HACK
    public void Deflate()
    {
      if (Stream != null && Stream.Value != null)
      {
        PdfItem item = Elements["/Filter"];
        if (item is PdfString)
        {
          if (Elements.GetString("/Filter") == "/FlateDecode")
          {
            Stream.Value = Filtering.FlateDecode.Decode(Stream.Value);
            Elements.Remove("/Filter");
            Elements.SetInteger("/Length", Stream.Length);
          }    
        }
      }
    }

    internal override void WriteObject(PdfWriter writer)
    {
      if (Stream == null || Stream.Value == null)
        Stream = new PdfStream(PdfEncoders.RawEncoding.GetBytes(this.pdfRenderer.GetContent()), this);
        //CreateStream(PdfEncoders.RawEncoding.GetBytes(this.pdfRenderer.GetContent());
      else
      {
        string content = "";
        PdfItem item = Elements["/Filter"];
        if (item is PdfName)
        {
          if (Elements.GetString("/Filter") == "/FlateDecode")
          {
            content = Filtering.FlateDecode.DecodeToString(Stream.Value);
            Elements.Remove("/Filter");
          }
          else
            throw new NotImplementedException("Unknown filter");
        }
        else
          content = PdfEncoders.RawEncoding.GetString(Stream.Value);

        if (this.pdfRenderer != null)
        {
          switch (this.pdfRenderer.PageOptions)
          {
            case XGraphicsPdfPageOptions.Replace:
              content = this.pdfRenderer.GetContent();
              break;

            case XGraphicsPdfPageOptions.Prepend:
              content = this.pdfRenderer.GetContent() + '\n' + content;
              break;

            case XGraphicsPdfPageOptions.Append:
              // HACK
              if (!content.StartsWith("q\n") || !content.EndsWith("Q\n"))
                content = "q\n" + content + "\nQ\n";
              else
                content += '\n';
              content += this.pdfRenderer.GetContent();
              break;
          }
        }
        Stream = new PdfStream(PdfEncoders.RawEncoding.GetBytes(content), this);
      }
      if (this.Document.Options.CompressContentStreams)
      {
        Stream.Value = Filtering.FlateDecode.Encode(Stream.Value);
        Elements["/Filter"] = new PdfName("/FlateDecode");
      }
      int length = Stream.Length;
      Elements.SetInteger("/Length", length);

      base.WriteObject(writer);
    }

    internal XGraphicsPdfRenderer pdfRenderer;

    /// <summary>
    /// Predefined keys of this dictionary.
    /// </summary>
    internal sealed class Keys : PdfDictionary.PdfStream.Keys
    {
      /// <summary>
      /// Gets the KeysMeta for these keys.
      /// </summary>
      public static KeysMeta Meta
      {
        get
        {
          if (Keys.meta == null)
            Keys.meta = CreateMeta(typeof(Keys));
          return Keys.meta;
        }
      }
      static KeysMeta meta;
    }

    /// <summary>
    /// Gets the KeysMeta of this dictionary type.
    /// </summary>
    internal override KeysMeta Meta
    {
      get {return Keys.Meta;}
    }
  }
}
