//
// PDFSharp - A library for processing PDF
//
// Authors:
//   Stefan Lange (mailto:Stefan.Lange@pdfsharp.com)
//
// Copyright (c) 2005 empira Software GmbH, Cologne (Germany)
//
// http://www.pdfsharp.com
//
// Permission is hereby granted, free of charge, to any person obtaining
// a copy of this software and associated documentation files (the
// "Software"), to deal in the Software without restriction, including
// without limitation the rights to use, copy, modify, merge, publish,
// distribute, sublicense, and/or sell copies of the Software, and to
// permit persons to whom the Software is furnished to do so, subject to
// the following conditions:
// 
// The above copyright notice and this permission notice shall be
// included in all copies or substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
// EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
// MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
// NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
// LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
// OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
// WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

using System;
using System.ComponentModel;
using System.Drawing;

namespace PdfSharp.Drawing
{
  /// <summary>
  /// Represents a pair of floating-point numbers, typically the width and height of a
  /// graphical object.
  /// </summary>
  public struct XSize
  {
    public XSize(XSize size)
    {
      this.width = size.width;
      this.height = size.height;
    }

    public XSize(XPoint pt)
    {
      this.width = pt.X;
      this.height = pt.Y;
    }

    public XSize(double width, double height)
    {
      this.width = width;
      this.height = height;
    }

    public override int GetHashCode()
    {
      return this.width.GetHashCode() ^ this.height.GetHashCode();
    }

    public override bool Equals(object obj)
    {
      if (obj is XSize)
      {
        XSize size = (XSize)obj;
        return size.width == this.width && size.height == this.height;
      }
      return false;
    }

    public static XSize FromSize(Size size)
    {
      return new XSize(size.Width, size.Height);
    }

    public static XSize FromSizeF(SizeF size)
    {
      return new XSize(size.Width, size.Height);
    }

    public override string ToString()
    {
      return string.Format("{{Width={0}, Height={1}}}", this.width, this.height);
    }

    /// <summary>
    /// Converts this XSize to a PointF.
    /// </summary>
    public PointF ToPointF()
    {
      return new PointF((float)this.width, (float)this.height);
    }

    /// <summary>
    /// Converts this XSize to an XPoint.
    /// </summary>
    public XPoint ToXPoint()
    {
      return new XPoint(this.width, this.height);
    }

    /// <summary>
    /// Converts this XSize to a SizeF.
    /// </summary>
    public SizeF ToSizeF()
    {
      return new SizeF((float)this.width, (float)this.height);
    }

    [Browsable(false)]
    public bool IsEmpty 
    { 
      get {return this.width == 0 && this.height == 0;}
    }

    public double Width 
    { 
      get {return this.width;}
      set {this.width = value;}
    }

    public double Height
    { 
      get {return this.height;}
      set {this.height = value;}
    }

    public static XSize operator +(XSize size1, XSize size2)
    {
      return new XSize(size1.width + size2.width, size1.height + size2.height);
    }

    public static XSize operator -(XSize size1, XSize size2)
    {
      return new XSize(size1.width - size2.width, size1.height - size2.height);
    }

    public static bool operator ==(XSize left, XSize right)
    {
      return left.width == right.width && left.height == right.height;
    }

    public static bool operator !=(XSize left, XSize right)
    {
      return !(left == right);
    }

    public static explicit operator XPoint(XSize size)
    {
      return new XPoint(size.width, size.height);
    }

    public static readonly XSize Empty = new XSize();

    internal double width;
    internal double height;
  }
}
