//
// PDFSharp - A library for processing PDF
//
// Authors:
//   Stefan Lange (mailto:Stefan.Lange@pdfsharp.com)
//
// Copyright (c) 2005 empira Software GmbH, Cologne (Germany)
//
// http://www.pdfsharp.com
//
// Permission is hereby granted, free of charge, to any person obtaining
// a copy of this software and associated documentation files (the
// "Software"), to deal in the Software without restriction, including
// without limitation the rights to use, copy, modify, merge, publish,
// distribute, sublicense, and/or sell copies of the Software, and to
// permit persons to whom the Software is furnished to do so, subject to
// the following conditions:
// 
// The above copyright notice and this permission notice shall be
// included in all copies or substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
// EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
// MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
// NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
// LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
// OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
// WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

using System;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.IO;
using PdfSharp.Internal;

namespace PdfSharp.Drawing
{
  /// <summary>
  /// Defines an object used to draw lines and curves.
  /// </summary>
  public sealed class XPen
  {
    public XPen(XColor color)
      : this(color, 1, false)
    { }

    public XPen(XColor color, double width) : this(color, width, false)
    { }

    internal XPen(XColor color, double width, bool immutable)
    {
      this.color      = color;
      this.width      = width;
      this.lineJoin   = XLineJoin.Miter;
      this.lineCap    = XLineCap.Flat;
      this.dashStyle  = XDashStyle.Solid;
      this.dashOffset = 0f;
      this.immutable  = immutable;
    }

    public XPen(XPen pen)
    {
      this.color       = pen.color;
      this.width       = pen.width;
      this.lineJoin    = pen.lineJoin;
      this.lineCap     = pen.lineCap;
      this.dashStyle   = pen.dashStyle;
      this.dashOffset  = pen.dashOffset;
      this.dashPattern = pen.dashPattern;
      if (this.dashPattern != null)
        this.dashPattern = (double[])this.dashPattern.Clone();
    }

    public XPen Clone()
    {
      return new XPen(this);
    }

    public XColor Color
    {
      get { return this.color; }
      set 
      { 
        if (this.immutable)
          throw new ArgumentException(PSSR.CannotChangeImmutableObject("XPen"));
        this.dirty = this.dirty || this.color != value;
        this.color = value; 
      }
    }
    XColor color;

    public double Width
    {
      get { return this.width; }
      set 
      { 
        if (this.immutable)
          throw new ArgumentException(PSSR.CannotChangeImmutableObject("XPen"));
        this.dirty = this.dirty || this.width != value; 
        this.width = value;
      }
    }
    double width;

    public XLineJoin LineJoin
    {
      get { return this.lineJoin; }
      set 
      { 
        if (this.immutable)
          throw new ArgumentException(PSSR.CannotChangeImmutableObject("XPen"));
        this.dirty = this.dirty || this.lineJoin != value; 
        this.lineJoin = value; 
      }
    }
    XLineJoin lineJoin;

    public XLineCap LineCap
    {
      get { return this.lineCap; }
      set 
      { 
        if (this.immutable)
          throw new ArgumentException(PSSR.CannotChangeImmutableObject("XPen"));
        this.dirty = this.dirty || this.lineCap != value; 
        this.lineCap = value;
      }
    }
    XLineCap lineCap;

    public double MiterLimit
    {
      get { return this.miterLimit; }
      set 
      { 
        if (this.immutable)
          throw new ArgumentException(PSSR.CannotChangeImmutableObject("XPen"));
        this.dirty = this.dirty || this.miterLimit != value;
        this.miterLimit = value;
      }
    }
    double miterLimit;

    public XDashStyle DashStyle
    {
      get { return this.dashStyle; }
      set 
      { 
        if (this.immutable)
          throw new ArgumentException(PSSR.CannotChangeImmutableObject("XPen"));
        this.dirty = this.dirty || this.dashStyle != value; 
        this.dashStyle = value;
      }
    }
    XDashStyle dashStyle;

    public double DashOffset
    {
      get { return this.dashOffset; }
      set 
      { 
        if (this.immutable)
          throw new ArgumentException(PSSR.CannotChangeImmutableObject("XPen"));
        this.dirty = this.dirty || this.dashOffset != value;
        this.dashOffset = value;
      }
    }
    double dashOffset;

    public double[] DashPattern
    {
      get { return this.dashPattern; }
      set 
      { 
        if (this.immutable)
          throw new ArgumentException(PSSR.CannotChangeImmutableObject("XPen"));
        this.dirty = this.dirty || this.dashPattern != value;
        this.dashPattern = value;
      }
    }
    double[] dashPattern;

    internal Pen RealizeGdiPen()
    {
      if (this.dirty)
      {
        if (this.pen == null)
          this.pen = new Pen(this.color.ToGdiColor(), (float)this.width);
        else
        {
          this.pen.Color = this.color.ToGdiColor();
          this.pen.Width = (float)this.width;
        }
        LineCap lineCap = XConvert.ToLineCap(this.lineCap);
        this.pen.StartCap = lineCap;
        this.pen.EndCap = lineCap;
        this.pen.LineJoin = XConvert.ToLineJoin(this.lineJoin);
        //this.pen.DashOffset
        this.pen.DashStyle = (DashStyle)this.dashStyle;
        this.dirty = false;
      }
      return this.pen;
    }
    bool dirty = true;
    bool immutable;
    Pen pen;
  }
}
