//
// PDFSharp - A library for processing PDF
//
// Authors:
//   Stefan Lange (mailto:Stefan.Lange@pdfsharp.com)
//
// Copyright (c) 2005 empira Software GmbH, Cologne (Germany)
//
// http://www.pdfsharp.com
//
// Permission is hereby granted, free of charge, to any person obtaining
// a copy of this software and associated documentation files (the
// "Software"), to deal in the Software without restriction, including
// without limitation the rights to use, copy, modify, merge, publish,
// distribute, sublicense, and/or sell copies of the Software, and to
// permit persons to whom the Software is furnished to do so, subject to
// the following conditions:
// 
// The above copyright notice and this permission notice shall be
// included in all copies or substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
// EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
// MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
// NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
// LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
// OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
// WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

using System;
using System.Diagnostics;
using System.Runtime.InteropServices;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Drawing.Imaging;
using System.IO;
using System.ComponentModel;
using PdfSharp.Internal;
using PdfSharp.Fonts.TrueType;
using PdfSharp.Pdf;
using PdfSharp.Pdf.IO;
using PdfSharp.Pdf.Advanced;

namespace PdfSharp.Drawing
{
  /// <summary>
  /// Represents a so called 'PDF form external object', which is typically an imported page of an external
  /// PDF document. XPdfForm objects are used like images to draw an existing PDF page of an external
  /// document in the current document. XPdfForm objects can only be placed in PDF documents. If you try
  /// to draw them using a XGraphics based on an GDI+ context no action is taken if no placeholder image
  /// is specified. Otherwise the place holder is drawn.
  /// </summary>
  public class XPdfForm : XImage  // TODO: make disposable
  {
    /// <summary>
    /// Initializes a new instance of the XPdfForm class from the specified path to an external PDF document.
    /// Although PDFsharp internally caches XPdfForm objects it is recommended to reuse PdfXFrom objects
    /// in your code and change the PageNumber property if more than one page is neede form the external
    /// document. Furthermore, because XPdfForm can occupy very much memory, it is recommended to
    /// dispose XPdfForm objects if not needed anymore.
    /// </summary>
    internal XPdfForm(string path)
    {
      int pageNumber;
      path = ExtractPageNumber(path, out pageNumber);

      path = Path.GetFullPath(path);
      if (!File.Exists(path))
        throw new FileNotFoundException(PSSR.FileNotFound(path), path);

      if (PdfReader.TestPdfFile(path) == 0)
        throw new ArgumentException("The specified file has no valid PDF file header.", "path");

      this.path = path;
      if (pageNumber != 0)
        PageNumber = pageNumber;
    }

    public static new XPdfForm FromFile(string path)
    {
      return new XPdfForm(path);
    }

    void Initialize()
    {
      // ImageFormat has no overridden Equals...
    }

    /// <summary>
    /// Frees the memory occupied by the underlying imported PDF document, even if other XPdfForm objects
    /// refer to this document. A reuse of this object doesnt fail, because the underlying PDF document
    /// is re-imported if neccessary. NYI
    /// </summary>
    public void Dispose()
    {
      // TODO: remove from thread local cache and from all other XPdfForm objects.
    }

    /// <summary>
    /// Gets or sets an image that is used for drawing if the current XGraphics object cannot handle
    /// PDF forms. A place holder is usedful for showing a preview of a page on the display, because
    /// PDFsharp cannot render native PDF objects.
    /// </summary>
    public XImage PlaceHolder
    {
      get {return this.placeHolder;}
      set {this.placeHolder = value;}
    }
    XImage placeHolder;

    /// <summary>
    /// Gets the number of pages in the PDF form.
    /// </summary>
    public int PageCount
    {
      get 
      { 
        if (this.pageCount == -1)
          this.pageCount = ExternalDocument.Pages.Count;
        return this.pageCount; 
      }
    }
    int pageCount = -1;

    /// <summary>
    /// Get the width of the page identified by the property PageNumber.
    /// </summary>
    public override int Width
    {
      get 
      {
        PdfPage page = ExternalDocument.Pages[this.pageNumber - 1];
        return (int)page.Width;
      }
    }

    /// <summary>
    /// Get the width of the page identified by the property PageNumber.
    /// </summary>
    public override int Height
    {
      get 
      {
        PdfPage page = ExternalDocument.Pages[this.pageNumber - 1];
        return (int)page.Height;
      }
    }

    public override XSize Size
    {
      get
      {
        PdfPage page = ExternalDocument.Pages[this.pageNumber - 1];
        return new XSize(page.Width, page.Height);
      }
    }

    public override double HorizontalResolution
    {
      get {return 72;}
    }

    public override double VerticalResolution
    {
      get {return 72;}
    }

    public XRect BoundingBox
    {
      get { return this.boundingBox; }
      set { this.boundingBox = value; }  // TODO: pdfForm = null
    }
    XRect boundingBox;

    public XMatrix Transform
    {
      get { return this.transform; }
      set 
      {
        if (this.transform != value)
        {
          // discard PdfFromXObject when Transform changed
          this.pdfForm = null;
          this.transform = value; 
        }
      }
    }
    XMatrix transform = XMatrix.Identity;

    /// <summary>
    /// Gets or sets the page number in the external PDF document this object refers to. The page number
    /// is one-based, i.e. it is in the range from 1 to PageCount. The default value is 1.
    /// </summary>
    public int PageNumber
    {
      get { return this.pageNumber; }
      set 
      { 
        if (this.pageNumber != value)
        {
          this.pageNumber = value;
          // dispose PdfFromXObject when number has changed
          this.pdfForm = null;
        }
      }
    }
    int pageNumber = 1;

    /// <summary>
    /// Gets the underlying document from which pages are imported.
    /// </summary>
    internal PdfDocument ExternalDocument
    {
      // The problem is that you can ask an XPdfForm about the number of its pages before it was
      // drawn the first time. At this moment the XPdfForm doesnt know the document where it will
      // be later draw one of its pages. To prevent the import of the same document more than
      // once, all imported documents of a thread are cached. The cache is local to the current 
      // thread and not to the appdomain, because I wont get problems in a multi-thread environment
      // that I dont understand.
      get
      {
        if (this.externalDocument == null)
          this.externalDocument = PdfDocument.Tls.GetDocument(path);
        return this.externalDocument;
      }
    }
    internal PdfDocument externalDocument;

    /// <summary>
    /// Extracts the page number if the path has the form 'MyFile.pdf#123' and returns
    /// the actual path without the number sign and the following digits.
    /// </summary>
    public static string ExtractPageNumber(string path, out int pageNumber)
    {
      if (path == null)
        throw new ArgumentNullException("path");

      pageNumber = 0;
      int length = path.Length;
      if (length != 0)
      {
        length--;
        if (Char.IsDigit(path, length))
        {
          while(Char.IsDigit(path, length) && length >= 0)
            length--;
          if (length > 0 && path[length] == '#')
          {
            // must have at least one dot left of colon to distinguish from e.g. '#123'
            if (path.IndexOf('.') != -1)
            {
              pageNumber = Int32.Parse(path.Substring(length + 1));
              path = path.Substring(0, length);
            }
          }
        }
      }
      return path;
    }

    // HACK: ImportedObjectTable here prevents to get reused from other documents
    // TODO: Rewrite PdfFormXObject ctor to get ImportedObjectTable from current document
    internal ImportedObjectTable importDocument;

    /// <summary>
    /// The PdfFormXObject gets invalid when PageNumber or transform changed. This is because a modification
    /// of an XPdfForm must not change objects that are already been drawn.
    /// </summary>
    internal PdfFormXObject pdfForm;
  }
}
