using System;
using System.Runtime.InteropServices;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Drawing.Imaging;
using System.IO;
using System.ComponentModel;
using PdfSharp.Internal;
using PdfSharp.Fonts.TrueType;
using PdfSharp.Pdf;
using PdfSharp.Pdf.IO;
using PdfSharp.Pdf.Advanced;

namespace PdfSharp.Drawing
{
  /// <summary>
  /// Defines an object used to draw lines and curves.
  /// An abstract base class that provides functionality for the Bitmap and Metafile descended classes.
  /// </summary>
  public class XImage //: IDisposable
  {
    protected XImage()
    {
    }

    XImage(Image image)
    {
      this.image = image;
      Initialize();
    }

    XImage(string path)
    {
      path = Path.GetFullPath(path);
      if (!File.Exists(path))
        throw new FileNotFoundException(PSSR.FileNotFound(path), path);

      this.path = path;

      //FileStream file = new FileStream(filename, FileMode.Open);
      //BitsLength = (int)file.Length;
      //Bits = new byte[BitsLength];
      //file.Read(Bits, 0, BitsLength);
      //file.Close();
      this.image = Image.FromFile(path);

#if false
      float vres = this.image.VerticalResolution;
      float hres = this.image.HorizontalResolution;
      SizeF size = this.image.PhysicalDimension;
      int flags  = this.image.Flags;
      Size sz    = this.image.Size;
      GraphicsUnit units = GraphicsUnit.Millimeter;
      RectangleF rect = this.image.GetBounds(ref units);
      int width = this.image.Width;
#endif
      Initialize();
    }

    public static XImage FromGdiPlusImage(Image image)
    {
      return new XImage(image);
    }

    public static XImage FromFile(string path)
    {
      if (PdfReader.TestPdfFile(path) > 0)
        return new XPdfForm(path);
      return new XImage(path);
    }

    void Initialize()
    {
      // ImageFormat has no overridden Equals...
      string guid = this.image.RawFormat.Guid.ToString("B").ToUpper();
      switch (guid)
      {
        case "{B96B3CAA-0728-11D3-9D7B-0000F81EF32E}":  // memoryBMP
        case "{B96B3CAB-0728-11D3-9D7B-0000F81EF32E}":  // bmp
        case "{B96B3CAF-0728-11D3-9D7B-0000F81EF32E}":  // png
          this.format = XImageFormat.Png;
          break;

        case "{B96B3CAE-0728-11D3-9D7B-0000F81EF32E}":  // jpeg
          this.format = XImageFormat.Jpeg;
          break;

        case "{B96B3CB0-0728-11D3-9D7B-0000F81EF32E}":  // gif
          this.format = XImageFormat.Gif;
          break;

        case "{B96B3CB1-0728-11D3-9D7B-0000F81EF32E}":  // tiff
          this.format = XImageFormat.Tiff;
          break;

        case "{B96B3CB5-0728-11D3-9D7B-0000F81EF32E}":  // icon
          this.format = XImageFormat.Icon;
          break;

        case "{B96B3CAC-0728-11D3-9D7B-0000F81EF32E}":  // emf
        case "{B96B3CAD-0728-11D3-9D7B-0000F81EF32E}":  // wmf
        case "{B96B3CB2-0728-11D3-9D7B-0000F81EF32E}":  // exif
        case "{B96B3CB3-0728-11D3-9D7B-0000F81EF32E}":  // photoCD
        case "{B96B3CB4-0728-11D3-9D7B-0000F81EF32E}":  // flashPIX
        default:
          throw new InvalidOperationException("Unsupported image format.");
      }
    }

    //public void Dispose()
    //{
    //  if (this.image != null)
    //    this.image.Dispose();
    //}

    public virtual int Width
    {
      get {return this.image.Width;}
    }

    public virtual int Height
    {
      get {return this.image.Height;}
    }

    public virtual XSize Size
    {
      get {return new XSize(this.image.Width, this.image.Height);}
    }

    public virtual double HorizontalResolution
    {
      get {return this.image.HorizontalResolution;}
    }

    public virtual double VerticalResolution
    {
      get {return this.image.VerticalResolution;}
    }

    public XImageFormat Format
    {
      get {return this.format;}
      //set {this.format = value;}
    }
    XImageFormat format;

#if DEBUG
    // TEST
    internal void CreateAllImages(string name)
    {
      if (this.image != null)
      {
        this.image.Save(name + ".bmp", ImageFormat.Bmp);
        this.image.Save(name + ".emf", ImageFormat.Emf);
        this.image.Save(name + ".exif", ImageFormat.Exif);
        this.image.Save(name + ".gif", ImageFormat.Gif);
        this.image.Save(name + ".ico", ImageFormat.Icon);
        this.image.Save(name + ".jpg", ImageFormat.Jpeg);
        this.image.Save(name + ".png", ImageFormat.Png);
        this.image.Save(name + ".tif", ImageFormat.Tiff);
        this.image.Save(name + ".wmf", ImageFormat.Wmf);
        this.image.Save(name + "2.bmp", ImageFormat.MemoryBmp);
      }
    }
#endif
    internal Image image;
    internal string path;

    /// <summary>
    /// Cache PdfImageTable.ImageSelector to speed up finding the right PdfImage
    /// if this image is used more than once.
    /// </summary>
    internal PdfImageTable.ImageSelector selector;
  }
}
