//
// PDFSharp - A library for processing PDF
//
// Authors:
//   Stefan Lange (mailto:Stefan.Lange@pdfsharp.com)
//
// Copyright (c) 2005 empira Software GmbH, Cologne (Germany)
//
// http://www.pdfsharp.com
//
// Permission is hereby granted, free of charge, to any person obtaining
// a copy of this software and associated documentation files (the
// "Software"), to deal in the Software without restriction, including
// without limitation the rights to use, copy, modify, merge, publish,
// distribute, sublicense, and/or sell copies of the Software, and to
// permit persons to whom the Software is furnished to do so, subject to
// the following conditions:
// 
// The above copyright notice and this permission notice shall be
// included in all copies or substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
// EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
// MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
// NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
// LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
// OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
// WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

using System;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.IO;
using PdfSharp.Pdf;
using PdfSharp.Drawing.Pdf;

using PdfSharp.Pdf.Advanced;

namespace PdfSharp.Drawing
{
  /// <summary>
  /// Represents a drawing surface (or canvas) for a fixed size page.
  /// </summary>
  public class XGraphics
  {
    /// <summary>
    /// Initializes a new instance of the XGraphics class.
    /// </summary>
    XGraphics(Graphics gfx, XSize size)
    {
      if (gfx == null)
        throw new ArgumentNullException("gfx");

      this.gfx = gfx;
      this.drawGraphics = true;
      this.pageSize = new XSize(size.width, size.height);
      Initialize();
    }

    /// <summary>
    /// Initializes a new instance of the XGraphics class.
    /// </summary>
    XGraphics(PdfPage page, XGraphicsPdfPageOptions options)
    {
      if (page == null)
        throw new ArgumentNullException("page");

      this.gfx = Graphics.FromHwnd(IntPtr.Zero);
      this.renderer = new PdfSharp.Drawing.Pdf.XGraphicsPdfRenderer(page, this, options);
      this.pageSize = new XSize(page.Width, page.Height);
      Initialize();
    }

    /// <summary>
    /// Creates a new instance of the XGraphics class from a System.Drawing.Graphics object.
    /// </summary>
    public static XGraphics FromGraphics(Graphics gfx, XSize size)
    {
      // TODO: Get object from cache...
      return new XGraphics(gfx, size);
    }

    /// <summary>
    /// Creates a new instance of the XGraphics class from a PdfSharp.Pdf.PdfPage object.
    /// </summary>
    public static XGraphics FromPdfPage(PdfPage page, XGraphicsPdfPageOptions options)
    {
      // TODO: Get object from cache...
      return new XGraphics(page, options);
    }

    /// <summary>
    /// Creates a new instance of the XGraphics class from a PdfSharp.Pdf.PdfPage object.
    /// </summary>
    public static XGraphics FromPdfPage(PdfPage page)
    {
      // TODO: Get object from cache...
      return new XGraphics(page, XGraphicsPdfPageOptions.Append);
    }

    /// <summary>
    /// Internal setup.
    /// </summary>
    void Initialize()
    {
      // TODO: PageScale???
      this.pageUnit          = XGraphicsUnit.Point;
      this.pageDirection     = XPageDirection.Downwards;
      this.pageOrigin        = new XPoint();
      this.defaultViewMatrix = gfx.Transform;
      this.transform         = new Matrix();
    }

    /// <summary>
    /// Gets or sets the unit of measure used for page coordinates.
    /// CURRENTLY ONLY POINT IS IMPLEMENTED.
    /// </summary>
    public XGraphicsUnit PageUnit
    {
      get {return this.pageUnit;}
      set 
      {
        //TODO
        throw new NotImplementedException("PageUnit");
      }
    }
    XGraphicsUnit pageUnit;

    /// <summary>
    /// Gets or sets the a value indicating in which direction y-value grow.
    /// </summary>
    public XPageDirection PageDirection
    {
      get {return this.pageDirection;}
      set 
      {
        //TODO
        throw new NotImplementedException("PageDirection");
      }
    }
    XPageDirection pageDirection;

    public XPoint PageOrigin
    {
      get {return this.pageOrigin;}
      set 
      {
        //TODO
        throw new NotImplementedException("PageOrigin cannot be modified.");
      }
    }
    XPoint pageOrigin;

    public XSize PageSize
    {
      get {return this.pageSize;}
      set 
      {
        //TODO
        throw new NotImplementedException("PageSize cannot be modified.");
      }
    }
    XSize pageSize;

    //public void Flush();
    //public void Flush(FlushIntention intention);

    #region Drawing

    // ----- Clear --------------------------------------------------------------------------------

    /// <summary>
    /// Fills the entire drawing surface with the specified color. The functions works only if
    /// the current transformation is identity, i.e. the function should be called only immediately
    /// after the XGraphics object was created.
    /// </summary>
    public void Clear(XColor color)
    {
      if (this.drawGraphics)
        this.gfx.Clear(color.ToGdiColor());

      if (this.renderer != null)
        this.renderer.Clear(color);
    }

    // ----- DrawLine -----------------------------------------------------------------------------

    /// <summary>
    /// Draws a line connecting two Point structures.
    /// </summary>
    public void DrawLine(XPen pen, Point pt1, Point pt2)
    {
      DrawLine(pen, (double)pt1.X, (double)pt1.Y, (double)pt2.X, (double)pt2.Y);
    }

    /// <summary>
    /// Draws a line connecting two PointF structures.
    /// </summary>
    public void DrawLine(XPen pen, PointF pt1, PointF pt2)
    {
      DrawLine(pen, pt1.X, pt1.Y, pt2.X, pt2.Y);
    }

    /// <summary>
    /// Draws a line connecting two XPoint structures.
    /// </summary>
    public void DrawLine(XPen pen, XPoint pt1, XPoint pt2)
    {
      DrawLine(pen, pt1.X, pt1.Y, pt2.X, pt2.Y);
    }

    /// <summary>
    /// Draws a line connecting the two points specified by coordinate pairs.
    /// </summary>
    public void DrawLine(XPen pen, int x1, int y1, int x2, int y2)
    {
      DrawLine(pen, (double)x1, (double)y1, (double)x2, (double)y2);
    }

    /// <summary>
    /// Draws a line connecting the two points specified by coordinate pairs.
    /// </summary>
    public void DrawLine(XPen pen, double x1, double y1, double x2, double y2)
    {
      if (pen == null)
        throw new ArgumentNullException("pen");

      if (this.drawGraphics)
        this.gfx.DrawLine(pen.RealizeGdiPen(), (float)x1, (float)y1, (float)x2, (float)y2);

      if (this.renderer != null)
        this.renderer.DrawLines(pen, new XPoint[2]{new XPoint(x1, y1), new XPoint(x2, y2)});
    }

    // ----- DrawLines ----------------------------------------------------------------------------

    public void DrawLines(XPen pen, Point[] points)
    {
      DrawLines(pen, MakePointFArray(points));
    }

    public void DrawLines(XPen pen, PointF[] points)
    {
      if (pen == null)
        throw new ArgumentNullException("pen");
      if (points == null)
        throw new ArgumentNullException("points");
      if (points.Length < 2)
        throw new ArgumentException("points", PSSR.PointArrayAtLeast(2));

      if (this.drawGraphics)
        this.gfx.DrawLines(pen.RealizeGdiPen(), points);

      if (this.renderer != null)
        this.renderer.DrawLines(pen, MakeXPointArray(points));
    }

    public void DrawLines(XPen pen, XPoint[] points)
    {
      if (pen == null)
        throw new ArgumentNullException("pen");
      if (points == null)
        throw new ArgumentNullException("points");
      if (points.Length < 2)
        throw new ArgumentException("points", PSSR.PointArrayAtLeast(2));

      if (this.drawGraphics)
        this.gfx.DrawLines(pen.RealizeGdiPen(), XGraphics.MakePointFArray(points));

      if (this.renderer != null)
        this.renderer.DrawLines(pen, points);
    }

    public void DrawLines(XPen pen, double x, double y, params double[] value)
    {
      if (pen == null)
        throw new ArgumentNullException("pen");
      if (value == null)
        throw new ArgumentNullException("value");

      int length = value.Length;
      XPoint[] points = new XPoint[length / 2 + 1];
      points[0].X = x;
      points[0].Y = y;
      for (int idx = 0; idx < length / 2; idx++)
      {
        points[idx + 1].X = value[2 * idx];
        points[idx + 1].Y = value[2 * idx + 1];
      }
      DrawLines(pen, points);
    }

    // ----- DrawBezier ---------------------------------------------------------------------------

    public void DrawBezier(XPen pen, Point pt1, Point pt2, Point pt3, Point pt4)
    {
      DrawBezier(pen, (double)pt1.X, (double)pt1.Y, (double)pt2.X, (double)pt2.Y,
        (double)pt3.X, (double)pt3.Y, (double)pt4.X, (double)pt4.Y);
    }

    public void DrawBezier(XPen pen, PointF pt1, PointF pt2, PointF pt3, PointF pt4)
    {
      DrawBezier(pen, pt1.X, pt1.Y, pt2.X, pt2.Y, pt3.X, pt3.Y, pt4.X, pt4.Y);
    }

    public void DrawBezier(XPen pen, XPoint pt1, XPoint pt2, XPoint pt3, XPoint pt4)
    {
      DrawBezier(pen, pt1.X, pt1.Y, pt2.X, pt2.Y, pt3.X, pt3.Y, pt4.X, pt4.Y);
    }

    public void DrawBezier(XPen pen, double x1, double y1, double x2, double y2, 
      double x3, double y3,double x4, double y4)
    {
      if (pen == null)
        throw new ArgumentNullException("pen");

      if (this.drawGraphics)
        this.gfx.DrawBezier(pen.RealizeGdiPen(), (float)x1, (float)y1, (float)x2, (float)y2, (float)x3, (float)y3, (float)x4, (float)y4);

      if (this.renderer != null)
        this.renderer.DrawBeziers(pen, new XPoint[4]{new XPoint(x1, y1), new XPoint(x2, y2), 
                                                     new XPoint(x3, y3), new XPoint(x4, y4)});
    }

    // ----- DrawBeziers --------------------------------------------------------------------------

    public void DrawBeziers(XPen pen, Point[] points)
    {
      DrawBeziers(pen, MakeXPointArray(points));
    }

    public void DrawBeziers(XPen pen, PointF[] points)
    {
      DrawBeziers(pen, MakeXPointArray(points));
    }

    public void DrawBeziers(XPen pen, XPoint[] points)
    {
      if (pen == null)
        throw new ArgumentNullException("pen");
      // TODO: check number of points: 4 + 3n or something like that...

      if (this.drawGraphics)
        this.gfx.DrawBeziers(pen.RealizeGdiPen(), MakePointFArray(points));

      if (this.renderer != null)
        this.renderer.DrawBeziers(pen, points);
    }

    // ----- DrawCurve ----------------------------------------------------------------------------

    public void DrawCurve(XPen pen, Point[] points)
    {
      DrawCurve(pen, MakePointFArray(points), 0.5);
    }

    public void DrawCurve(XPen pen, PointF[] points)
    {
      DrawCurve(pen, MakeXPointArray(points), 0.5);
    }

    public void DrawCurve(XPen pen, XPoint[] points)
    {
      DrawCurve(pen, points, 0.5);
    }

    public void DrawCurve(XPen pen, Point[] points, double tension)
    {
      DrawCurve(pen, MakeXPointArray(points), tension);
    }

    public void DrawCurve(XPen pen, PointF[] points, double tension)
    {
      DrawCurve(pen, MakeXPointArray(points), tension);
    }

    public void DrawCurve(XPen pen, XPoint[] points, double tension)
    {
      if (pen == null)
        throw new ArgumentNullException("pen");
      if (points == null)
        throw new ArgumentNullException("points");
      if (points.Length < 2)
        throw new ArgumentException("DrawCurve requires two or more points.", "points");

      if (this.drawGraphics)
        this.gfx.DrawCurve(pen.RealizeGdiPen(), MakePointFArray(points), (float)tension);

      if (this.renderer != null)
        this.renderer.DrawCurve(pen, points, tension);
    }

    // TODO:
    //public void DrawCurve(XPen pen, PointF[] points, int offset, int numberOfSegments);
    //public void DrawCurve(XPen pen, Point[] points, int offset, int numberOfSegments, double tension);
    //public void DrawCurve(XPen pen, PointF[] points, int offset, int numberOfSegments, double tension);

    // ----- DrawArc ------------------------------------------------------------------------------

    public void DrawArc(XPen pen, Rectangle rect, double startAngle, double sweepAngle)
    {
      DrawArc(pen, (double)rect.X, (double)rect.Y, (double)rect.Width, (double)rect.Height, startAngle, sweepAngle);
    }

    public void DrawArc(XPen pen, RectangleF rect, double startAngle, double sweepAngle)
    {
      DrawArc(pen, rect.X, rect.Y, rect.Width, rect.Height, startAngle, sweepAngle);
    }

    public void DrawArc(XPen pen, XRect rect, double startAngle, double sweepAngle)
    {
      DrawArc(pen, rect.X, rect.Y, rect.Width, rect.Height, startAngle, sweepAngle);
    }

    public void DrawArc(XPen pen, int x, int y, int width, int height, int startAngle, int sweepAngle)
    {
      DrawArc(pen, (double)x, (double)y, (double)width, (double)height, startAngle, sweepAngle);
    }

    public void DrawArc(XPen pen, double x, double y, double width, double height, double startAngle, double sweepAngle)
    {
      if (pen == null)
        throw new ArgumentNullException("pen");

      if (Math.Abs(sweepAngle) >= 360)
      {
        DrawEllipse(pen, x, y, width, height);
      }
      else
      {
        if (this.drawGraphics)
          this.gfx.DrawArc(pen.RealizeGdiPen(), (float)x, (float)y, (float)width, (float)height, (float)startAngle, (float)sweepAngle);

        if (this.renderer != null)
          this.renderer.DrawArc(pen, x, y, width, height, startAngle, sweepAngle);
      }
    }

    // ----- DrawRectangle ------------------------------------------------------------------------

    // ----- stroke -----

    public void DrawRectangle(XPen pen, Rectangle rect)
    {
      DrawRectangle(pen, (double)rect.X, (double)rect.Y, (double)rect.Width, (double)rect.Height);
    }

    public void DrawRectangle(XPen pen, RectangleF rect)
    {
      DrawRectangle(pen, rect.X, rect.Y, rect.Width, rect.Height);
    }

    public void DrawRectangle(XPen pen, XRect rect)
    {
      DrawRectangle(pen, rect.X, rect.Y, rect.Width, rect.Height);
    }

    public void DrawRectangle(XPen pen, int x, int y, int width, int height)
    {
      DrawRectangle(pen, (double)x, (double)y, (double)width, (double)height);
    }

    public void DrawRectangle(XPen pen, double x, double y, double width, double height)
    {
      if (pen == null)
        throw new ArgumentNullException("pen");

      if (this.drawGraphics)
        this.gfx.DrawRectangle(pen.RealizeGdiPen(), (float)x, (float)y, (float)width, (float)height);

      if (this.renderer != null)
        this.renderer.DrawRectangle(pen, null, x, y, width, height);
    }

    // ----- fill -----

    public void DrawRectangle(XBrush brush, Rectangle rect)
    {
      DrawRectangle(brush, (double)rect.X, (double)rect.Y, (double)rect.Width, (double)rect.Height);
    }

    public void DrawRectangle(XBrush brush, RectangleF rect)
    {
      DrawRectangle(brush, rect.X, rect.Y, rect.Width, rect.Height);
    }

    public void DrawRectangle(XBrush brush, XRect rect)
    {
      DrawRectangle(brush, rect.X, rect.Y, rect.Width, rect.Height);
    }

    public void DrawRectangle(XBrush brush, int x, int y, int width, int height)
    {
      DrawRectangle(brush, (double)x, (double)y, (double)width, (double)height);
    }

    public void DrawRectangle(XBrush brush, double x, double y, double width, double height)
    {
      if (brush == null)
        throw new ArgumentNullException("brush");

      if (this.drawGraphics)
        this.gfx.FillRectangle(brush.RealizeGdiBrush(), (float)x, (float)y, (float)width, (float)height);

      if (this.renderer != null)
        this.renderer.DrawRectangle(null, brush, x, y, width, height);
    }

    // ----- stroke and fill -----

    public void DrawRectangle(XPen pen, XBrush brush, Rectangle rect)
    {
      DrawRectangle(pen, brush, (double)rect.X, (double)rect.Y, (double)rect.Width, (double)rect.Height);
    }

    public void DrawRectangle(XPen pen, XBrush brush, RectangleF rect)
    {
      DrawRectangle(pen, brush, rect.X, rect.Y, rect.Width, rect.Height);
    }

    public void DrawRectangle(XPen pen, XBrush brush, XRect rect)
    {
      DrawRectangle(pen, brush, rect.X, rect.Y, rect.Width, rect.Height);
    }

    public void DrawRectangle(XPen pen, XBrush brush, int x, int y, int width, int height)
    {
      DrawRectangle(pen, brush, (double)x, (double)y, (double)width, (double)height);
    }

    public void DrawRectangle(XPen pen, XBrush brush, double x, double y, double width, double height)
    {
      if (pen == null && brush == null)
        throw new ArgumentNullException("pen and brush", PSSR.NeedPenOrBrush);

      if (this.drawGraphics)
      {
        this.gfx.FillRectangle(brush.RealizeGdiBrush(), (float)x, (float)y, (float)width, (float)height);
        this.gfx.DrawRectangle(pen.RealizeGdiPen(), (float)x, (float)y, (float)width, (float)height);
      }
      if (this.renderer != null)
        this.renderer.DrawRectangle(pen, brush, x, y, width, height);
    }

    // ----- DrawRectangles -----------------------------------------------------------------------

    // ----- stroke -----

    public void DrawRectangles(XPen pen, Rectangle[] rectangles)
    {
      if (pen == null)
        throw new ArgumentNullException("pen");
      if (rectangles == null)
        throw new ArgumentNullException("rectangles");

      DrawRectangles(pen, null, rectangles);
    }

    public void DrawRectangles(XPen pen, RectangleF[] rectangles)
    {
      if (pen == null)
        throw new ArgumentNullException("pen");
      if (rectangles == null)
        throw new ArgumentNullException("rectangles");

      DrawRectangles(pen, null, rectangles);
    }

    public void DrawRectangles(XPen pen, XRect[] rectangles)
    {
      if (pen == null)
        throw new ArgumentNullException("pen");
      if (rectangles == null)
        throw new ArgumentNullException("rectangles");

      DrawRectangles(pen, null, rectangles);
    }

    // ----- fill -----

    public void DrawRectangles(XBrush brush, Rectangle[] rectangles)
    {
      if (brush == null)
        throw new ArgumentNullException("brush");
      if (rectangles == null)
        throw new ArgumentNullException("rectangles");

      DrawRectangles(null, brush, rectangles);
    }

    public void DrawRectangles(XBrush brush, RectangleF[] rectangles)
    {
      if (brush == null)
        throw new ArgumentNullException("brush");
      if (rectangles == null)
        throw new ArgumentNullException("rectangles");

      DrawRectangles(null, brush, rectangles);
    }

    public void DrawRectangles(XBrush brush, XRect[] rectangles)
    {
      if (brush == null)
        throw new ArgumentNullException("brush");
      if (rectangles == null)
        throw new ArgumentNullException("rectangles");

      DrawRectangles(null, brush, rectangles);
    }

    // ----- stroke and fill -----

    public void DrawRectangles(XPen pen, XBrush brush, Rectangle[] rectangles)
    {
      if (pen == null && brush == null)
        throw new ArgumentNullException("pen and brush", PSSR.NeedPenOrBrush);
      if (rectangles == null)
        throw new ArgumentNullException("rectangles");

      if (this.drawGraphics)
      {
        this.gfx.FillRectangles(brush.RealizeGdiBrush(), rectangles);
        this.gfx.DrawRectangles(pen.RealizeGdiPen(), rectangles);
      }
      if (this.renderer != null)
      {
        int count = rectangles.Length;
        for (int idx = 0; idx < count; idx++)
        {
          Rectangle rect = rectangles[idx];
          this.renderer.DrawRectangle(pen, brush, rect.X, rect.Y, rect.Width, rect.Height);
        }
      }
    }

    public void DrawRectangles(XPen pen, XBrush brush, RectangleF[] rectangles)
    {
      if (pen == null && brush == null)
        throw new ArgumentNullException("pen and brush", PSSR.NeedPenOrBrush);
      if (rectangles == null)
        throw new ArgumentNullException("rectangles");

      if (this.drawGraphics)
      {
        this.gfx.FillRectangles(brush.RealizeGdiBrush(), rectangles);
        this.gfx.DrawRectangles(pen.RealizeGdiPen(), rectangles);
      }
      if (this.renderer != null)
      {
        int count = rectangles.Length;
        for (int idx = 0; idx < count; idx++)
        {
          RectangleF rect = rectangles[idx];
          this.renderer.DrawRectangle(pen, brush, rect.X, rect.Y, rect.Width, rect.Height);
        }
      }
    }

    public void DrawRectangles(XPen pen, XBrush brush, XRect[] rectangles)
    {
      if (pen == null && brush == null)
        throw new ArgumentNullException("pen and brush", PSSR.NeedPenOrBrush);
      if (rectangles == null)
        throw new ArgumentNullException("rectangles");

      int count = rectangles.Length;
      if (this.drawGraphics)
      {
        RectangleF[] rects = new RectangleF[count];
        for (int idx = 0; idx < count; idx++)
        {
          XRect rect = rectangles[idx];
          rects[idx] = new RectangleF((float)rect.X, (float)rect.Y, (float)rect.Width, (float)rect.Height);
        }
        this.gfx.FillRectangles(brush.RealizeGdiBrush(), rects);
        this.gfx.DrawRectangles(pen.RealizeGdiPen(), rects);
      }
      if (this.renderer != null)
      {
        for (int idx = 0; idx < count; idx++)
        {
          XRect rect = rectangles[idx];
          this.renderer.DrawRectangle(pen, brush, rect.X, rect.Y, rect.Width, rect.Height);
        }
      }
    }

    // ----- DrawRoundedRectangle -----------------------------------------------------------------

    // ----- stroke -----

    public void DrawRoundedRectangle(XPen pen, Rectangle rect, Size ellipseSize)
    {
      DrawRoundedRectangle(pen, (double)rect.X, (double)rect.Y, (double)rect.Width, (double)rect.Height,
        (double)ellipseSize.Width, (double)ellipseSize.Height);
    }

    public void DrawRoundedRectangle(XPen pen, RectangleF rect, SizeF ellipseSize)
    {
      DrawRoundedRectangle(pen, rect.X, rect.Y, rect.Width, rect.Height, ellipseSize.Width, ellipseSize.Height);
    }

    public void DrawRoundedRectangle(XPen pen, XRect rect, XSize ellipseSize)
    {
      DrawRoundedRectangle(pen, rect.X, rect.Y, rect.Width, rect.Height, ellipseSize.Width, ellipseSize.Height);
    }

    public void DrawRoundedRectangle(XPen pen, int x, int y, int width, int height, int ellipseWidth, int ellipseHeight)
    {
      DrawRoundedRectangle(pen, (double)x, (double)y, (double)width, (double)height, (double)ellipseWidth, (double)ellipseHeight);
    }

    public void DrawRoundedRectangle(XPen pen, double x, double y, double width, double height, double ellipseWidth, double ellipseHeight)
    {
      if (pen == null)
        throw new ArgumentNullException("pen");

      DrawRoundedRectangle(pen, null, x, y, width, height, ellipseWidth, ellipseHeight);
    }

    // ----- fill -----

    public void DrawRoundedRectangle(XBrush brush, Rectangle rect, Size ellipseSize)
    {
      DrawRoundedRectangle(brush, (double)rect.X, (double)rect.Y, (double)rect.Width, (double)rect.Height,
        (double)ellipseSize.Width, (double)ellipseSize.Height);
    }

    public void DrawRoundedRectangle(XBrush brush, RectangleF rect, SizeF ellipseSize)
    {
      DrawRoundedRectangle(brush, rect.X, rect.Y, rect.Width, rect.Height, ellipseSize.Width, ellipseSize.Height);
    }

    public void DrawRoundedRectangle(XBrush brush, XRect rect, XSize ellipseSize)
    {
      DrawRoundedRectangle(brush, rect.X, rect.Y, rect.Width, rect.Height, ellipseSize.Width, ellipseSize.Height);
    }

    public void DrawRoundedRectangle(XBrush brush, int x, int y, int width, int height, int ellipseWidth, int ellipseHeight)
    {
      DrawRoundedRectangle(brush, (double)x, (double)y, (double)width, (double)height, (double)ellipseWidth, (double)ellipseHeight);
    }

    public void DrawRoundedRectangle(XBrush brush, double x, double y, double width, double height, double ellipseWidth, double ellipseHeight)
    {
      if (brush == null)
        throw new ArgumentNullException("brush");

      DrawRoundedRectangle(null, brush, x, y, width, height, ellipseWidth, ellipseHeight);
    }

    // ----- stroke and fill -----

    public void DrawRoundedRectangle(XPen pen, XBrush brush, Rectangle rect, Size ellipseSize)
    {
      DrawRoundedRectangle(pen, brush, (double)rect.X, (double)rect.Y, (double)rect.Width, (double)rect.Height,
        (double)ellipseSize.Width, (double)ellipseSize.Height);
    }

    public void DrawRoundedRectangle(XPen pen, XBrush brush, RectangleF rect, SizeF ellipseSize)
    {
      DrawRoundedRectangle(pen, brush, rect.X, rect.Y, rect.Width, rect.Height, ellipseSize.Width, ellipseSize.Height);
    }

    public void DrawRoundedRectangle(XPen pen, XBrush brush, XRect rect, XSize ellipseSize)
    {
      DrawRoundedRectangle(pen, brush, rect.X, rect.Y, rect.Width, rect.Height, ellipseSize.Width, ellipseSize.Height);
    }

    public void DrawRoundedRectangle(XPen pen, XBrush brush, int x, int y, int width, int height, int ellipseWidth, int ellipseHeight)
    {
      DrawRoundedRectangle(pen, brush, (double)x, (double)y, (double)width, (double)height, (double)ellipseWidth, (double)ellipseHeight);
    }

    public void DrawRoundedRectangle(XPen pen, XBrush brush, double x, double y, double width, double height, 
      double ellipseWidth, double ellipseHeight)
    {
      if (pen == null && brush == null)
        throw new ArgumentNullException("pen and brush", PSSR.NeedPenOrBrush);


      if (this.drawGraphics)
      {
        XGraphicsPath path = new XGraphicsPath();
        path.AddRoundedRectangle(x, y, width, height, ellipseWidth, ellipseHeight);
        DrawPath(pen, brush, path);

        //GraphicsPath gdiPath = path.RealizeGdiPath();
        //if (brush != null)
        //  this.gfx.FillPath(brush.RealizeGdiBrush(), gdiPath);
        //if (pen != null)
        //  this.gfx.DrawPath(pen.RealizeGdiPen(), gdiPath);
      }
      if (this.renderer != null)
        this.renderer.DrawRoundedRectangle(pen, brush, x, y, width, height, ellipseWidth, ellipseHeight);
    }

    // ----- DrawEllipse --------------------------------------------------------------------------

    // ----- stroke -----

    public void DrawEllipse(XPen pen, Rectangle rect)
    {
      DrawEllipse(pen, (double)rect.X, (double)rect.Y, (double)rect.Width, (double)rect.Height);
    }

    public void DrawEllipse(XPen pen, RectangleF rect)
    {
      DrawEllipse(pen, rect.X, rect.Y, rect.Width, rect.Height);
    }

    public void DrawEllipse(XPen pen, XRect rect)
    {
      DrawEllipse(pen, rect.X, rect.Y, rect.Width, rect.Height);
    }

    public void DrawEllipse(XPen pen, int x, int y, int width, int height)
    {
      DrawEllipse(pen, (double)x, (double)y, (double)width, (double)height);
    }

    public void DrawEllipse(XPen pen, double x, double y, double width, double height)
    {
      if (pen == null)
        throw new ArgumentNullException("pen");

      // No DrawArc defined?
      if (this.drawGraphics)
        this.gfx.DrawArc(pen.RealizeGdiPen(), (float)x, (float)y, (float)width, (float)height, 0, 360);

      if (this.renderer != null)
        this.renderer.DrawEllipse(pen, null, x, y, width, height);
    }

    // ----- fill -----

    public void DrawEllipse(XBrush brush, Rectangle rect)
    {
      DrawEllipse(brush, (double)rect.X, (double)rect.Y, (double)rect.Width, (double)rect.Height);
    }

    public void DrawEllipse(XBrush brush, RectangleF rect)
    {
      DrawEllipse(brush, rect.X, rect.Y, rect.Width, rect.Height);
    }

    public void DrawEllipse(XBrush brush, XRect rect)
    {
      DrawEllipse(brush, rect.X, rect.Y, rect.Width, rect.Height);
    }

    public void DrawEllipse(XBrush brush, int x, int y, int width, int height)
    {
      DrawEllipse(brush, (double)x, (double)y, (double)width, (double)height);
    }

    public void DrawEllipse(XBrush brush, double x, double y, double width, double height)
    {
      if (brush == null)
        throw new ArgumentNullException("brush");

      if (this.drawGraphics)
        this.gfx.FillEllipse(brush.RealizeGdiBrush(), (float)x, (float)y, (float)width, (float)height);

      if (this.renderer != null)
        this.renderer.DrawEllipse(null, brush, x, y, width, height);
    }

    // ----- stroke and fill -----

    public void DrawEllipse(XPen pen, XBrush brush, Rectangle rect)
    {
      DrawEllipse(pen, brush, (double)rect.X, (double)rect.Y, (double)rect.Width, (double)rect.Height);
    }

    public void DrawEllipse(XPen pen, XBrush brush, RectangleF rect)
    {
      DrawEllipse(pen, brush, rect.X, rect.Y, rect.Width, rect.Height);
    }

    public void DrawEllipse(XPen pen, XBrush brush, XRect rect)
    {
      DrawEllipse(pen, brush, rect.X, rect.Y, rect.Width, rect.Height);
    }

    public void DrawEllipse(XPen pen, XBrush brush, int x, int y, int width, int height)
    {
      DrawEllipse(pen, brush, (double)x, (double)y, (double)width, (double)height);
    }

    public void DrawEllipse(XPen pen, XBrush brush, double x, double y, double width, double height)
    {
      if (pen == null && brush == null)
        throw new ArgumentNullException("pen and brush", PSSR.NeedPenOrBrush);

      if (this.drawGraphics)
      {
        this.gfx.FillEllipse(brush.RealizeGdiBrush(), (float)x, (float)y, (float)width, (float)height);
        this.gfx.DrawArc(pen.RealizeGdiPen(), (float)x, (float)y, (float)width, (float)height, 0, 360);
      }
      if (this.renderer != null)
        this.renderer.DrawEllipse(pen, brush, x, y, width, height);
    }

    // ----- DrawPolygon --------------------------------------------------------------------------

    // ----- stroke -----

    public void DrawPolygon(XPen pen, Point[] points)
    {
      DrawPolygon(pen, MakePointFArray(points));
    }

    public void DrawPolygon(XPen pen, PointF[] points)
    {
      if (pen == null)
        throw new ArgumentNullException("pen");
      if (points == null)
        throw new ArgumentNullException("points");
      if (points.Length < 2)
        throw new ArgumentException("points", PSSR.PointArrayAtLeast(2));

      if (this.drawGraphics)
        this.gfx.DrawPolygon(pen.RealizeGdiPen(), points);

      if (this.renderer != null)
        this.renderer.DrawPolygon(pen, null, MakeXPointArray(points), XFillMode.Alternate);  // XFillMode is ignored
    }

    public void DrawPolygon(XPen pen, XPoint[] points)
    {
      if (pen == null)
        throw new ArgumentNullException("pen");
      if (points == null)
        throw new ArgumentNullException("points");
      if (points.Length < 2)
        throw new ArgumentException("points", PSSR.PointArrayAtLeast(2));

      if (this.drawGraphics)
        this.gfx.DrawPolygon(pen.RealizeGdiPen(), MakePointFArray(points));

      if (this.renderer != null)
        this.renderer.DrawPolygon(pen, null, points, XFillMode.Alternate);  // XFillMode is ignored
    }

    // ----- fill -----

    public void DrawPolygon(XBrush brush, Point[] points, XFillMode fillmode)
    {
      DrawPolygon(brush, MakePointFArray(points), fillmode);
    }

    public void DrawPolygon(XBrush brush, PointF[] points, XFillMode fillmode)
    {
      if (brush == null)
        throw new ArgumentNullException("brush");
      if (points == null)
        throw new ArgumentNullException("points");
      if (points.Length < 2)
        throw new ArgumentException("points", PSSR.PointArrayAtLeast(2));

      if (this.drawGraphics)
        this.gfx.FillPolygon(brush.RealizeGdiBrush(), points, (FillMode)fillmode);

      if (this.renderer != null)
        this.renderer.DrawPolygon(null, brush, MakeXPointArray(points), fillmode);
    }

    public void DrawPolygon(XBrush brush, XPoint[] points, XFillMode fillmode)
    {
      if (brush == null)
        throw new ArgumentNullException("brush");
      if (points == null)
        throw new ArgumentNullException("points");
      if (points.Length < 2)
        throw new ArgumentException("points", PSSR.PointArrayAtLeast(2));

      if (this.drawGraphics)
        this.gfx.FillPolygon(brush.RealizeGdiBrush(), MakePointFArray(points), (FillMode)fillmode);

      if (this.renderer != null)
        this.renderer.DrawPolygon(null, brush, points, fillmode);
    }

    // ----- stroke and fill -----

    public void DrawPolygon(XPen pen, XBrush brush, Point[] points, XFillMode fillmode)
    {
      DrawPolygon(pen, brush, MakePointFArray(points), fillmode);
    }

    public void DrawPolygon(XPen pen, XBrush brush, PointF[] points, XFillMode fillmode)
    {
      if (pen == null && brush == null)
        throw new ArgumentNullException("pen and brush", PSSR.NeedPenOrBrush);
      if (points == null)
        throw new ArgumentNullException("points");
      if (points.Length < 2)
        throw new ArgumentException("points", PSSR.PointArrayAtLeast(2));

      if (this.drawGraphics)
      {
        this.gfx.FillPolygon(brush.RealizeGdiBrush(), points, (FillMode)fillmode);
        this.gfx.DrawPolygon(pen.RealizeGdiPen(), points);
      }
      if (this.renderer != null)
        this.renderer.DrawPolygon(pen, brush, MakeXPointArray(points), fillmode);
    }

    public void DrawPolygon(XPen pen, XBrush brush, XPoint[] points, XFillMode fillmode)
    {
      if (pen == null && brush == null)
        throw new ArgumentNullException("pen and brush", PSSR.NeedPenOrBrush);
      if (points == null)
        throw new ArgumentNullException("points");
      if (points.Length < 2)
        throw new ArgumentException("points", PSSR.PointArrayAtLeast(2));

      if (this.drawGraphics)
      {
        PointF[] pts = MakePointFArray(points);
        this.gfx.FillPolygon(brush.RealizeGdiBrush(), pts, (FillMode)fillmode);
        this.gfx.DrawPolygon(pen.RealizeGdiPen(), pts);
      }
      if (this.renderer != null)
        this.renderer.DrawPolygon(pen, brush, points, fillmode);
    }

    // ----- DrawPie ------------------------------------------------------------------------------

    // ----- stroke -----

    public void DrawPie(XPen pen, Rectangle rect, double startAngle, double sweepAngle)
    {
      DrawPie(pen, (double)rect.X, (double)rect.Y, (double)rect.Width, (double)rect.Height, startAngle, sweepAngle);
    }

    public void DrawPie(XPen pen, RectangleF rect, double startAngle, double sweepAngle)
    {
      DrawPie(pen, rect.X, rect.Y, rect.Width, rect.Height, startAngle, sweepAngle);
    }

    public void DrawPie(XPen pen, XRect rect, double startAngle, double sweepAngle)
    {
      DrawPie(pen, rect.X, rect.Y, rect.Width, rect.Height, startAngle, sweepAngle);
    }

    public void DrawPie(XPen pen, int x, int y, int width, int height, int startAngle, int sweepAngle)
    {
      DrawPie(pen, (double)x, (double)y, (double)width, (double)height, (double)startAngle, (double)sweepAngle);
    }

    public void DrawPie(XPen pen, double x, double y, double width, double height, double startAngle, double sweepAngle)
    {
      if (pen == null)
        throw new ArgumentNullException("pen", PSSR.NeedPenOrBrush);

      if (this.drawGraphics)
        this.gfx.DrawPie(pen.RealizeGdiPen(), (float)x, (float)y, (float)width, (float)height, (float)startAngle, (float)sweepAngle);

      if (this.renderer != null)
        this.renderer.DrawPie(pen, null, x, y, width, height, startAngle, sweepAngle);
    }

    // ----- fill -----

    public void DrawPie(XBrush brush, Rectangle rect, double startAngle, double sweepAngle)
    {
      DrawPie(brush, (double)rect.X, (double)rect.Y, (double)rect.Width, (double)rect.Height, startAngle, sweepAngle);
    }

    public void DrawPie(XBrush brush, RectangleF rect, double startAngle, double sweepAngle)
    {
      DrawPie(brush, rect.X, rect.Y, rect.Width, rect.Height, startAngle, sweepAngle);
    }

    public void DrawPie(XBrush brush, XRect rect, double startAngle, double sweepAngle)
    {
      DrawPie(brush, rect.X, rect.Y, rect.Width, rect.Height, startAngle, sweepAngle);
    }

    public void DrawPie(XBrush brush, int x, int y, int width, int height, int startAngle, int sweepAngle)
    {
      DrawPie(brush, (double)x, (double)y, (double)width, (double)height, (double)startAngle, (double)sweepAngle);
    }

    public void DrawPie(XBrush brush, double x, double y, double width, double height, double startAngle, double sweepAngle)
    {
      if (brush == null)
        throw new ArgumentNullException("brush", PSSR.NeedPenOrBrush);

      if (this.drawGraphics)
        this.gfx.FillPie(brush.RealizeGdiBrush(), (float)x, (float)y, (float)width, (float)height, (float)startAngle, (float)sweepAngle);

      if (this.renderer != null)
        this.renderer.DrawPie(null, brush, x, y, width, height, startAngle, sweepAngle);
    }

    // ----- stroke and fill -----

    public void DrawPie(XPen pen, XBrush brush, Rectangle rect, double startAngle, double sweepAngle)
    {
      DrawPie(pen, brush, (double)rect.X, (double)rect.Y, (double)rect.Width, (double)rect.Height, startAngle, sweepAngle);
    }

    public void DrawPie(XPen pen, XBrush brush, RectangleF rect, double startAngle, double sweepAngle)
    {
      DrawPie(pen, brush, rect.X, rect.Y, rect.Width, rect.Height, startAngle, sweepAngle);
    }

    public void DrawPie(XPen pen, XBrush brush, XRect rect, double startAngle, double sweepAngle)
    {
      DrawPie(pen, brush, rect.X, rect.Y, rect.Width, rect.Height, startAngle, sweepAngle);
    }

    public void DrawPie(XPen pen, XBrush brush, int x, int y, int width, int height, int startAngle, int sweepAngle)
    {
      DrawPie(pen, brush, (double)x, (double)y, (double)width, (double)height, (double)startAngle, (double)sweepAngle);
    }

    public void DrawPie(XPen pen, XBrush brush, double x, double y, double width, double height, double startAngle, double sweepAngle)
    {
      if (pen == null)
        throw new ArgumentNullException("pen", PSSR.NeedPenOrBrush);
      if (brush == null)
        throw new ArgumentNullException("brush", PSSR.NeedPenOrBrush);

      if (this.drawGraphics)
      {
        this.gfx.FillPie(brush.RealizeGdiBrush(), (float)x, (float)y, (float)width, (float)height, (float)startAngle, (float)sweepAngle);
        this.gfx.DrawPie(pen.RealizeGdiPen(), (float)x, (float)y, (float)width, (float)height, (float)startAngle, (float)sweepAngle);
      }
      if (this.renderer != null)
        this.renderer.DrawPie(pen, brush, x, y, width, height, startAngle, sweepAngle);
    }

    // ----- DrawClosedCurve ----------------------------------------------------------------------

    //TODO parameter check

    // ----- stroke -----

    public void DrawClosedCurve(XPen pen, Point[] points)
    {
      DrawClosedCurve(pen, null, MakeXPointArray(points), XFillMode.Alternate, 0.5);
    }

    public void DrawClosedCurve(XPen pen, PointF[] points)
    {
      DrawClosedCurve(pen, null, MakeXPointArray(points), XFillMode.Alternate, 0.5);
    }

    public void DrawClosedCurve(XPen pen, XPoint[] points)
    {
      DrawClosedCurve(pen, null, points, XFillMode.Alternate, 0.5);
    }

    public void DrawClosedCurve(XPen pen, Point[] points, double tension)
    {
      DrawClosedCurve(pen, null, MakeXPointArray(points), XFillMode.Alternate, tension);
    }

    public void DrawClosedCurve(XPen pen, PointF[] points, double tension)
    {
      DrawClosedCurve(pen, null, MakeXPointArray(points), XFillMode.Alternate, tension);
    }

    public void DrawClosedCurve(XPen pen, XPoint[] points, double tension)
    {
      if (pen == null)
        throw new ArgumentNullException("pen");

      if (this.drawGraphics)
        this.gfx.DrawClosedCurve(pen.RealizeGdiPen(), MakePointFArray(points), (float)tension, FillMode.Alternate);

      if (this.renderer != null)
        this.renderer.DrawClosedCurve(pen, null, points, tension, XFillMode.Alternate);
    }

    // ----- fill -----

    public void DrawClosedCurve(XBrush brush, Point[] points)
    {
      DrawClosedCurve(null, brush, MakeXPointArray(points), XFillMode.Alternate, 0.5);
    }

    public void DrawClosedCurve(XBrush brush, PointF[] points)
    {
      DrawClosedCurve(null, brush, MakeXPointArray(points), XFillMode.Alternate, 0.5);
    }

    public void DrawClosedCurve(XBrush brush, XPoint[] points)
    {
      DrawClosedCurve(null, brush, points, XFillMode.Alternate, 0.5);
    }

    public void DrawClosedCurve(XBrush brush, Point[] points, XFillMode fillmode)
    {
      DrawClosedCurve(null, brush, MakeXPointArray(points), fillmode, 0.5);
    }

    public void DrawClosedCurve(XBrush brush, PointF[] points, XFillMode fillmode)
    {
      DrawClosedCurve(null, brush, MakeXPointArray(points), fillmode, 0.5);
    }

    public void DrawClosedCurve(XBrush brush, XPoint[] points, XFillMode fillmode)
    {
      DrawClosedCurve(null, brush, points, fillmode, 0.5);
    }

    public void DrawClosedCurve(XBrush brush, Point[] points, XFillMode fillmode, double tension)
    {
      DrawClosedCurve(null, brush, MakeXPointArray(points), fillmode, tension);
    }

    public void DrawClosedCurve(XBrush brush, PointF[] points, XFillMode fillmode, double tension)
    {
      DrawClosedCurve(null, brush, MakeXPointArray(points), fillmode, tension);
    }
    
    public void DrawClosedCurve(XBrush brush, XPoint[] points, XFillMode fillmode, double tension)
    {
      if (brush == null)
        throw new ArgumentNullException("brush");

      if (this.drawGraphics)
        this.gfx.FillClosedCurve(brush.RealizeGdiBrush(), MakePointFArray(points), (FillMode)fillmode, (float)tension);

      if (this.renderer != null)
        this.renderer.DrawClosedCurve(null, brush, points, tension, fillmode);
    }
    
    // ----- stroke and fill -----

    public void DrawClosedCurve(XPen pen, XBrush brush, Point[] points)
    {
      DrawClosedCurve(pen, brush, MakeXPointArray(points), XFillMode.Alternate, 0.5);
    }

    public void DrawClosedCurve(XPen pen, XBrush brush, PointF[] points)
    {
      DrawClosedCurve(pen, brush, MakeXPointArray(points), XFillMode.Alternate, 0.5);
    }

    public void DrawClosedCurve(XPen pen, XBrush brush, XPoint[] points)
    {
      DrawClosedCurve(pen, brush, points, XFillMode.Alternate, 0.5);
    }

    public void DrawClosedCurve(XPen pen, XBrush brush, Point[] points, XFillMode fillmode)
    {
      DrawClosedCurve(pen, brush, MakeXPointArray(points), fillmode, 0.5);
    }

    public void DrawClosedCurve(XPen pen, XBrush brush, PointF[] points, XFillMode fillmode)
    {
      DrawClosedCurve(pen, brush, MakeXPointArray(points), fillmode, 0.5);
    }

    public void DrawClosedCurve(XPen pen, XBrush brush, XPoint[] points, XFillMode fillmode)
    {
      DrawClosedCurve(pen, brush, points, fillmode, 0.5);
    }

    public void DrawClosedCurve(XPen pen, XBrush brush, Point[] points, XFillMode fillmode, double tension)
    {
      DrawClosedCurve(pen, brush, MakeXPointArray(points), fillmode, tension);
    }

    public void DrawClosedCurve(XPen pen, XBrush brush, PointF[] points, XFillMode fillmode, double tension)
    {
      DrawClosedCurve(pen, brush, MakeXPointArray(points), fillmode, tension);
    }

    public void DrawClosedCurve(XPen pen, XBrush brush, XPoint[] points, XFillMode fillmode, double tension)
    {
      if (pen == null && brush == null)
        throw new ArgumentNullException("pen and brush", PSSR.NeedPenOrBrush);

      if (this.drawGraphics)
      {
        this.gfx.FillClosedCurve(brush.RealizeGdiBrush(), MakePointFArray(points), (FillMode)fillmode, (float)tension);
        // The fillmode is not used by DrawClosedCurve
        this.gfx.DrawClosedCurve(pen.RealizeGdiPen(), MakePointFArray(points), (float)tension, (FillMode)fillmode);
      }

      if (this.renderer != null)
        this.renderer.DrawClosedCurve(pen, brush, points, tension, fillmode);
    }

    // ----- DrawPath -----------------------------------------------------------------------------

    // ----- stroke -----

    public void DrawPath(XPen pen, XGraphicsPath path)
    {
      if (pen == null)
        throw new ArgumentNullException("pen");
      if (path == null)
        throw new ArgumentNullException("path");

      if (this.drawGraphics)
        this.gfx.DrawPath(pen.RealizeGdiPen(), path.RealizeGdiPath());

      if (this.renderer != null)
        this.renderer.DrawPath(pen, null, path);
    }

    // ----- fill -----

    public void DrawPath(XBrush brush, XGraphicsPath path)
    {
      if (brush == null)
        throw new ArgumentNullException("brush");
      if (path == null)
        throw new ArgumentNullException("path");

      if (this.drawGraphics)
        this.gfx.FillPath(brush.RealizeGdiBrush(), path.RealizeGdiPath());

      if (this.renderer != null)
        this.renderer.DrawPath(null, brush, path);
    }

    // ----- stroke and fill -----

    public void DrawPath(XPen pen, XBrush brush, XGraphicsPath path)
    {
      if (pen == null && brush == null)
        throw new ArgumentNullException("pen and brush", PSSR.NeedPenOrBrush);
      if (path == null)
        throw new ArgumentNullException("path");

      if (this.drawGraphics)
      {
        if (brush != null)
          this.gfx.FillPath(brush.RealizeGdiBrush(), path.RealizeGdiPath());
        if (pen != null)
          this.gfx.DrawPath(pen.RealizeGdiPen(), path.RealizeGdiPath());
      }
      if (this.renderer != null)
        this.renderer.DrawPath(pen, brush, path);
    }

    // ----- DrawString ---------------------------------------------------------------------------

    public void DrawString(string s, XFont font, XBrush brush, PointF point)
    {
      DrawString(s, font, brush, new XRect(point.X, point.Y, 0, 0), XStringFormat.Default);
    }

    public void DrawString(string s, XFont font, XBrush brush, XPoint point)
    {
      DrawString(s, font, brush, new XRect(point.X, point.Y, 0, 0), XStringFormat.Default);
    }

    public void DrawString(string s, XFont font, XBrush brush, PointF point, XStringFormat format)
    {
      DrawString(s, font, brush, new XRect(point.X, point.Y, 0, 0), format);
    }

    public void DrawString(string s, XFont font, XBrush brush, XPoint point, XStringFormat format)
    {
      DrawString(s, font, brush, new XRect(point.X, point.Y, 0, 0), format);
    }

    public void DrawString(string s, XFont font, XBrush brush, double x, double y)
    {
      DrawString(s, font, brush, new XRect(x, y, 0, 0), XStringFormat.Default);
    }

    public void DrawString(string s, XFont font, XBrush brush, double x, double y, XStringFormat format)
    {
      DrawString(s, font, brush, new XRect(x, y, 0, 0), format);
    }

    public void DrawString(string s, XFont font, XBrush brush, RectangleF layoutRectangle)
    {
      DrawString(s, font, brush, new XRect(layoutRectangle), XStringFormat.Default);
    }

    public void DrawString(string s, XFont font, XBrush brush, XRect layoutRectangle)
    {
      DrawString(s, font, brush, layoutRectangle, XStringFormat.Default);
    }

    public void DrawString(string s, XFont font, XBrush brush, RectangleF layoutRectangle, XStringFormat format)
    {
      DrawString(s, font, brush, new XRect(layoutRectangle), format);
    }

    public void DrawString(string s, XFont font, XBrush brush, XRect layoutRectangle, XStringFormat format)
    {
      if (s == null)
        throw new ArgumentNullException("s");
      if (font == null)
        throw new ArgumentNullException("font");
      if (brush == null)
        throw new ArgumentNullException("brush");

      if (format.LineAlignment == XLineAlignment.BaseLine && layoutRectangle.Height != 0)
        throw new InvalidOperationException("DrawString: With XLineAlignment.BaseLine the height of the layout rectangle must be 0.");

      if (s.Length == 0)
        return;

      if (format == null)
        format = XStringFormat.Default;

      if (this.drawGraphics)
      {
        RectangleF rect = layoutRectangle.ToRectangleF();
        if (format.LineAlignment == XLineAlignment.BaseLine)
        {
          // HACK optimze
          double lineSpace = font.GetHeight(this);
          int cellSpace = font.FontFamily.GetLineSpacing(font.Style);
          int cellAscent = font.FontFamily.GetCellAscent(font.Style);
          int cellDescent = font.FontFamily.GetCellDescent(font.Style);
          double cyAscent = lineSpace * cellAscent / cellSpace;
          cyAscent = lineSpace * font.cellAscent / font.cellSpace;
          rect.Offset(0, (float)-cyAscent);
        }
        this.gfx.DrawString(s, font.RealizeGdiFont(), brush.RealizeGdiBrush(), rect, 
          format != null ? format.RealizeGdiStringFormat() : null);
      }

      if (this.renderer != null)
        this.renderer.DrawString(s, font, brush, layoutRectangle, format);
    }

    // ----- MeasureString ------------------------------------------------------------------------

    public XSize MeasureString(string text, XFont font)
    {
      if (text == null)
        throw new ArgumentNullException("text");
      if (font == null)
        throw new ArgumentNullException("font");

      // TODO: Here comes a lot of code in the future: kerning etc...
      return XSize.FromSizeF(this.gfx.MeasureString(text, font.RealizeGdiFont(), new PointF(0, 0), 
        XStringFormat.Default.RealizeGdiStringFormat()));
    }

    //public SizeF MeasureString(string text, XFont font, SizeF layoutArea);
    //public SizeF MeasureString(string text, XFont font, int width);
    //public SizeF MeasureString(string text, XFont font, PointF origin, XStringFormat stringFormat);
    //public SizeF MeasureString(string text, XFont font, SizeF layoutArea, XStringFormat stringFormat);
    //public SizeF MeasureString(string text, XFont font, int width, XStringFormat format);
    //public SizeF MeasureString(string text, XFont font, SizeF layoutArea, XStringFormat stringFormat, out int charactersFitted, out int linesFilled);

    // ----- DrawImage ----------------------------------------------------------------------------

    public void DrawImage(XImage image, Point point)
    {
      DrawImage(image, (double)point.X, (double)point.Y);
    }

    public void DrawImage(XImage image, PointF point)
    {
      DrawImage(image, point.X, point.Y);
    }

    public void DrawImage(XImage image, XPoint point)
    {
      DrawImage(image, point.X, point.Y);
    }

    //TODO trapezoid transformation
    ////public void DrawImage(XImage image, Point[] destPoints);
    ////public void DrawImage(XImage image, PointF[] destPoints);
    ////public void DrawImage(XImage image, XPoint[] destPoints);

    public void DrawImage(XImage image, int x, int y)
    {
      DrawImage(image, (double)x, (double)y);
    }

    public void DrawImage(XImage image, double x, double y)
    {
      if (image == null)
        throw new ArgumentNullException("image");

      if (this.drawGraphics)
      {
        if (image.image != null)
          this.gfx.DrawImage(image.image, (float)x, (float)y);
        else
        {
          RectangleF rect = new RectangleF((float)x, (float)y, image.Width, image.Height);
          this.gfx.DrawRectangle(Pens.Red, (float)x, (float)y, image.Width, image.Height);
          this.gfx.DrawLine(Pens.Red, (float)x, (float)y, (float)(x + image.Width), (float)(y + image.Height));
          this.gfx.DrawLine(Pens.Red, (float)(x + image.Width), (float)y, (float)x, (float)(y + image.Height));
        }
      }

      if (this.renderer != null)
        //this.renderer.DrawImage(image, x, y, image.image.Width, image.image.Height);
        this.renderer.DrawImage(image, x, y, 
          image.Width * 72 / image.HorizontalResolution,
          image.Height * 72 / image.HorizontalResolution);
    }

    public void DrawImage(XImage image, Rectangle rect)
    {
      DrawImage(image, (double)rect.X, (double)rect.Y, (double)rect.Width, (double)rect.Height);
    }
    
    public void DrawImage(XImage image, RectangleF rect)
    {
      DrawImage(image, rect.X, rect.Y, rect.Width, rect.Height);
    }
    
    public void DrawImage(XImage image, XRect rect)
    {
      DrawImage(image, rect.X, rect.Y, rect.Width, rect.Height);
    }
    
    public void DrawImage(XImage image, int x, int y, int width, int height)
    {
      DrawImage(image, (double)x, (double)y, (double)width, (double)height);
    }
    
    public void DrawImage(XImage image, double x, double y, double width, double height)
    {
      if (image == null)
        throw new ArgumentNullException("image");

      if (this.drawGraphics)
      {
        if (image.image != null)
          this.gfx.DrawImage(image.image, (float)x, (float)y, (float)width, (float)height);
        else
        {
          RectangleF rect = new RectangleF((float)x, (float)y, (float)width, (float)height);
          this.gfx.DrawRectangle(Pens.Red, (float)x, (float)y, (float)width, (float)height);
          this.gfx.DrawRectangle(Pens.Red, (float)x, (float)y, image.Width, image.Height);
          this.gfx.DrawLine(Pens.Red, (float)x, (float)y, (float)(x + image.Width), (float)(y + image.Height));
          this.gfx.DrawLine(Pens.Red, (float)(x + image.Width), (float)y, (float)x, (float)(y + image.Height));
        }
      }

      if (this.renderer != null)
        this.renderer.DrawImage(image, x, y, width, height);
    }


    //TODO trapezoid transformation
    //public void DrawImage(XImage image, Point[] destPoints, Rectangle srcRect, GraphicsUnit srcUnit);
    //public void DrawImage(XImage image, PointF[] destPoints, RectangleF srcRect, GraphicsUnit srcUnit);
    //public void DrawImage(XImage image, XPoint[] destPoints, XRect srcRect, GraphicsUnit srcUnit);


    // TODO: calculate destination size
    //public void DrawImage(XImage image, int x, int y, Rectangle srcRect, XGraphicsUnit srcUnit)
    //public void DrawImage(XImage image, double x, double y, RectangleF srcRect, XGraphicsUnit srcUnit)
    //public void DrawImage(XImage image, double x, double y, XRect srcRect, XGraphicsUnit srcUnit)

    public void DrawImage(XImage image, Rectangle destRect, Rectangle srcRect, XGraphicsUnit srcUnit)
    {
      XRect destRectX = new XRect(destRect.X, destRect.Y, destRect.Width, destRect.Height);
      XRect srcRectX  = new XRect(srcRect.X, srcRect.Y, srcRect.Width, srcRect.Height);
      DrawImage(image, destRectX, srcRectX, srcUnit);
    }

    public void DrawImage(XImage image, RectangleF destRect, RectangleF srcRect, XGraphicsUnit srcUnit)
    {
      XRect destRectX = new XRect(destRect.X, destRect.Y, destRect.Width, destRect.Height);
      XRect srcRectX  = new XRect(srcRect.X, srcRect.Y, srcRect.Width, srcRect.Height);
      DrawImage(image, destRectX, srcRectX, srcUnit);
    }

    public void DrawImage(XImage image, XRect destRect, XRect srcRect, XGraphicsUnit srcUnit)
    {
      if (image == null)
        throw new ArgumentNullException("image");

      if (this.drawGraphics)
      {
        if (image.image != null)
        {
          RectangleF destRectF = new RectangleF((float)destRect.X, (float)destRect.Y, (float)destRect.Width, (float)destRect.Height);
          RectangleF srcRectF  = new RectangleF((float)srcRect.X, (float)srcRect.Y, (float)srcRect.Width, (float)srcRect.Height);
          this.gfx.DrawImage(image.image, destRectF, srcRectF, GraphicsUnit.Pixel);
        }
        else
        {
          this.gfx.DrawRectangle(Pens.Red, (float)destRect.X, (float)destRect.Y, 
            (float)destRect.Width, (float)destRect.Height);
        }
      }

      if (this.renderer != null)
        this.renderer.DrawImage(image, destRect, srcRect, srcUnit);
    }

    //TODO?
    //public void DrawImage(XImage image, Rectangle destRect, int srcX, int srcY, int srcWidth, int srcHeight, GraphicsUnit srcUnit);
    //public void DrawImage(XImage image, Rectangle destRect, double srcX, double srcY, double srcWidth, double srcHeight, GraphicsUnit srcUnit);
    //public void DrawImage(XImage image, Rectangle destRect, double srcX, double srcY, double srcWidth, double srcHeight, GraphicsUnit srcUnit);

    // ----- DrawGrit -----------------------------------------------------------------------------

    [Conditional("DEBUG")]
    public void DrawGridlines(XPoint origin, XPen majorpen, double majordelta, XPen minorpen, double minordelta)
    {
      RectangleF box = new RectangleF(0, 0, 600, 850);
      DrawGridline(origin, minorpen, minordelta, box);
      DrawGridline(origin, majorpen, majordelta, box);
      /*
            float xmin = -10000f, ymin = -10000f, xmax = 10000f, ymax = 10000f;
            float x, y;
            x = origin.X;
            while (x < xmax)
            {
              DrawLine(majorpen, x, ymin, x, ymax);
              x += majordelta;
            }
            x = origin.X - majordelta;
            while (x > xmin)
            {
              DrawLine(majorpen, x, ymin, x, ymax);
              x -= majordelta;
            }
            y = origin.Y;
            while (y < ymax)
            {
              DrawLine(majorpen, xmin, y, xmax, y);
              y += majordelta;
            }
            y = origin.Y - majordelta;
            while (y > ymin)
            {
              DrawLine(majorpen, xmin, y, xmax, y);
              y -= majordelta;
            }
       */
    }

    [Conditional("DEBUG")]
    void DrawGridline(XPoint origin, XPen pen, double delta, XRect box)
    {
      double xmin = box.X, ymin = box.Y, xmax = box.X + box.Width, ymax = box.Y + box.Height;
      double x, y;
      y = origin.Y;
      while (y < ymax)
      {
        DrawLine(pen, xmin, y, xmax, y);
        y += delta;
      }
      y = origin.Y - delta;
      while (y > ymin)
      {
        DrawLine(pen, xmin, y, xmax, y);
        y -= delta;
      }
      x = origin.X;
      while (x < xmax)
      {
        DrawLine(pen, x, ymin, x, ymax);
        x += delta;
      }
      x = origin.X - delta;
      while (x > xmin)
      {
        DrawLine(pen, x, ymin, x, ymax);
        x -= delta;
      }
    }
    #endregion

    // --------------------------------------------------------------------------------------------

    #region Save and Restore

    // TODO: Needs reimplementation

    public XGraphicsState Save()
    {
      XGraphicsState state = new XGraphicsState(this.gfx.Save());
      state.Transform = this.transform;

      if (this.renderer != null)
        this.renderer.Save(state);

      return state;
    }

    public void Restore(XGraphicsState state)
    {
      if (state == null)
        throw new ArgumentNullException("state");

      this.gfx.Restore(state.state);
      this.transform = state.Transform;

      if (this.renderer != null)
        this.renderer.Restore(state);
    }

    public XGraphicsContainer BeginContainer()
    {
      return BeginContainer(new RectangleF(0, 0, 1, 1), new RectangleF(0, 0, 1, 1), XGraphicsUnit.Point);
    }

    public XGraphicsContainer BeginContainer(Rectangle dstrect, Rectangle srcrect, XGraphicsUnit unit)
    {
      return BeginContainer(new RectangleF(0, 0, 1, 1), new RectangleF(0, 0, 1, 1), unit);
    }

    public XGraphicsContainer BeginContainer(RectangleF dstrect, RectangleF srcrect, XGraphicsUnit unit)
    {
      // TODO: unit
#if true
      XGraphicsContainer container = new XGraphicsContainer(this.gfx.Save());
      container.Transform = this.transform;

      if (this.renderer != null)
        this.renderer.BeginContainer(container, dstrect, srcrect, unit);

      Matrix matrix = new Matrix();
      matrix.Translate(srcrect.X, srcrect.Y);
      matrix.Scale(dstrect.Width / srcrect.Width, dstrect.Height / srcrect.Height);
      matrix.Translate(dstrect.X, dstrect.Y);
      Transform = matrix;

      return container;
#else
      XGraphicsContainer container = new XGraphicsContainer(this.gfx.BeginContainer(dstrect, srcrect, this.gfx.PageUnit));
      container.Transform = this.transform;

      if (this.renderer != null)
        this.renderer.BeginContainer(container, dstrect, srcrect, unit);

      return container;
#endif
    }

    public void EndContainer(XGraphicsContainer container)
    {
      if (container == null)
        throw new ArgumentNullException("container");
#if true
#else
      this.gfx.EndContainer(container.container);
      this.transform = container.Transform;

      if (this.renderer != null)
        this.renderer.EndContainer(container);
#endif
    }

    #endregion

    // --------------------------------------------------------------------------------------------

    #region Properties

    public XSmoothingMode SmoothingMode 
    {
      get { return (XSmoothingMode)this.gfx.SmoothingMode; }
      set { this.gfx.SmoothingMode = (SmoothingMode)value; }
    }

//public Region Clip { get; set; }
//public RectangleF ClipBounds { get; }
//public CompositingMode CompositingMode { get; set; }
//public CompositingQuality CompositingQuality { get; set; }
//public float DpiX { get; }
//public float DpiY { get; }
//public InterpolationMode InterpolationMode { get; set; }
//public bool IsClipEmpty { get; }
//public bool IsVisibleClipEmpty { get; }
//public float PageScale { get; set; }
//public GraphicsUnit PageUnit { get; set; }
//public PixelOffsetMode PixelOffsetMode { get; set; }
//public Point RenderingOrigin { get; set; }
//public SmoothingMode SmoothingMode { get; set; }
//public int TextContrast { get; set; }
//public TextRenderingHint TextRenderingHint { get; set; }
//public Matrix Transform { get; set; }
//public RectangleF VisibleClipBounds { get; }

    #endregion

    // --------------------------------------------------------------------------------------------

    #region Transformation

    /// <summary>
    /// Applies the specified translation operation to the transformation matrix of this object by 
    /// prepending it to the object's transformation matrix.
    /// </summary>
    public void TranslateTransform(double dx, double dy)
    {
      TranslateTransform(dx, dy, XMatrixOrder.Prepend);
    }

    /// <summary>
    /// Applies the specified translation operation to the transformation matrix of this object
    /// in the specified order.
    /// </summary>
    public void TranslateTransform(double dx, double dy, XMatrixOrder order)
    {
      XMatrix matrix = this.transform;
      matrix.Translate(dx, dy, order);
      Transform = matrix;
    }

    /// <summary>
    /// Applies the specified scaling operation to the transformation matrix of this object by 
    /// prepending it to the object's transformation matrix.
    /// </summary>
    public void ScaleTransform(double scaleX, double scaleY)
    {
      ScaleTransform(scaleX, scaleY, XMatrixOrder.Prepend);
    }

    /// <summary>
    /// Applies the specified scaling operation to the transformation matrix of this object
    /// in the specified order.
    /// </summary>
    public void ScaleTransform(double scaleX, double scaleY, XMatrixOrder order)
    {
      XMatrix matrix = this.transform;
      matrix.Scale(scaleX, scaleY, order);
      Transform = matrix;
    }

    /// <summary>
    /// Applies the specified scaling operation to the transformation matrix of this object by 
    /// prepending it to the object's transformation matrix.
    /// </summary>
    public void ScaleTransform(double scaleXY)
    {
      ScaleTransform(scaleXY, scaleXY, XMatrixOrder.Prepend);
    }

    /// <summary>
    /// Applies the specified scaling operation to the transformation matrix of this object
    /// in the specified order.
    /// </summary>
    public void ScaleTransform(double scaleXY, XMatrixOrder order)
    {
      XMatrix matrix = this.transform;
      matrix.Scale(scaleXY, scaleXY, order);
      Transform = matrix;
    }

    /// <summary>
    /// Applies the specified rotaion operation to the transformation matrix of this object by 
    /// prepending it to the object's transformation matrix.
    /// </summary>
    public void RotateTransform(double angle)
    {
      RotateTransform(angle, XMatrixOrder.Prepend);
    }

    /// <summary>
    /// Applies the specified rotation operation to the transformation matrix of this object
    /// in the specified order. The angle unit of measure is degree.
    /// </summary>
    public void RotateTransform(double angle, XMatrixOrder order)
    {
      XMatrix matrix = this.transform;
      matrix.Rotate(angle, order);
      Transform = matrix;
    }

    /// <summary>
    /// Multiplies the transformation matrix of this object and specified matrix.
    /// </summary>
    public void MultiplyTransform(XMatrix matrix)
    {
      MultiplyTransform(matrix, XMatrixOrder.Prepend);
    }

    /// <summary>
    /// Multiplies the transformation matrix of this object and specified matrix in the specified order.
    /// </summary>
    public void MultiplyTransform(XMatrix matrix, XMatrixOrder order)
    {
      XMatrix matrix2 = this.transform;
      matrix2.Multiply(matrix, order);
      Transform = matrix2;
    }

    /// <summary>
    /// Gets or sets the transformation matrix.
    /// </summary>
    public XMatrix Transform 
    { 
      get {return this.transform;}
      set 
      {
        if (!this.transform.Equals(value))
        {
          this.transform = value;
          Matrix matrix = (Matrix)this.defaultViewMatrix;
          matrix.Multiply(value.ToMatrix());
          this.gfx.Transform = matrix;

          if (this.renderer != null)
            this.renderer.Transform = value;
        }
      }
    }

    /// <summary>
    /// Resets the transformation matrix of this object to the identity matrix.
    /// </summary>
    public void ResetTransform()
    {
      if (!this.transform.IsIdentity)
      {
        this.transform = XMatrix.Identity;
        this.gfx.Transform = (Matrix)this.defaultViewMatrix;
        if (this.renderer != null)
          this.renderer.Transform = this.transform;
      }
    }

    /// <summary>
    /// Calculates and sets the actual tranformation matrix.
    /// </summary>
    void RealizeTransform()
    {
      float factor;
      Matrix matrix = (Matrix)this.defaultViewMatrix;
      XMatrix xmatrix = this.defaultViewMatrix;
      if (!this.pageOrigin.IsEmpty)
      {
        // TODO
        throw new NotImplementedException("pageOrigin must be (0, 0)");
      }
      switch (this.pageUnit)
      {
        case XGraphicsUnit.Inch:
          matrix.Scale(72, 72);
          xmatrix.Scale(72);
          break;

        case XGraphicsUnit.Millimeter:
          factor = (float)(72.0 / 25.4);
          matrix.Scale(factor, factor);
          xmatrix.Scale(72 / 25.4);
          break;

        case XGraphicsUnit.Centimeter:
          factor = (float)(72.0 / 2.54);
          matrix.Scale(factor, factor);
          xmatrix.Scale(72 / 2.54);
          break;
      }
      matrix.Multiply(this.transform.ToMatrix());
      xmatrix.Multiply(this.transform);
      //this.gfx.Transform = matrix;
      this.gfx.Transform = (Matrix)xmatrix;
    }

    //public void TransformPoints(CoordinateSpace destSpace, CoordinateSpace srcSpace, Point[] points)
    //{
    //}
    //
    //public void TransformPoints(CoordinateSpace destSpace, CoordinateSpace srcSpace, PointF[] points)
    //{
    //}

    #endregion

    // --------------------------------------------------------------------------------------------

    #region Clipping

    public void SetClip(Rectangle rect)
    {
      XGraphicsPath path = new XGraphicsPath();
      path.AddRectangle(rect);
      SetClip(path);
    }

    public void SetClip(RectangleF rect)
    {
      XGraphicsPath path = new XGraphicsPath();
      path.AddRectangle(rect);
      SetClip(path);
    }

    public void SetClip(XRect rect)
    {
      XGraphicsPath path = new XGraphicsPath();
      path.AddRectangle(rect);
      SetClip(path);
    }

    public void ExcludeClip(Rectangle rect)
    {
      throw new NotImplementedException("ExcludeClip");
    }

    public void ExcludeClip(RectangleF rect)
    {
      throw new NotImplementedException("ExcludeClip");
    }

    public void ExcludeClip(XRect rect)
    {
      throw new NotImplementedException("ExcludeClip");
    }

    public void IntersectClip(Rectangle rect)
    {
      throw new NotImplementedException("IntersectClip");
    }

    public void IntersectClip(RectangleF rect)
    {
      throw new NotImplementedException("IntersectClip");
    }

    public void IntersectClip(XRect rect)
    {
      throw new NotImplementedException("IntersectClip");
    }

    public void SetClip(XGraphicsPath path)
    {
      if (this.drawGraphics)
        this.gfx.SetClip(path.RealizeGdiPath());

      if (this.renderer != null)
        this.renderer.SetClip(path, true);
    }

    public void ResetClip()
    {
    }

    //public void SetClip(Graphics g);
    //public void SetClip(Graphics g, CombineMode combineMode);
    //public void SetClip(GraphicsPath path, CombineMode combineMode);
    //public void SetClip(Rectangle rect, CombineMode combineMode);
    //public void SetClip(RectangleF rect, CombineMode combineMode);
    //public void SetClip(Region region, CombineMode combineMode);
    //public void IntersectClip(Region region);
    //public void ExcludeClip(Region region);

    #endregion

    // --------------------------------------------------------------------------------------------

    #region Miscellaneous
    /// <summary>
    /// Writes a comment to the output stream. Comments have no effect on the rendering of the output.
    /// They may be useful to mark a position in a content stream of a PDF document.
    /// </summary>
    public void WriteComment(string comment)
    {
      if (comment == null)
        throw new ArgumentNullException("comment");

      if (this.drawGraphics)
      {
        // TODO: Do something if metafile?
      }

      if (this.renderer != null)
        this.renderer.WriteComment(comment);
    }
    #endregion

    // --------------------------------------------------------------------------------------------

    #region Internal Helper Functions

    /// <summary>
    /// Converts a Point[] into a PointF[].
    /// </summary>
    internal static PointF[] MakePointFArray(Point[] points)
    {
      if (points == null)
        return null;

      int count = points.Length;
      PointF[] result = new PointF[count];
      for (int idx = 0; idx < count; idx++)
      {
        result[idx].X = points[idx].X;
        result[idx].Y = points[idx].Y;
      }
      return result;
    }

    /// <summary>
    /// Converts a XPoint[] into a PointF[].
    /// </summary>
    internal static PointF[] MakePointFArray(XPoint[] points)
    {
      if (points == null)
        return null;

      int count = points.Length;
      PointF[] result = new PointF[count];
      for (int idx = 0; idx < count; idx++)
      {
        result[idx].X = (float)points[idx].x;
        result[idx].Y = (float)points[idx].y;
      }
      return result;
    }

    /// <summary>
    /// Converts a Point[] into a XPoint[].
    /// </summary>
    internal static XPoint[] MakeXPointArray(Point[] points)
    {
      if (points == null)
        return null;

      int count = points.Length;
      XPoint[] result = new XPoint[count];
      for (int idx = 0; idx < count; idx++)
      {
        result[idx].x = points[idx].X;
        result[idx].y = points[idx].Y;
      }
      return result;
    }

    /// <summary>
    /// Converts a PointF[] into a XPoint[].
    /// </summary>
    internal static XPoint[] MakeXPointArray(PointF[] points)
    {
      if (points == null)
        return null;

      int count = points.Length;
      XPoint[] result = new XPoint[count];
      for (int idx = 0; idx < count; idx++)
      {
        result[idx].x = points[idx].X;
        result[idx].y = points[idx].Y;
      }
      return result;
    }

    #endregion

    ////    /// <summary>
    ////    /// Testcode
    ////    /// </summary>
    ////    public void TestXObject(PdfDocument thisDoc, PdfPage thisPage, int page, 
    ////      PdfDocument externalDoc, ImportedObjectTable impDoc)
    ////    {
    ////      PdfPage impPage = externalDoc.Pages[page];
    ////      //      impDoc.ImportPage(impPage);
    ////      PdfFormXObject form = new PdfFormXObject(thisDoc, impDoc, impPage);
    ////      thisDoc.xrefTable.Add(form);
    ////
    ////      PdfDictionary xobjects = new PdfDictionary();
    ////      xobjects.Elements["/X42"] = form.XRef;
    ////      thisPage.Resources.Elements[PdfResources.Keys.XObject] = xobjects;
    ////      ((XGraphicsPdfRenderer)this.renderer).DrawXObject("/X42");
    ////    }

    /// <summary>
    /// Always defined System.Drawing.Graphics object. Used as 'query context' for PDF pages.
    /// </summary>
    internal Graphics gfx;

    /// <summary>
    /// The transformation matrix from the XGraphics page space to the Graphics world space.
    /// (The name 'default view matrix' comes from Microsoft OS/2 Presentation Manager. I choose
    /// this name because I have no better one.)
    /// </summary>
    internal XMatrix defaultViewMatrix = XMatrix.Identity;

    /// <summary>
    /// Indicates whether to send drawing operations to this.gfx.
    /// </summary>
    bool drawGraphics;

    /// <summary>
    /// Interface to an (optional) renderer. Currently it is the PdfRenderer, if defined.
    /// </summary>
    IXGraphicsRenderer renderer;

    /// <summary>
    /// The transformation matrix from XGraphics world space to page unit space.
    /// </summary>
    XMatrix transform = XMatrix.Identity;
  }
}
