//
// PDFSharp - A library for processing PDF
//
// Authors:
//   Stefan Lange (mailto:Stefan.Lange@pdfsharp.com)
//
// Copyright (c) 2005 empira Software GmbH, Cologne (Germany)
//
// http://www.pdfsharp.com
//
// Permission is hereby granted, free of charge, to any person obtaining
// a copy of this software and associated documentation files (the
// "Software"), to deal in the Software without restriction, including
// without limitation the rights to use, copy, modify, merge, publish,
// distribute, sublicense, and/or sell copies of the Software, and to
// permit persons to whom the Software is furnished to do so, subject to
// the following conditions:
// 
// The above copyright notice and this permission notice shall be
// included in all copies or substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
// EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
// MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
// NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
// LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
// OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
// WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

// Test hack for alpha values
#define ALPHA

using System;
using System.Diagnostics;
using System.Globalization;
using System.Text;
using System.IO;
using System.Drawing;
using System.Drawing.Drawing2D;
using PdfSharp.Internal;
using PdfSharp.Pdf;
using PdfSharp.Pdf.Internal;
using PdfSharp.Pdf.Advanced;

namespace PdfSharp.Drawing.Pdf
{
  enum StreamMode
  {
    None, Text, Graphic,
  }

  [Flags]
  enum DirtyFlags
  {
    Ctm        = 0x00000001,
    ClipPath   = 0x00000002,
    LineWidth  = 0x00000010,
    LineJoin   = 0x00000020,
    MiterLimit = 0x00000040,
    StrokeFill = 0x00000070,
  }

  /// <summary>
  /// Represents a drawing surface for PdfPages.
  /// </summary>
  public class XGraphicsPdfRenderer : IXGraphicsRenderer
  {
    enum PathStart
    {
      MoveTo1st, LineTo1st, Ignore1st,
    }

    public XGraphicsPdfRenderer(PdfPage page, XGraphics gfx, XGraphicsPdfPageOptions options)
    {
      this.page = page;
      this.options = options;
      this.gfx = gfx;
      this.content = new StringBuilder();
      page.Content.pdfRenderer = this;

      this.gfxState = new PdfGraphicsState();
      this.gfxStateStack = new PdfGraphicsStateStack();
      this.gfxStateStack.Push(this.gfxState);
    }

    void InitGraphics()
    {
      if (!this.initialized)
      {
        this.initialized = true;
#if true
        // Flip page horizontaly and mirror text.
        // TODO: PageDirection downward, upward
        // TODO: Is PageOriging and PageScale (== Viewport) useful? Or just public DefaultViewMatrix (like Presentation Manager has had)
        XMatrix transform = XMatrix.Identity;
        if (page.Elements.GetInteger("/Rotate") == 90)  // HACK for InDesign flyer
        {
          //XPoint[] pts = new XPoint[1]{ new XPoint(20, 10)};
          transform.Rotate(90);
          transform.Scale(1, -1);
          //transform.TransformPoints(pts);
          //pts.GetType();
        }
        else
        {
          // Recall that the value of Height depends on Orientation.
          transform.Translate(0, page.Height);
          transform.Scale(1, -1);
        }
        double[] cm = transform.Elements;
        // There is no autoboxing for value arrays...
        // AppendFormat("{0} {1} {2} {3} {4} {5} cm\n", cm);
        AppendFormat("q {0:0.###} {1:0.###} {2:0.###} {3:0.###} {4:0.###} {5:0.###} cm ", 
          cm[0], cm[1], cm[2], cm[3], cm[4], cm[5]);
        AppendFormat("-100 Tz\n");
        this.gfxStateLevel = GraphicsStateLevel.PageSpace;
#endif

#if true_
        // Set a global alpha value for testing purposes
        PdfExtGState extGState = new PdfExtGState(this.page.Document);
        extGState.Elements[PdfExtGState.Keys.CA] = new PdfReal(0.3);
        extGState.Elements[PdfExtGState.Keys.ca] = new PdfReal(0.3);
        string gs = this.page.Resources.AddExtGState(extGState);
        AppendFormat("{0} gs\n", gs);
#endif
      }
    }
    bool initialized;

    public string GetContent()
    {
      EndPage();
      return this.content.ToString();
    }

    public XGraphicsPdfPageOptions PageOptions
    {
      get {return this.options;}
    }

    // --------------------------------------------------------------------------------------------

    #region  Drawing

    //void SetPageLayout(down, point(0, 0), unit

    // ----- Clear --------------------------------------------------------------------------------

    public void Clear(XColor color)
    {
      if (!this.gfx.Transform.IsIdentity)
        throw new NotImplementedException("Transform must be identity to clear the canvas.");

      // TODO: this is implementation is bogus. Reset transformation to identity an then fill
      XBrush brush = new XSolidBrush(color);
      DrawRectangle(null, brush, 0, 0, this.page.Width, this.page.Height);
    }

    // ----- DrawLine -----------------------------------------------------------------------------

    /// <summary>
    /// Strokes a single connection of two points.
    /// </summary>
    public void DrawLine(XPen pen, double x1, double y1, double x2, double y2)
    {
      DrawLines(pen, new XPoint[2]{new XPoint(x1, y1), new XPoint(x2, y2)});
    }

    // ----- DrawLines ----------------------------------------------------------------------------

    /// <summary>
    /// Strokes a series of connected points.
    /// </summary>
    public void DrawLines(XPen pen, XPoint[] points)
    {
      if (pen == null)
        throw new ArgumentNullException("pen");
      if (points == null)
        throw new ArgumentNullException("points");

      int count = points.Length;
      if (count == 0)
        return;

      Realize(pen);

      AppendFormat("{0:0.###} {1:0.###} m\n", points[0].X, points[0].Y);
      for (int idx = 1; idx < count; idx++)
        AppendFormat("{0:0.###} {1:0.###} l\n", points[idx].X, points[idx].Y);
      this.content.Append("S\n");
    }

    // ----- DrawBezier ---------------------------------------------------------------------------

    public void DrawBezier(XPen pen, double x1, double y1, double x2, double y2, double x3, double y3, double x4, double y4)
    {
      DrawBeziers(pen, new XPoint[4]{new XPoint(x1, y1), new XPoint(x2, y2), new XPoint(x3, y3), new XPoint(x4, y4)});
    }

    // ----- DrawBeziers --------------------------------------------------------------------------

    public void DrawBeziers(XPen pen, XPoint[] points)
    {
      if (pen == null)
        throw new ArgumentNullException("pen");
      if (points == null)
        throw new ArgumentNullException("points");

      int count = points.Length;
      if (count == 0)
        return;

      if ((points.Length - 1) % 3 != 0)
        throw new ArgumentException("Invalid number of points for bezier curve. Number must fulfil 4+3n.", "points");

      Realize(pen);

      AppendFormat("{0:0.###} {1:0.###} m\n", points[0].X, points[0].Y);
      for (int idx = 1; idx < count; idx += 3)
        AppendFormat("{0:0.###} {1:0.###} {2:0.###} {3:0.###} {4:0.###} {5:0.###} c\n",
          points[idx].X,     points[idx].Y,
          points[idx + 1].X, points[idx + 1].Y,
          points[idx + 2].X, points[idx + 2].Y);
      
      AppendStrokeFill(pen, null, XFillMode.Alternate, false);
    }

    // ----- DrawCurve ----------------------------------------------------------------------------

    public void DrawCurve(XPen pen, XPoint[] points, double tension)
    {
      if (pen == null)
        throw new ArgumentNullException("pen");
      if (points == null)
        throw new ArgumentNullException("points");

      int count = points.Length;
      if (count == 0)
        return;
      if (count < 2)
        throw new ArgumentException("Not enough points", "points");

      // See http://pubpages.unh.edu/~cs770/a5/cardinal.html
      tension /= 3;

      Realize(pen);

      AppendFormat("{0:0.###} {1:0.###} m\n", points[0].X, points[0].Y);
      if (count == 2)
      {
        // Just draws a line...
        AppendCurveSegment(points[0], points[0], points[1], points[1], tension);
      }
      else
      {
        AppendCurveSegment(points[0], points[0], points[1], points[2], tension);
        for (int idx = 1; idx < count - 2; idx++)
          AppendCurveSegment(points[idx - 1], points[idx], points[idx + 1], points[idx + 2], tension);
        AppendCurveSegment(points[count - 3], points[count - 2], points[count - 1], points[count - 1], tension);
      }
      AppendStrokeFill(pen, null, XFillMode.Alternate, false);
    }

    // ----- DrawArc ------------------------------------------------------------------------------

    public void DrawArc(XPen pen, double x, double y, double width, double height, double startAngle, double sweepAngle)
    {
      if (pen == null)
        throw new ArgumentNullException("pen");

      Realize(pen);

      AppendPartialArc(x, y, width, height, startAngle, sweepAngle, PathStart.MoveTo1st);
      AppendStrokeFill(pen, null, XFillMode.Alternate, false);
    }

    // ----- DrawRectangle ------------------------------------------------------------------------

    public void DrawRectangle(XPen pen, XBrush brush, double x, double y, double width, double height)
    {
      if (pen == null && brush == null)
        throw new ArgumentNullException("pen and brush");

      Realize(pen, brush);

      AppendFormat("{0:0.###} {1:0.###} {2:0.###} {3:0.###} re\n",
        x, y, width, height);
      if (pen != null && brush != null)
        this.content.Append("B\n");
      else if (pen != null)
        this.content.Append("S\n");
      else
        this.content.Append("f\n");
    }

    // ----- DrawRectangles -----------------------------------------------------------------------

    public void DrawRectangles(XPen pen, XBrush brush, XRect[] rects)
    {
      // TODO: check whether GDI+ does more than drawing rectangles in a loop
      int count = rects.Length;
      for (int idx = 0; idx < count; idx++)
      {
        XRect rect = rects[idx];
        DrawRectangle(pen, brush, rect.X, rect.Y, rect.Width, rect.Height);
      }
    }

    // ----- DrawRoundedRectangle -----------------------------------------------------------------

    public void DrawRoundedRectangle(XPen pen, XBrush brush, double x, double y, double width, double height, double ellipseWidth, double ellipseHeight)
    {
      XGraphicsPath path = new XGraphicsPath();
      path.AddRoundedRectangle(x, y, width, height, ellipseWidth, ellipseHeight);
      this.DrawPath(pen, brush, path);
    }

    // ----- DrawEllipse --------------------------------------------------------------------------

    public void DrawEllipse(XPen pen, XBrush brush, double x, double y, double width, double height)
    {
      Realize(pen, brush);

      // Useful information are here http://home.t-online.de/home/Robert.Rossmair/ellipse.htm
      // or here http://www.whizkidtech.redprince.net/bezier/circle/
      // Deeper but more difficult: http://www.tinaja.com/cubic01.asp
      // Petzold: 4/3 * tan(α / 4)
      const double κ = 0.552284749f;  // := 4/3 * (1 - cos(-π/4)) / sin(π/4)) <=> 4/3 * sqrt(2) - 1
      XRect rect = new XRect(x, y, width, height);
      double δx = rect.Width / 2;
      double δy = rect.Height / 2;
      double fx = δx * κ;
      double fy = δy * κ;
      double x0 = rect.X + δx;
      double y0 = rect.Y + δy;

      AppendFormat("{0:0.###} {1:0.###} m\n", x0 + δx, y0);
      AppendFormat("{0:0.###} {1:0.###} {2:0.###} {3:0.###} {4:0.###} {5:0.###} c\n",
        x0 + δx, y0 + fy, x0 + fx, y0 + δy, x0, y0 + δy);
      AppendFormat("{0:0.###} {1:0.###} {2:0.###} {3:0.###} {4:0.###} {5:0.###} c\n",
        x0 - fx, y0 + δy, x0 - δx, y0 + fy, x0 - δx, y0);
      AppendFormat("{0:0.###} {1:0.###} {2:0.###} {3:0.###} {4:0.###} {5:0.###} c\n",
        x0 - δx, y0 - fy, x0 - fx, y0 - δy, x0, y0 - δy);
      AppendFormat("{0:0.###} {1:0.###} {2:0.###} {3:0.###} {4:0.###} {5:0.###} c\n",
        x0 + fx, y0 - δy, x0 + δx, y0 - fy, x0 + δx, y0);
      AppendStrokeFill(pen, brush, XFillMode.Winding, true);
    }

    // ----- DrawPolygon --------------------------------------------------------------------------

    public void DrawPolygon(XPen pen, XBrush brush, XPoint[] points, XFillMode fillmode)
    {
      Realize(pen, brush);

      // TODO: what about count == 1?
      int count = points.Length;
      if (count == 0)
        return;

      AppendFormat("{0:0.###} {1:0.###} m\n", points[0].X, points[0].Y);
      for (int idx = 1; idx < count; idx++)
        AppendFormat("{0:0.###} {1:0.###} l\n", points[idx].X, points[idx].Y);

      AppendStrokeFill(pen, brush, fillmode, true);
    }

    // ----- DrawPie ------------------------------------------------------------------------------

    public void DrawPie(XPen pen, XBrush brush, double x, double y, double width, double height, 
      double startAngle, double sweepAngle)
    {
      Realize(pen, brush);

      AppendFormat("{0:0.###} {1:0.###} m\n", x + width / 2, y + height / 2);
      AppendPartialArc(x, y, width, height, startAngle, sweepAngle, PathStart.LineTo1st);
      AppendStrokeFill(pen, brush, XFillMode.Alternate, true);
    }

    // ----- DrawClosedCurve ----------------------------------------------------------------------

    public void DrawClosedCurve(XPen pen, XBrush brush, XPoint[] points, double tension, XFillMode fillmode)
    {
      int count = points.Length;
      if (count == 0)
        return;
      if (count < 2)
        throw new ArgumentException("Not enough points", "points");

      // Simply tried out. Not proofed why it is correct.
      tension /= 3;

      Realize(pen, brush);

      AppendFormat("{0:0.###} {1:0.###} m\n", points[0].X, points[0].Y);
      if (count == 2)
      {
        // Just draws a line...
        AppendCurveSegment(points[0], points[0], points[1], points[1], tension);
      }
      else
      {
        AppendCurveSegment(points[count - 1], points[0], points[1], points[2], tension);
        for (int idx = 1; idx < count - 2; idx++)
          AppendCurveSegment(points[idx - 1], points[idx], points[idx + 1], points[idx + 2], tension);
        AppendCurveSegment(points[count - 3], points[count - 2], points[count - 1], points[0], tension);
        AppendCurveSegment(points[count - 2], points[count - 1], points[0], points[1], tension);
      }
      AppendStrokeFill(pen, brush, fillmode, true);
    }

    // ----- DrawPath -----------------------------------------------------------------------------

    public void DrawPath(XPen pen, XBrush brush, XGraphicsPath path)
    {
      if (pen == null && brush == null)
        throw new ArgumentNullException("pen and brush");

      Realize(pen, brush);
      AppendPath(path.gdipPath);
      AppendStrokeFill(pen, brush, path.FillMode, false);
    }

    // ----- DrawString ---------------------------------------------------------------------------

    public void DrawString(string s, XFont font, XBrush brush, XRect rect, XStringFormat format)
    {
      Realize(font, brush, 0);

      double x = rect.X;
      double y = rect.Y;

      // HACK optimze
      double lineSpace = font.GetHeight(this.gfx);
      int cellSpace = font.FontFamily.GetLineSpacing(font.Style);
      int cellAscent = font.FontFamily.GetCellAscent(font.Style);
      int cellDescent = font.FontFamily.GetCellDescent(font.Style);
      double cyAscent = lineSpace * cellAscent / cellSpace;
      double cyDescent = lineSpace * cellDescent / cellSpace;
      cyAscent = lineSpace * font.cellAscent / font.cellSpace;
      cyDescent = lineSpace * font.cellDescent / font.cellSpace;
      double width = this.gfx.MeasureString(s, font).Width;

      switch (format.Alignment)
      {
        case XStringAlignment.Near:
          // nothing to do
          break;

        case XStringAlignment.Center:
          x += (rect.Width - width) / 2;
          break;

        case XStringAlignment.Far:
          x += rect.Width - width;
          break;
      }

      switch (format.LineAlignment)
      {
        case XLineAlignment.Near:
          y += cyAscent;
          break;

        case XLineAlignment.Center:
          // TODO use CapHeight. PDFlib also uses 3/4 of ascent
          y += (cyAscent * 3 / 4) / 2 + rect.Height / 2;
          break;

        case XLineAlignment.Far:
          y += -cyDescent + rect.Height;
          break;

        case XLineAlignment.BaseLine:
          // nothing to do
          break;
      }

      byte[] bytes = PdfEncoders.WinAnsiEncoding.GetBytes(s);
      AppendFormat(
      //"1 0 0 1 {0} {1} Tm\n {2} Tj\n", rect.X, rect.Y, PdfEncoders.EncodeAsLiteral(s));
        "1 0 0 1 {0:0.###} {1:0.###} Tm {2} Tj\n", x, y, //PdfEncoders.EncodeAsLiteral(s));
        PdfEncoders.ToStringLiteral(bytes, false,  null));
    }

    // ----- DrawImage ----------------------------------------------------------------------------

    //public void DrawImage(Image image, Point point);
    //public void DrawImage(Image image, PointF point);
    //public void DrawImage(Image image, Point[] destPoints);
    //public void DrawImage(Image image, PointF[] destPoints);
    //public void DrawImage(Image image, Rectangle rect);
    //public void DrawImage(Image image, RectangleF rect);
    //public void DrawImage(Image image, int x, int y);
    //public void DrawImage(Image image, float x, float y);
    //public void DrawImage(Image image, Point[] destPoints, Rectangle srcRect, GraphicsUnit srcUnit);
    //public void DrawImage(Image image, Rectangle destRect, Rectangle srcRect, GraphicsUnit srcUnit);
    //public void DrawImage(Image image, RectangleF destRect, RectangleF srcRect, GraphicsUnit srcUnit);
    //public void DrawImage(Image image, PointF[] destPoints, RectangleF srcRect, GraphicsUnit srcUnit);
    //public void DrawImage(Image image, int x, int y, Rectangle srcRect, GraphicsUnit srcUnit);
    //public void DrawImage(Image image, float x, float y, RectangleF srcRect, GraphicsUnit srcUnit);
    //public void DrawImage(Image image, Point[] destPoints, Rectangle srcRect, GraphicsUnit srcUnit, ImageAttributes imageAttr);
    //public void DrawImage(Image image, PointF[] destPoints, RectangleF srcRect, GraphicsUnit srcUnit, ImageAttributes imageAttr);
    //public void DrawImage(Image image, int x, int y, int width, int height);
    //public void DrawImage(Image image, float x, float y, float width, float height);
    //public void DrawImage(Image image, Point[] destPoints, Rectangle srcRect, GraphicsUnit srcUnit, ImageAttributes imageAttr, DrawImageAbort callback);
    //public void DrawImage(Image image, PointF[] destPoints, RectangleF srcRect, GraphicsUnit srcUnit, ImageAttributes imageAttr, DrawImageAbort callback);
    //public void DrawImage(Image image, Rectangle destRect, int srcX, int srcY, int srcWidth, int srcHeight, GraphicsUnit srcUnit);
    //public void DrawImage(Image image, Rectangle destRect, float srcX, float srcY, float srcWidth, float srcHeight, GraphicsUnit srcUnit);
    //public void DrawImage(Image image, Point[] destPoints, Rectangle srcRect, GraphicsUnit srcUnit, ImageAttributes imageAttr, DrawImageAbort callback, int callbackData);
    //public void DrawImage(Image image, PointF[] destPoints, RectangleF srcRect, GraphicsUnit srcUnit, ImageAttributes imageAttr, DrawImageAbort callback, int callbackData);
    //public void DrawImage(Image image, Rectangle destRect, int srcX, int srcY, int srcWidth, int srcHeight, GraphicsUnit srcUnit, ImageAttributes imageAttr);
    //public void DrawImage(Image image, Rectangle destRect, float srcX, float srcY, float srcWidth, float srcHeight, GraphicsUnit srcUnit, ImageAttributes imageAttrs);
    //public void DrawImage(Image image, Rectangle destRect, int srcX, int srcY, int srcWidth, int srcHeight, GraphicsUnit srcUnit, ImageAttributes imageAttr, DrawImageAbort callback);
    //public void DrawImage(Image image, Rectangle destRect, float srcX, float srcY, float srcWidth, float srcHeight, GraphicsUnit srcUnit, ImageAttributes imageAttrs, DrawImageAbort callback);
    //public void DrawImage(Image image, Rectangle destRect, int srcX, int srcY, int srcWidth, int srcHeight, GraphicsUnit srcUnit, ImageAttributes imageAttrs, DrawImageAbort callback, IntPtr callbackData);
    //public void DrawImage(Image image, Rectangle destRect, float srcX, float srcY, float srcWidth, float srcHeight, GraphicsUnit srcUnit, ImageAttributes

    public void DrawImage(XImage image, double x, double y, double width, double height)
    {
      string name = Realize(image);
      if (!(image is XPdfForm))
      {
        AppendFormat("q {2:0.###} 0 0 -{3:0.###} {0:0.###} {4:0.###} cm\n" +
          "{5} Do\nQ\n", 
          x, y, width, height, y + height, name);
      }
      else
      {
        InitGraphics();

        //this.Elements["/Matrix"] = new PdfRawItem("[0.5 0 0 -0.5 0 421]");
        PdfFormXObject pdfForm = this.page.Document.FormTable.GetForm(image as XPdfForm);

        double cx = width / image.Width;
        double cy = height / image.Height;

        if (cx != 0 && cy != 0)
        {
          string matrix = String.Format(CultureInfo.InvariantCulture, 
            "[{0:0.###} {1:0.###} {2:0.###} {3:0.###} {4:0.###} {5:0.###}]",
            cx, 0, 0, -cy, 0, height);

          XMatrix m = XMatrix.Identity;
          m.Translate(x, -(y + height));
          m.Rotate(45);
          //m.Translate(-x, -y);
          double[] elements = m.Elements;
          matrix = String.Format(CultureInfo.InvariantCulture, 
            "[{0:0.###} {1:0.###} {2:0.###} {3:0.###} {4:0.###} {5:0.###}]",
            elements[0], elements[1], elements[2], elements[3], elements[4], elements[5]);
          //pdfForm.Elements["/Matrix"] = new PdfRawItem(matrix);

          AppendFormat("q {0:0.###} {1:0.###} {2:0.###} {3:0.###} {4:0.###} {5:0.###} cm ", 
            cx, 0, 0, -cy, x, y + height);
          //if (this.page.Direction = downward ...)
          AppendFormat("100 Tz\n");
          //string name = this.page.GetFormName(image as XPdfForm);
          AppendFormat("{0} Do\nQ\n", name);
        }
      }
    }

    // TODO: incomplete - srcRect not used
    public void DrawImage(XImage image, XRect destRect, XRect srcRect, XGraphicsUnit srcUnit)
    {
      double x      = destRect.X;
      double y      = destRect.Y;
      double width  = destRect.Width;
      double height = destRect.Height;

      string name = Realize(image);
      if (!(image is XPdfForm))
      {
        AppendFormat("q {2:0.###} 0 0 -{3:0.###} {0:0.###} {4:0.###} cm\n" +
          "{5} Do\nQ\n", 
          x, y, width, height, y + height, name);
      }
      else
      {
        InitGraphics();

        //this.Elements["/Matrix"] = new PdfRawItem("[0.5 0 0 -0.5 0 421]");
        PdfFormXObject pdfForm = this.page.Document.FormTable.GetForm(image as XPdfForm);

        double cx = width / image.Width;
        double cy = height / image.Height;

        string matrix = String.Format(CultureInfo.InvariantCulture, 
          "[{0:0.###} {1:0.###} {2:0.###} {3:0.###} {4:0.###} {5:0.###}]",
          cx, 0, 0, -cy, 0, height);

        XMatrix m = XMatrix.Identity;
        m.Translate(x, -(y + height));
        m.Rotate(45);
        //m.Translate(-x, -y);
        double[] elements = m.Elements;
        matrix = String.Format(CultureInfo.InvariantCulture, 
          "[{0:0.###} {1:0.###} {2:0.###} {3:0.###} {4:0.###} {5:0.###}]",
          elements[0], elements[1], elements[2], elements[3], elements[4], elements[5]);
        //pdfForm.Elements["/Matrix"] = new PdfRawItem(matrix);

        AppendFormat("q {0:0.###} {1:0.###} {2:0.###} {3:0.###} {4:0.###} {5:0.###} cm ", 
          cx, 0, 0, -cy, x, y + height);
        //if (this.page.Direction = downward ...)
        AppendFormat("100 Tz\n");
        //string name = this.page.GetFormName(image as XPdfForm);
        AppendFormat("{0} Do\nQ\n", name);
      }
    }

    #endregion

//    // HACK
//    internal void DrawXObject(string id)
//    {
//      AppendFormat("{0} Do\n", id);
//    }

    // --------------------------------------------------------------------------------------------

    #region Save and Restore

    /// <summary>
    /// Clones the current graphics state and push it on a stack.
    /// </summary>
    public void Save(XGraphicsState state)
    {
      PdfGraphicsState pdfstate = (PdfGraphicsState)this.gfxState.Clone();
      //this.gfxStateStack.Push(pdfstate);
      state.Handle = this.gfxState.Clone();
      //this.gfxState = pdfstate;
#if MIGRADOC__
      BeginGraphic();
      AppendFormat("q\n % Hack for MigraDoc");
#endif
    }

    public void Restore(XGraphicsState state)
    {
      PdfGraphicsState pdfstate = (PdfGraphicsState)state.Handle;
      this.dirty |= DirtyFlags.Ctm;
      this.gfxState = pdfstate;
      //this.gfxState = this.gfxStateStack.Restore((PdfGraphicsState)state.Handle);
#if MIGRADOC__
      BeginGraphic();
      AppendFormat("Q\n % Hack for MigraDoc");
#endif
    }

    public void BeginContainer(XGraphicsContainer container, XRect dstrect, XRect srcrect, XGraphicsUnit unit)
    {
//      PdfGraphicsState pdfstate = (PdfGraphicsState)this.gfxState.Clone();
//      this.gfxStateStack.Push(pdfstate);
//      container.Handle = pdfstate;
      container.Handle = this.gfxState.Clone();

//      Matrix matrix = new Matrix();
//      matrix.Translate(srcrect.X, srcrect.Y);
//      matrix.Scale(dstrect.Width / srcrect.Width, dstrect.Height / srcrect.Height);
//      matrix.Translate(dstrect.X, dstrect.Y);
//      Transform = matrix;
    }

    public void EndContainer(XGraphicsContainer container)
    {
      PdfGraphicsState pdfstate = (PdfGraphicsState)container.Handle;
      this.dirty |= DirtyFlags.Ctm;
      this.gfxState = pdfstate;
    }

    #endregion

    // --------------------------------------------------------------------------------------------

    #region Transformation

    public void SetPageTransform(XPageDirection direction, XPoint origion, XGraphicsUnit unit)
    {
    }

    public XMatrix Transform
    {
      get {return this.gfxState.Ctm;}
      set 
      {
        this.gfxState.Ctm = value;
        this.dirty |= DirtyFlags.Ctm;
      }
    }

    #endregion

    // --------------------------------------------------------------------------------------------

    #region Clipping

    public void SetClip(XGraphicsPath path, bool intersect)
    {
      this.dirty |= DirtyFlags.ClipPath;
      if (path == null)
      {
        this.gfxState.ClipPath = null;
      }
      else
      {
        // TODO
        if (intersect)
        {
        }
        else
        {
        }
        this.gfxState.ClipPath = path.Clone();
      }
      RealizeClipPath();
    }

    #endregion

    // --------------------------------------------------------------------------------------------

    #region Miscellaneous

    /// <summary>
    /// Writes a comment to the PDF stream. May be useful for debugging purposes.
    /// </summary>
    public void WriteComment(string comment)
    {
      comment.Replace("\n", "\n% ");
      // TODO: Some more checks neccessary?
      Append("% " + comment + "\n");
    }

    #endregion

    // --------------------------------------------------------------------------------------------

    void BeginGraphic()
    {
      if (this.streamMode != StreamMode.Graphic)
      {
        if (this.streamMode == StreamMode.Text)
          this.content.Append("ET\n");

        //this.inPath = false;
        this.streamMode = StreamMode.Graphic;
      }
      //this.inPath.GetType();
    }
    
    void EndPage()
    {
      if (this.streamMode == StreamMode.Text)
      {
        this.content.Append("ET\n");
        this.streamMode = StreamMode.None;
      }

      // HACK balance 
      switch (this.gfxStateLevel)
      {
        case GraphicsStateLevel.Initial:
          break;

        case GraphicsStateLevel.PageSpace:
          this.content.Append("Q ");
          break;

        case GraphicsStateLevel.WorldSpace:
          this.content.Append("Q Q ");
          break;
      }
    }

    StreamMode streamMode;
    //bool inPath;

    #region Append to PDF stream

    /// <summary>
    /// Appends one or up to five Bézier curves that interpolate the arc.
    /// </summary>
    void AppendPartialArc(double x, double y, double width, double height, double startAngle, double sweepAngle,
      PathStart pathStart)
    {
      // Normalize the angles
      double α = startAngle;
      if (α < 0)
        α =  α + (1 + Math.Floor((Math.Abs(α) / 360))) * 360;
      else if (α > 360)
        α =  α - Math.Floor(α / 360) * 360;
      Debug.Assert(α >= 0 && α <= 360);

      double β = sweepAngle;
      if (β < -360)
        β = -360;
      else if (β > 360)
        β = 360;

      if (α == 0 && β < 0)
        α = 360;
      else if (α == 360 && β > 0)
        α = 0;

      // Is it possible that the arc is small starts and ends in same quadrant?
      bool smallAngle = Math.Abs(β) <= 90;

      β = α + β;
      if (β < 0)
        β = β + (1 + Math.Floor((Math.Abs(β) / 360))) * 360;

      bool clockwise = sweepAngle > 0;
      int startQuadrant = Quatrant(α, true, clockwise);
      int endQuadrant   = Quatrant(β, false, clockwise);

      if (startQuadrant == endQuadrant && smallAngle)
        AppendPartialArcQuadrant(x, y, width, height, α, β, pathStart);
      else
      {
        int currentQuadrant = startQuadrant; 
        bool firstLoop = true;
        do
        {
          if (currentQuadrant == startQuadrant && firstLoop)
          {
            double ξ = currentQuadrant * 90 + (clockwise ? 90 : 0);
            AppendPartialArcQuadrant(x, y, width, height, α, ξ, pathStart);
          }
          else if (currentQuadrant == endQuadrant)
          {
            double ξ = currentQuadrant * 90 + (clockwise ? 0 : 90);
            AppendPartialArcQuadrant(x, y, width, height, ξ, β, PathStart.Ignore1st);
          }
          else
          {
            double ξ1 = currentQuadrant * 90 + (clockwise ? 0 : 90);
            double ξ2 = currentQuadrant * 90 + (clockwise ? 90 : 0);
            AppendPartialArcQuadrant(x, y, width, height, ξ1, ξ2, PathStart.Ignore1st);
          }

          // Don´t stop immediately if arc is greater than 270 degrees
          if (currentQuadrant == endQuadrant && smallAngle)
            break;
          else
            smallAngle = true;

          if (clockwise)
            currentQuadrant = currentQuadrant == 3 ? 0 : currentQuadrant + 1;
          else
            currentQuadrant = currentQuadrant == 0 ? 3 : currentQuadrant - 1;

          firstLoop = false;
        } while (true);
      }
    }

    /// <summary>
    /// Gets the quadrant (0 through 3) of the specified angle. If the angle lies on an edge
    /// (0, 90, 180, etc.) the result depends on the details how the angle is used.
    /// </summary>
    int Quatrant(double φ, bool start, bool clockwise)
    {
      Debug.Assert(φ >= 0);
      if (φ > 360)
        φ = φ - Math.Floor(φ / 360) * 360;

      int quadrant = (int)(φ / 90);
      if (quadrant * 90 == φ)
      {
        if ((start && !clockwise) || (!start && clockwise))
          quadrant = quadrant == 0 ? 3 : quadrant - 1;
      }
      else
        quadrant = clockwise ? ((int)Math.Floor(φ / 90)) % 4 : (int)Math.Floor(φ / 90);
      return quadrant;
    }

    /// <summary>
    /// Appends a Bézier curve for an arc within a quadrant.
    /// </summary>
    void AppendPartialArcQuadrant(double x, double y, double width, double height, double α, double β, PathStart pathStart)
    {
      Debug.Assert(α >= 0 && α <= 360);
      Debug.Assert(β >= 0);
      if (β > 360)
        β = β - Math.Floor(β / 360) * 360;
      Debug.Assert(Math.Abs(α - β) <= 90);

      // Scanling factor
      double δx = width / 2;
      double δy = height / 2;

      // Center of ellipse
      double x0 = x + δx;
      double y0 = y + δy;

      // We have the following quarters:
      //     |
      //   2 | 3
      // ----+-----
      //   1 | 0
      //     |
      // If the angles lie in quarter 2 or 3, their values are subtracted by 180 and the
      // resulting curve is reflected at the center. This algorythm works as expected (simply tried out).
      // There may be a mathematical more elegant solution...
      bool reflect = false;
      if (α >= 180 && β >= 180)
      {
        α -= 180;
        β -= 180;
        reflect = true;
      }

      double cosα, cosβ, sinα, sinβ;
      if (width == height)
      {
        // Circular arc needs no correction.
        α = α * Calc.Deg2Rad;
        β = β * Calc.Deg2Rad;
      }
      else
      {
        // Elliptic arc needs the angles to be adjusted such that the scaling transformation is compensated.
        α = α * Calc.Deg2Rad;
        sinα = Math.Sin(α);
        if (Math.Abs(sinα) > 1E-10)
          α = Calc.πHalf - Math.Atan(δy * Math.Cos(α) / (δx * sinα));
        β = β * Calc.Deg2Rad;
        sinβ = Math.Sin(β);
        if (Math.Abs(sinβ) > 1E-10)
          β = Calc.πHalf - Math.Atan(δy * Math.Cos(β) / (δx * sinβ));
      }

      double κ = 4 * (1 - Math.Cos((α - β) / 2)) / (3 * Math.Sin((β - α) / 2));
      sinα = Math.Sin(α);
      cosα = Math.Cos(α);
      sinβ = Math.Sin(β);
      cosβ = Math.Cos(β);

      if (!reflect)
      {
        // Calculation for quarter 0 and 1
        switch (pathStart)
        {
          case PathStart.MoveTo1st:
            AppendFormat("{0:0.###} {1:0.###} m\n", x0 + δx * cosα, y0 + δy * sinα);
            break;

          case PathStart.LineTo1st:
            AppendFormat("{0:0.###} {1:0.###} l\n", x0 + δx * cosα, y0 + δy * sinα);
            break;

          case PathStart.Ignore1st:
            break;
        }
        AppendFormat("{0:0.###} {1:0.###} {2:0.###} {3:0.###} {4:0.###} {5:0.###} c\n",
          x0 + δx * (cosα - κ * sinα), y0 + δy * (sinα + κ * cosα),
          x0 + δx * (cosβ + κ * sinβ), y0 + δy * (sinβ - κ * cosβ),
          x0 + δx * cosβ,              y0 + δy * sinβ);
      }
      else
      {
        // Calculation for quarter 2 and 3
        switch (pathStart)
        {
          case PathStart.MoveTo1st:
            AppendFormat("{0:0.###} {1:0.###} m\n", x0 - δx * cosα, y0 - δy * sinα);
            break;

          case PathStart.LineTo1st:
            AppendFormat("{0:0.###} {1:0.###} l\n", x0 - δx * cosα, y0 - δy * sinα);
            break;

          case PathStart.Ignore1st:
            break;
        }
        AppendFormat("{0:0.###} {1:0.###} {2:0.###} {3:0.###} {4:0.###} {5:0.###} c\n",
          x0 - δx * (cosα - κ * sinα), y0 - δy * (sinα + κ * cosα),
          x0 - δx * (cosβ + κ * sinβ), y0 - δy * (sinβ - κ * cosβ),
          x0 - δx * cosβ,              y0 - δy * sinβ);
      }
    }

    /// <summary>
    /// Appends a Bézier curve for a cardinal spline through pt1 and pt2.
    /// </summary>
    void AppendCurveSegment(XPoint pt0, XPoint pt1, XPoint pt2, XPoint pt3, double tension3)
    {
      AppendFormat("{0:0.###} {1:0.###} {2:0.###} {3:0.###} {4:0.###} {5:0.###} c\n",
        pt1.X + tension3 * (pt2.X - pt0.X), pt1.Y + tension3 * (pt2.Y - pt0.Y),
        pt2.X - tension3 * (pt3.X - pt1.X), pt2.Y - tension3 * (pt3.Y - pt1.Y),
        pt2.X, pt2.Y);
    }

#if out_of_order
    /// <summary>
    /// Appends the content of an XGraphicsPath object.
    /// </summary>
    void AppendPath(XGraphicsPath path)
    {
      //bool closed = false;
      bool inPath = false;

      //StringBuilder pdf = new StringBuilder();

      // not correct: use inPath
      XGraphicsPathItem[] items = path.GetPathData();
      int count = items.Length;
      for (int itemidx = 0; itemidx < count; itemidx++)
      {
        XGraphicsPathItem item = items[itemidx];
        int pointCount = item.points.Length;
        switch (item.type)
        {
          case XGraphicsPathItemType.Lines:
          {
            int idx = 0;
            if (!inPath)
            {
              AppendFormat("{0} {1} m\n", item.points[0].X, item.points[0].Y);
              idx++;
              inPath = true;
            }
            for (; idx < pointCount; idx++)
              AppendFormat("{0} {1} l\n", item.points[idx].X, item.points[idx].Y);
          }
            break;

          case XGraphicsPathItemType.Beziers:
            DrawEllipse:
            {
              int idx = 0;
              if (!inPath)
              {
                AppendFormat("{0} {1} m\n", item.points[0].X, item.points[0].Y);
                idx++;
                inPath = true;
              }
              for (; idx < pointCount; idx += 3)
                AppendFormat("{0} {1} {2} {3} {4} {5} c\n",
                  item.points[idx].X, item.points[idx].Y,
                  item.points[idx + 1].X, item.points[idx + 1].Y,
                  item.points[idx + 2].X, item.points[idx + 2].Y);
            }
            break;

          case XGraphicsPathItemType.Curve:
            throw new NotImplementedException("XGraphicsPathItemType.Curve");

          case XGraphicsPathItemType.Arc:
            AppendPartialArc(item.points[0].X, item.points[0].Y, item.points[1].X, item.points[1].Y, 
              item.points[2].X, item.points[2].Y, inPath ? PathStart.LineTo1st : PathStart.MoveTo1st);
            inPath = true;
            break;

          case XGraphicsPathItemType.Rectangle:
            AppendFormat("{0} {1} {2} {3} re\n",
              item.points[0].X, item.points[0].Y, item.points[1].X, item.points[1].Y);
            break;

          case XGraphicsPathItemType.RoundedRectangle:
          {
            double x = item.points[0].X;
            double y = item.points[0].Y;
            double width = item.points[1].X;
            double height = item.points[1].Y;
            double ellipseWidth = item.points[2].X;
            double ellipseHeight = item.points[2].Y;
            PathStart pathStart = inPath ? PathStart.LineTo1st : PathStart.MoveTo1st;
            AppendPartialArc(x + width - ellipseWidth, y, ellipseWidth, ellipseHeight, -90, 90, pathStart);
            AppendPartialArc(x + width - ellipseWidth, y + height - ellipseHeight, ellipseWidth, ellipseHeight, 0, 90, PathStart.LineTo1st);
            AppendPartialArc(x, y + height - ellipseHeight, ellipseWidth, ellipseHeight, 90, 90, PathStart.LineTo1st);
            AppendPartialArc(x, y, ellipseWidth, ellipseHeight, 180, 90, PathStart.LineTo1st);
            Append("h\n");
          }
            break;

          case XGraphicsPathItemType.Ellipse:
          {
            // Source: http://home.t-online.de/home/Robert.Rossmair/ellipse.htm
            // (This is a german site but written in english)
            // 
            // Formula for circles: 4 * (sqrt(2) - 1) / 3
            // float factor = (float)((Math.Sqrt(2) - 1) * 4 / 3);
            const double factor = 0.552284749f;
            XRect rect = new XRect(item.points[0], new XSize(item.points[1]));
            double rx = rect.Width / 2;
            double ry = rect.Height / 2;
            double fx = rx * factor;
            double fy = ry * factor;
            double x  = rect.X + rx;
            double y  = rect.Y + ry;

            XPoint[] ellipse = new XPoint[13]
            {
              new XPoint(x + rx, y),
              new XPoint(x + rx, y + fy),
              new XPoint(x + fx, y + ry),
              new XPoint(x,      y + ry),
              new XPoint(x - fx, y + ry),
              new XPoint(x - rx, y + fy),
              new XPoint(x - rx, y),
              new XPoint(x - rx, y - fy),
              new XPoint(x - fx, y - ry),
              new XPoint(x,      y - ry),
              new XPoint(x + fx, y - ry),
              new XPoint(x + rx, y - fy),
              new XPoint(x + rx, y),
            };
            pointCount = ellipse.Length;
            item = new XGraphicsPathItem(XGraphicsPathItemType.Beziers, ellipse);
            // TODO close the figure...
            goto DrawEllipse;
          }

          case XGraphicsPathItemType.Polygon:
          {
            int idx = 0;
            if (!inPath)
            {
              AppendFormat("{0} {1} m\n", item.points[0].X, item.points[0].Y);
              idx++;
              inPath = true;
            }
            for (; idx < pointCount; idx++)
              AppendFormat("{0} {1} l\n", item.points[idx].X, item.points[idx].Y);
            Append("h\n");
          }
            break;

          case XGraphicsPathItemType.CloseFigure:
            Append("h\n");
            break;

          case XGraphicsPathItemType.StartFigure:
            inPath = false;
            break;
        }
      }
      //Append(pdf.ToString());
    }
#endif

    /// <summary>
    /// Appends the content of a GraphicsPath object.
    /// </summary>
    void AppendPath(GraphicsPath path)
    {
      //bool closed = false;
      bool inPath = false;
      inPath.GetType();

      //StringBuilder pdf = new StringBuilder();

      int count = path.PointCount;
      if (count == 0)
        return;
      PointF[] points = path.PathPoints;
      Byte[] types = path.PathTypes;
#if DEBUG__
      PathData pathData = path.PathData;
      pathData.GetType();
      DumpPathData(pathData);
#endif

      for (int idx = 0; idx < count; idx++)
      {
        // GDI+ documentation
        const byte PathPointTypeStart        = 0;    // move
        const byte PathPointTypeLine         = 1;    // line
        const byte PathPointTypeBezier       = 3;    // default Bezier (= cubic Bezier)
        const byte PathPointTypePathTypeMask = 0x07; // type mask (lowest 3 bits).
        //const byte PathPointTypeDashMode     = 0x10; // currently in dash mode.
        //const byte PathPointTypePathMarker   = 0x20; // a marker for the path.
        const byte PathPointTypeCloseSubpath = 0x80; // closed flag

        byte type = types[idx];
        switch (type & PathPointTypePathTypeMask)
        {
          case PathPointTypeStart:
            //PDF_moveto(pdf, points[idx].X, points[idx].Y);
            AppendFormat("{0} {1} m\n", points[idx].X, points[idx].Y);
            inPath = true;
            break;

          case PathPointTypeLine:
            //PDF_lineto(pdf, points[idx].X, points[idx].Y);
            AppendFormat("{0} {1} l\n", points[idx].X, points[idx].Y);
            inPath = true;
            if ((type & PathPointTypeCloseSubpath) != 0)
              Append("h\n");
            break;

          case PathPointTypeBezier:
            Debug.Assert(idx + 2 < count);
            //PDF_curveto(pdf, points[idx].X, points[idx].Y, 
            //                 points[idx + 1].X, points[idx + 1].Y, 
            //                 points[idx + 2].X, points[idx + 2].Y);
            AppendFormat("{0} {1} {2} {3} {4} {5} c\n", points[idx].X, points[idx].Y, 
              points[++idx].X, points[idx].Y, points[++idx].X, points[idx].Y);
            inPath = true;
            if ((types[idx] & PathPointTypeCloseSubpath) != 0)
              Append("h\n");
            break;
        }
      }   
      //Append(pdf.ToString());
    }

    void Append(string value)
    {
      this.content.Append(value);
    }

    void AppendFormat(string format, params object[] args)
    {
      this.content.AppendFormat(CultureInfo.InvariantCulture, format, args);
    }

    void AppendStrokeFill(XPen pen, XBrush brush, XFillMode fillMode, bool closePath)
    {
      if (closePath)
        this.content.Append("h ");

      if (fillMode == XFillMode.Winding)
      {
        if (pen != null && brush != null)
          this.content.Append("B\n");
        else if (pen != null)
          this.content.Append("S\n");
        else
          this.content.Append("f\n");
      }
      else
      {
        if (pen != null && brush != null)
          this.content.Append("B*\n");
        else if (pen != null)
          this.content.Append("S\n");
        else
          this.content.Append("f*\n");
      }
    }
    #endregion

    #region Realizing
    void Realize(XPen pen, XBrush brush)
    {
      InitGraphics();
      BeginGraphic();
      RealizeTransform();
      RealizeClipPath();

#if DEBUG_
      if (pen != null && brush is XSolidBrush)
      {
        //Debug.Assert(pen.Color.Alpha != ((XSolidBrush)brush).Color.Alpha,
        //  "Current implementation limit: pen and brush must have same alpha value.");
        double alpha = Math.Min(pen.Color.A, ((XSolidBrush)brush).Color.A);
        XColor color = pen.Color;
        color.A = alpha;
        pen.Color = color;

        color = ((XSolidBrush)brush).Color;
        color.A = alpha;
        ((XSolidBrush)brush).Color = color;
      }
#endif
      if (pen != null)
      {
        if (this.gfxState.LineWidth != pen.Width)
        {
          AppendFormat("{0:0.###} w\n", pen.Width);
          this.gfxState.LineWidth = pen.Width;
        }
        if (this.gfxState.LineCap != (int)pen.LineCap)
        {
          AppendFormat("{0} J\n", (int)pen.LineCap);
          this.gfxState.LineCap = (int)pen.LineCap;
        }
        if (this.gfxState.LineJoin != (int)pen.LineJoin)
        {
          AppendFormat("{0} j\n", (int)pen.LineJoin);
          this.gfxState.LineJoin = (int)pen.LineJoin;
        }
        if (this.gfxState.DashStyle != pen.DashStyle)
        {
          double dash = 3 * pen.Width;
          double dot = pen.Width;
          switch (pen.DashStyle)
          {
            case XDashStyle.Solid:
              Append("[]0 d\n");
              break;

            case XDashStyle.Dash:
              AppendFormat("[{0:0.##} {1:0.##}]0 d\n", dash, dot);
              break;

            case XDashStyle.Dot:
              AppendFormat("[{0:0.##}]0 d\n", dot);
              break;

            case XDashStyle.DashDot:
              AppendFormat("[{0:0.##} {1:0.##} {1:0.##} {1:0.##}]0 d\n", dash, dot);
              break;

            case XDashStyle.DashDotDot:
              AppendFormat("[{0:0.##} {1:0.##} {1:0.##} {1:0.##} {1:0.##} {1:0.##}]0 d\n", dash, dot);
              break;
          }
          this.gfxState.DashStyle = pen.DashStyle;
        }
        if (this.gfxState.LineCap == (int)XLineJoin.Miter)
        {
          if (this.gfxState.MiterLimit != (int)pen.MiterLimit && (int)pen.MiterLimit != 0)
          {
            AppendFormat("{0} M\n", (int)pen.MiterLimit);
            this.gfxState.MiterLimit = (int)pen.MiterLimit;
          }
        }
        XColor color = pen.Color;
        if (this.gfxState.StrokeColor != color)
        {
          AppendFormat("{0} {1} {2} RG\n", 
            PdfEncoders.ToString(color.R / 255f), PdfEncoders.ToString(color.G / 255f), PdfEncoders.ToString(color.B / 255f));
          this.gfxState.StrokeColor = pen.Color;
#if ALPHA
          if (this.page.Document.Version >= 14)
          {
            PdfExtGState extGState = this.page.Document.ExtGStateTable.GetExtGStateStroke(color.A);
            string gs = this.page.Resources.AddExtGState(extGState);
            AppendFormat("{0} gs\n", gs);
          }
#endif
        }
      }

      if (brush != null)
      {
        if (brush is XSolidBrush)
        {
          XColor color = ((XSolidBrush)brush).Color;
          if (this.gfxState.FillColor != color)
          {
            AppendFormat("{0} {1} {2} rg\n", 
              PdfEncoders.ToString(color.R / 255f), PdfEncoders.ToString(color.G / 255f), PdfEncoders.ToString(color.B / 255f));
            this.gfxState.FillColor = color;
#if ALPHA
            if (this.page.Document.Version >= 14)
            {
              PdfExtGState extGState = this.page.Document.ExtGStateTable.GetExtGStateNonStroke(color.A);
              string gs = this.page.Resources.AddExtGState(extGState);
              AppendFormat("{0} gs\n", gs);
            }
#endif
          }
        }
      }
    }

    void Realize(XPen pen)
    {
      Realize(pen, null);
    }

    void Realize(XBrush brush)
    {
      Realize(null, brush);
    }

    void Realize(XFont font, XBrush brush, int renderMode)
    {
      InitGraphics();
      RealizeTransform();
      RealizeClipPath();
      if (this.streamMode != StreamMode.Text)
      {
        //this.inPath = false;
        this.streamMode = StreamMode.Text;
        this.content.Append("BT\n");
      }
      // So far rendering mode 0 only.
      if (true)
      {
        //Realize(brush);
        //if (brush is XSolidBrush)
        //{
        //  XColor color = ((XSolidBrush)brush).Color;
        //  AppendFormat("{0} {1} {2} rg\n", 
        //    PdfEncoders.ToString(color.R / 255f), PdfEncoders.ToString(color.G / 255f), PdfEncoders.ToString(color.B / 255f));
        //}
        XColor color = ((XSolidBrush)brush).Color;
        if (true || this.gfxState.FillColor != color)
        {
          AppendFormat("{0} {1} {2} rg\n", 
            PdfEncoders.ToString(color.R / 255f), PdfEncoders.ToString(color.G / 255f), PdfEncoders.ToString(color.B / 255f));
          this.gfxState.FillColor = color;
#if ALPHA
          if (this.page.Document.Version >= 14)
          {
            PdfExtGState extGState = this.page.Document.ExtGStateTable.GetExtGState(color.A);
            string gs = this.page.Resources.AddExtGState(extGState);
            AppendFormat("{0} gs\n", gs);
          }
#endif
        }
      }

      string resfont = this.page.GetFontName(font);

      // TODO: pagedirection
      AppendFormat("{0} {1} Tf\n", resfont, -font.Size);
    }

    string Realize(XImage image)
    {
      InitGraphics();
      BeginGraphic();
      RealizeTransform();
      RealizeClipPath();

      string imageName;
      if (image is XPdfForm)
        imageName = this.page.GetFormName(image as XPdfForm);
      else
        imageName = this.page.GetImageName(image);
      return imageName;
    }

    void RealizeState()
    {
      InitGraphics();
      BeginGraphic();
      switch (this.gfxStateLevel)
      {
        case GraphicsStateLevel.Initial:
          this.content.Append("q q ");
          break;

        case GraphicsStateLevel.PageSpace:
          this.content.Append("q ");
          break;

        case GraphicsStateLevel.WorldSpace:
          this.content.Append("Q q ");
          break;

        case GraphicsStateLevel.ClipPath:
          break;
      }
      this.gfxStateLevel = GraphicsStateLevel.WorldSpace;
      double[] matrix = this.gfxState.Ctm.Elements;
      AppendFormat("{0:0.###} {1:0.###} {2:0.###} {3:0.###} {4:0.###} {5:0.###} cm\n",
        matrix[0], matrix[1], matrix[2], matrix[3], matrix[4], matrix[5]);
    }

    /// <summary>
    /// Realizes the current transformation matrix.
    /// </summary>
    void RealizeTransform()
    {
      InitGraphics();
      BeginGraphic();
      if ((this.dirty & DirtyFlags.Ctm) != 0)
      {
        this.dirty &= ~(DirtyFlags.Ctm | DirtyFlags.StrokeFill);
        this.gfxState.InvalidateStrokeFill();
        
        switch (this.gfxStateLevel)
        {
          case GraphicsStateLevel.Initial:
            this.content.Append("q q ");
            break;

          case GraphicsStateLevel.PageSpace:
            this.content.Append("q ");
            break;

          case GraphicsStateLevel.WorldSpace:
            this.content.Append("Q q ");
            break;

          case GraphicsStateLevel.ClipPath:
            this.content.Append("Q Q q ");
            break;
        }
        this.gfxStateLevel = GraphicsStateLevel.WorldSpace;
        if (!this.gfxState.Ctm.IsIdentity)
        {
          double[] matrix = this.gfxState.Ctm.Elements;
          AppendFormat("{0:0.###} {1:0.###} {2:0.###} {3:0.###} {4:0.###} {5:0.###} cm\n",
            matrix[0], matrix[1], matrix[2], matrix[3], matrix[4], matrix[5]);
        }
      }
    }

    /// <summary>
    /// Realizes the clipping path, if any. 
    /// TODO: The current implementation cannot change the transformation if a clipping path is 
    /// established. Setting the transformation implicitely empties the clipping path.
    /// </summary>
    void RealizeClipPath()
    {
      InitGraphics();
      BeginGraphic();

      if ((this.dirty & DirtyFlags.ClipPath) != 0)
      {
        this.dirty &= ~(DirtyFlags.ClipPath | DirtyFlags.StrokeFill);
        this.gfxState.InvalidateStrokeFill();

        switch (this.gfxStateLevel)
        {
          case GraphicsStateLevel.Initial:
            this.content.Append("q q q ");
            break;

          case GraphicsStateLevel.PageSpace:
            this.content.Append("q q ");
            break;

          case GraphicsStateLevel.WorldSpace:
            this.content.Append("q ");
            break;

          case GraphicsStateLevel.ClipPath:
            this.content.Append("Q q ");
            break;
        }

        this.gfxStateLevel = GraphicsStateLevel.ClipPath;
        if (this.gfxState.ClipPath != null)
        {
          AppendPath(this.gfxState.ClipPath.gdipPath);
          if (this.gfxState.ClipPath.FillMode == XFillMode.Winding)
            Append("W n\n");
          else
            Append("W* n\n");
        }
      }
    }
    #endregion

    [Conditional("DEBUG")]
    void DumpPathData(PathData pathData)
    {
      int count = pathData.Points.Length;
      for (int idx = 0; idx < count; idx++)
      {
        string info = String.Format("{0:X}   {1:####0.000} {2:####0.000}", pathData.Types[idx], pathData.Points[idx].X, pathData.Points[idx].Y);
        Debug.WriteLine(info, "PathData");
      }
    }

    PdfPage page;
    XGraphicsPdfPageOptions options;
    XGraphics gfx;
    StringBuilder content;

    enum GraphicsStateLevel
    {
      /// <summary>
      /// The q/Q nesting level is 0.
      /// </summary>
      Initial, 

      /// <summary>
      /// The q/Q nesting level is 1.
      /// </summary>
      PageSpace, 

      /// <summary>
      /// The q/Q nesting level is 2.
      /// </summary>
      WorldSpace,

      /// <summary>
      /// The q/Q nesting level is 3.
      /// </summary>
      ClipPath,
    }
    GraphicsStateLevel gfxStateLevel;

    /// <summary>
    /// 
    /// </summary>
    PdfGraphicsState gfxState;
    PdfGraphicsStateStack gfxStateStack;
    DirtyFlags dirty;
  }
}
