//
// PDFSharp - A library for processing PDF
//
// Authors:
//   Stefan Lange (mailto:Stefan.Lange@pdfsharp.com)
//
// Copyright (c) 2005 empira Software GmbH, Cologne (Germany)
//
// http://www.pdfsharp.com
//
// Permission is hereby granted, free of charge, to any person obtaining
// a copy of this software and associated documentation files (the
// "Software"), to deal in the Software without restriction, including
// without limitation the rights to use, copy, modify, merge, publish,
// distribute, sublicense, and/or sell copies of the Software, and to
// permit persons to whom the Software is furnished to do so, subject to
// the following conditions:
// 
// The above copyright notice and this permission notice shall be
// included in all copies or substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
// EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
// MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
// NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
// LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
// OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
// WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

using System;
using System.Globalization;
using System.Text;
using System.IO;
using System.Drawing;
using System.Drawing.Drawing2D;
using PdfSharp.Internal;
using PdfSharp.Pdf;

namespace PdfSharp.Drawing.Pdf
{
  /// <summary>
  /// In PDF the current transformation matrix (CTM) can only be modified, but not set. The XGraphics
  /// object allows to set the transformation matrix, which leads to a problem. In PDF the only way
  /// to reset the CTM to its original value is to save and restore the PDF graphics state. Dont try
  /// to keep track of every modification and then reset the CTM by multiplying with the inverse matrix
  /// of the product of all modifications. PDFlib uses this 'trick', but it does not work. Because of
  /// rounding errors everything on the PDF page looks sloping after some resets. Saving and restoring
  /// the graphics state is the only possible way to reset the CTM, but because the PDF restore operator
  /// 'Q' resets not only the CTM but all other graphics state values, we have to implement our own 
  /// graphics state management. This is apparently the only safe way to give the XGrahics users the 
  /// illusion that they can arbitrarily set the transformation matrix.
  /// 
  /// The current implementation is just a draft. Save/Restore works only once and clipping is not
  /// correctly restored in any case.
  /// </summary>
  internal class PdfGraphicsState : ICloneable
  {
    public PdfGraphicsState()
    {
      InvalidateStrokeFill();
    }

    public PdfGraphicsState Clone()
    {
      PdfGraphicsState state = (PdfGraphicsState)MemberwiseClone();
      state.Ctm = this.Ctm;
      if (state.ClipPath != null)
        state.ClipPath = state.ClipPath.Clone();
      return state;
    }

    object ICloneable.Clone()
    {
      return Clone();
    }

    public void InvalidateStroke()
    {
      LineWidth = -1;
      LineCap = -1;
      LineJoin = -1;
      MiterLimit = -1;
      DashStyle = (XDashStyle)(-1);
      StrokeColor = XColor.Empty;
    }

    public void InvalidateFill()
    {
      FillColor = XColor.Empty;
    }

    public void InvalidateStrokeFill()
    {
      InvalidateStroke();
      InvalidateFill();
    }

    public XMatrix Ctm = XMatrix.Identity;
    public XGraphicsPath ClipPath;
    public double LineWidth;
    public int LineCap;
    public int LineJoin;
    public double MiterLimit;
    public XDashStyle DashStyle;
    //public int[] LineDash;
    //TODO  pen, brush, clippath etc.
    public XColor StrokeColor;
    public XColor FillColor;
  }
}
