/* Copyright(C) 2004-2006 Brazil

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifdef STANDARD
#include <stdio.h>
#include <string.h>
#ifdef __WIN__
typedef unsigned __int64 ulonglong;     /* Microsofts 64 bit types */
typedef __int64 longlong;
#else
typedef unsigned long long ulonglong;
typedef long long longlong;
#endif /*__WIN__*/
#else
/* avoid warning */
#ifdef PACKAGE
#undef PACKAGE
#endif
#ifdef PACKAGE_BUGREPORT
#undef PACKAGE_BUGREPORT
#endif
#ifdef PACKAGE_NAME
#undef PACKAGE_NAME
#endif
#ifdef PACKAGE_STRING
#undef PACKAGE_STRING
#endif
#ifdef PACKAGE_TARNAME
#undef PACKAGE_TARNAME
#endif
#ifdef PACKAGE_VERSION
#undef PACKAGE_VERSION
#endif
#ifdef VERSION
#undef VERSION
#endif
#include <my_global.h>
#include <my_sys.h>
#endif
#include <mysql.h>
#include <m_ctype.h>
#include <m_string.h>           // To get strmov()

#include <stdlib.h>
#include <string.h>
#include <limits.h>
#include <senna.h>

#ifdef HAVE_DLOPEN

#define MINIMUM_RESULT_LENGTH 64

my_bool snippet_init(UDF_INIT * initid, UDF_ARGS * args, char *message);
void snippet_deinit(UDF_INIT * initid);
char *snippet(UDF_INIT * initid, UDF_ARGS * args, char *result,
              unsigned long *length, char *is_null, char *error);

typedef struct
{
  char *result;
  char *buffer;
  char *source;
  size_t result_len;
  size_t buffer_len;
  size_t source_len;
  char *keytags;
  sen_snip *snip;
} snip_struct;

my_bool
snippet_init(UDF_INIT * initid, UDF_ARGS * args, char *message)
{

  snip_struct *ptr;
  sen_encoding enc;
  sen_snip *snip;
  sen_snip_mapping *map;
  int i, keytagslength;
  long long slength, snumber, htmlflag;
  char *keytags, *p, *q, *r;

  initid->ptr = NULL;

  if (args->arg_count < 10
      || (args->arg_count % 3) != 1
      || args->arg_type[1] != INT_RESULT
      || args->args[1] == NULL
      || args->arg_type[2] != INT_RESULT
      || args->args[2] == NULL
      || args->arg_type[3] != STRING_RESULT
      || args->args[3] == NULL
      || args->lengths[3] > INT_MAX
      || args->arg_type[4] != INT_RESULT
      || args->args[4] == NULL
      || args->arg_type[5] != STRING_RESULT
      || args->args[5] == NULL
      || args->lengths[5] > INT_MAX
      || args->arg_type[6] != STRING_RESULT
      || args->args[6] == NULL || args->lengths[6] > INT_MAX) {
    strcpy(message, "invalid parameters");
    return 1;
  }

  slength = *((long long *) args->args[1]);
  snumber = *((long long *) args->args[2]);
  htmlflag = *((long long *) args->args[4]);

  if (slength < 0) {
    strcpy(message, "too short length");
    return 1;
  }

  if (snumber < 0) {
    strcpy(message, "too small number");
    return 1;
  }

  if (strcmp(args->args[3], "ujis") == 0) {
    enc = sen_enc_euc_jp;
  } else if (strcmp(args->args[3], "utf8") == 0) {
    enc = sen_enc_utf8;
  } else if (strcmp(args->args[3], "sjis") == 0) {
    enc = sen_enc_sjis;
  } else if (strcmp(args->args[3], "default") == 0) {
    enc = sen_enc_default;
  } else {
    strcpy(message, "encode is invalid");
    return 1;
  }

  map = NULL;
  if (htmlflag) {
    map = (sen_snip_mapping *) - 1;
  }

  keytagslength = 0;
  for (i = 7; i < args->arg_count; i++) {
    if (args->arg_type[i] != STRING_RESULT) {
      sprintf(message, "invalid parameters(%dth)", i);
      return 1;
    }
    if (args->lengths[i] >= INT_MAX) {
      sprintf(message, "too long parameters(%dth)", i);
      return 1;
    }
    keytagslength += args->lengths[i] + 1;
  }

  if (!(keytags = (char *) malloc(sizeof(char) * keytagslength))) {
    strcpy(message, "memory allocation error !(tags)");
    return 1;
  }

  p = keytags;
  snip = sen_snip_open(enc, SEN_SNIP_NORMALIZE, slength, snumber, "", 0, "", 0, map);
  for (i = 7; i < args->arg_count; i += 3) {

    memcpy(p, args->args[i], args->lengths[i]); /* keyword copy */
    p[args->lengths[i]] = '\0';
    q = p + args->lengths[i] + 1;
    memcpy(q, args->args[i + 1], args->lengths[i + 1]); /* opentag copy */
    q[args->lengths[i + 1]] = '\0';
    r = q + args->lengths[i + 1] + 1;
    memcpy(r, args->args[i + 2], args->lengths[i + 2]); /* closetag copy */
    r[args->lengths[i + 2]] = '\0';

    if (sen_snip_add_cond(snip, p, args->lengths[i],
                          q, args->lengths[i + 1],
                          r, args->lengths[i + 2]) != sen_success) {
      strcpy(message, "cannot add conditions");
      return 1;
    }
    p = r + args->lengths[i + 2] + 1;
  }

  if (!(ptr = (snip_struct *) malloc(sizeof(snip_struct)))) {
    strcpy(message, "memory allocation error !(snip_struct)");
    return 1;
  }
  memset(ptr, 0, sizeof(snip_struct));

  initid->ptr = (char *) ptr;
  initid->max_length = 65535;   /* represents blob */
  initid->maybe_null = FALSE;

  ptr->snip = snip;
  ptr->keytags = keytags;
  ptr->result_len = (size_t) ((long double) slength * snumber * 1.3);
  if (ptr->result_len < MINIMUM_RESULT_LENGTH) {
    ptr->result_len = MINIMUM_RESULT_LENGTH;
  }

  ptr->buffer_len = (size_t) ((long double) slength * 1.3);
  if (ptr->buffer_len >= INT_MAX) {
    snippet_deinit(initid);
    strcpy(message, "too long snippet length!");
    return 1;
  }

  ptr->source_len = 0;

  if (!(ptr->result = (char *) malloc(sizeof(char) * ptr->result_len))) {
    snippet_deinit(initid);
    strcpy(message, "memory allocation error !(result)");
    return 1;
  }

  if (!(ptr->buffer = (char *) malloc(sizeof(char) * ptr->buffer_len))) {
    snippet_deinit(initid);
    strcpy(message, "memory allocation error !(buffer)");
    return 1;
  }

  /* zero length allocation */
  if (!(ptr->source = (char *) malloc(sizeof(char) * ptr->source_len))) {
    snippet_deinit(initid);
    strcpy(message, "memory allocation error !(source)");
    return 1;
  }

  return 0;
}

void
snippet_deinit(UDF_INIT * initid)
{
  snip_struct *ptr;

  ptr = (snip_struct *) initid->ptr;
  if (ptr) {
    if (ptr->snip) {
      sen_snip_close(ptr->snip);
    }
    if (ptr->result) {
      free(ptr->result);
    }
    if (ptr->buffer) {
      free(ptr->buffer);
    }
    if (ptr->source) {
      free(ptr->source);
    }
    if (ptr->keytags) {
      free(ptr->keytags);
    }
    free(ptr);
  }
}

char *
snippet(UDF_INIT * initid, UDF_ARGS * args, char *result,
        unsigned long *length, char *is_null, char *error)
{
  snip_struct *ptr;
  sen_snip *snip;
  int i;
  unsigned int n_results, estimated_length;
  size_t max_tagged_len;
  char *p, *q;

  ptr = (snip_struct *) initid->ptr;
  snip = ptr->snip;

  p = ptr->result;
  if (args->arg_type[0] != STRING_RESULT) {
    strcpy(ptr->result, "cannot make snippet");
    goto exit;
  } else if (args->lengths[0] >= INT_MAX) {
    strcpy(ptr->result, "string is too long");
    goto exit;
  } else if (args->args[0]) {

    if ((args->args[0])[args->lengths[0]] == '\0') {
      q = args->args[0];        /* null terminated */
    } else {                    /* not null terminated */
      /* source realloc */
      if (ptr->source_len <= args->lengths[0]) {
        ptr->source =
          (char *) realloc(ptr->source, sizeof(char) * (args->lengths[0] + 1));
        if (!ptr->result) {
          strcpy(ptr->result, "cannot reallocate memory for source");
          goto exit;
        }
        ptr->source_len = args->lengths[0] + 1;
      }
      q = ptr->source;
      memcpy(q, args->args[0], args->lengths[0]);
      q[args->lengths[0]] = '\0';
    }

    if (sen_snip_exec(snip, q, args->lengths[0], &n_results, &max_tagged_len)
        != sen_success) {
      strcpy(ptr->result, "cannot make snippet");
      goto exit;
    }

    /* buffer realloc */
    if (ptr->buffer_len < max_tagged_len) {
      ptr->buffer = (char *) realloc(ptr->buffer, sizeof(char) * max_tagged_len);
      if (!ptr->buffer) {
        strcpy(ptr->buffer, "cannot reallocate memory for buffer");
        goto exit;
      }
      ptr->buffer_len = max_tagged_len;
    }

    /* result realloc */
    estimated_length =
      (args->lengths[5] + args->lengths[6] + max_tagged_len - 1) * n_results + 1;
    if (ptr->result_len < estimated_length) {
      ptr->result = (char *) realloc(ptr->result, sizeof(char) * estimated_length);
      if (!ptr->result) {
        strcpy(ptr->result, "cannot reallocate memory for result");
        goto exit;
      }
      ptr->result_len = estimated_length;
    }

    p = ptr->result;
    for (i = 0; i < n_results; i++) {
      if (sen_snip_get_result(snip, i, ptr->buffer, NULL) != sen_success) {
        strcpy(ptr->result, "cannot get result");
        goto exit;
      }
      memcpy(p, args->args[5], args->lengths[5]);
      p += args->lengths[5];
      for (q = ptr->buffer; *q != '\0'; *p++ = *q++) {}
      memcpy(p, args->args[6], args->lengths[6]);
      p += args->lengths[6];
    }
  }

exit:
  *p = '\0';
  *length = strlen(ptr->result);

  return ptr->result;
}

#endif /* HAVE_DLOPEN */
